<?php

namespace UncannyAutomator\AddOn\UserLists\User;

use UncannyAutomator\AddOn\UserLists\Lists\User_List;

/**
 *
 */
class User_Lists_Table {


	/**
	 * @var
	 */
	private $user_id;

	/**
	 * @var
	 */
	private $lists;

	/**
	 * @var
	 */
	private $show_only_subscribed;

	/**
	 * @var
	 */
	public $rows;

	/**
	 * @var
	 */
	public $show_unsubscribe_all;

	/**
	 * @var
	 */
	public $is_unsubscribed = false;

	/**
	 *
	 */
	public function __construct( $user_id, $show_only_subscribed = true, $show_unsubscribe_all = false ) {
		$this->user_id              = $user_id;
		$this->show_only_subscribed = $show_only_subscribed;
		$this->show_unsubscribe_all = $show_unsubscribe_all;

		if ( $this->show_only_subscribed ) {
			$this->lists = User_List::get_user_lists( $user_id );
		} else {
			$this->lists = User_List::get_all();
		}

		$unsublist = User_List::get_by_slug( User_List::UNSUBSCRIBED_LIST_SLUG );

		if ( null !== $unsublist && $unsublist->has_user( $this->user_id ) ) {
			$this->is_unsubscribed = true;
		}

		$this->generate_table_rows();
	}

	/**
	 * @return void
	 */
	public function load_assets() {
		// Enqueue the JS
		$script_url = plugins_url( 'dist/user-manage-lists.js', UAUL_PLUGIN_FILE );

		wp_enqueue_script(
			'uaul-user-manage-lists',
			$script_url,
			array(),
			true,
			true
		);

		// Enqueue the CSS
		$style_url = plugins_url( 'dist/style-user-manage-lists.css', UAUL_PLUGIN_FILE );

		wp_enqueue_style(
			'uaul-user-manage-lists',
			$style_url,
			array(),
			true
		);
	}

	/**
	 * @return void
	 */
	public function html() {
		// Load the assets
		$this->load_assets();

		?>

        <div class="uaul-user-lists">
            <table class="uaul-user-lists-table form-table">
				<?php echo wp_kses( implode( '', $this->rows ), $this->allowed_html() ); ?>
            </table>
        </div>

		<?php
	}

	/**
	 * generate_table_rows
	 *
	 * @return void
	 */
	public function generate_table_rows() {

		foreach ( $this->lists as $list ) {
			if ( User_List::UNSUBSCRIBED_LIST_SLUG === $list->slug ) {
				continue;
			}
			$this->rows[] = $this->row( $list );
		}

		if ( $this->show_unsubscribe_all ) {
			$list         = User_List::get_unsubscribed_list();
			$this->rows[] = $this->row( $list );
		}

	}

	/**
	 * @param $list
	 *
	 * @return void
	 */
	public function row( $list ) {
		ob_start();

		// Get the list name
		$list_name = $list->get_input_name();

		?>

        <tr data-list="<?php echo esc_attr( $list_name ); ?>" class="uaul-user-lists-item">
			<?php echo wp_kses( $this->label( $list ), $this->allowed_html() ); ?>
			<?php echo wp_kses( $this->input( $list ), $this->allowed_html() ); ?>
        </tr>

		<?php
		return ob_get_clean();
	}


	/**
	 * @param $list
	 *
	 * @return void
	 */
	public function label( $list ) {
		$list_name = $list->get_name();

		if ( User_List::UNSUBSCRIBED_LIST_SLUG === $list->slug ) {
			$list_name = __( 'Unsubscribe from all', 'uaul' );
		}

		ob_start();
		?>
        <th>
            <label for="<?php echo esc_attr( $list->get_input_name() ); ?>"><?php echo esc_attr( $list_name ); ?></label>
        </th>
		<?php
		return ob_get_clean();
	}

	/**
	 * @param $list
	 *
	 * @return void
	 */
	public function input( $list ) {

		$checked  = '';
		$disabled = '';

		if ( $list->has_user( $this->user_id ) ) {
			$checked = ' checked="checked"';
		}

		if ( $this->is_unsubscribed && $list->slug !== User_List::UNSUBSCRIBED_LIST_SLUG ) {
			$disabled = ' disabled="disabled"';
		}

		ob_start();
		?>
        <td>
            <input type="checkbox" name="<?php echo esc_attr( $list->get_input_name() ); ?>"
                   id="<?php echo esc_attr( $list->get_input_name() ); ?>"
				<?php echo esc_attr( $checked ); ?>
				<?php echo esc_attr( $disabled ); ?>
                   class="regular-text"/><br/>
        </td>
		<?php
		return ob_get_clean();
	}

	/**
	 * @return array
	 */
	public function allowed_html() {
		return array(
			'tr'    => array(
				'data-list' => array(),
				'class'     => array(),
			),
			'th'    => array(),
			'label' => array(
				'id'   => array(),
				'type' => array(),
				'for'  => array(),
			),
			'td'    => array(),
			'input' => array(
				'disabled' => array(),
				'checked'  => array(),
				'name'     => array(),
				'id'       => array(),
				'type'     => array(),
			),
		);
	}
}
