<?php

namespace UncannyAutomator\AddOn\UserLists\Lists;

/**
 *
 */
class User_List {


	// Factory methods are defined as static, see the object methods below

	const POST_TYPE = 'ua-user-list';

	const UNSUBSCRIBED_LIST_SLUG = 'uaul_unsubscribed';

	private static $all_lists = array();

	/**
	 * get_restriction_message
	 *
	 * @return string
	 */
	public static function get_restriction_message() {
		$default_restriction_message = esc_html__( 'Access denied. Please contact support if you believe you should have access to this page.', 'uaul' );
		$current_message             = automator_get_option( 'uaul_restriction_message', $default_restriction_message );

		return apply_filters( 'automator_ul_user_list_restriction_message', $current_message );
	}

	/**
	 * get_unsubscribed_list
	 *
	 * @return User_List
	 */
	public static function get_unsubscribed_list() {

		$unsubscribed_list = self::get_by_slug( self::UNSUBSCRIBED_LIST_SLUG );

		if ( null === $unsubscribed_list ) {
			$unsubscribed_list = self::create( __( 'Unsubscribed', 'uaul' ), self::UNSUBSCRIBED_LIST_SLUG );
		}

		return $unsubscribed_list;
	}

	/**
	 * @param $id
	 *
	 * @return false|mixed|null
	 * @throws \Exception
	 */
	public static function get( $id = null ) {

		$post = \get_post( $id );

		if ( null === $post ) {
			return null;
		}

		return new self( $post );
	}

	/**
	 * @param $id
	 *
	 * @return false|mixed|null
	 * @throws \Exception
	 */
	public static function get_by_slug( $slug ) {
		$post = \get_page_by_path( $slug, OBJECT, self::POST_TYPE );

		if ( null === $post ) {
			return null;
		}

		return new self( $post );
	}

	/**
	 * @return false|mixed|null
	 */
	public static function get_all( $flush_cache = false ) {

		if ( $flush_cache ) {
			self::$all_lists = array();
		}

		// Only query everything once per session
		if ( ! empty( self::$all_lists ) ) {
			return self::$all_lists;
		}

		$lists = \get_posts(
			array(
				'post_type'      => self::POST_TYPE,
				'post_status'    => 'publish',
				'posts_per_page' => 9999,  // phpcs:ignore WordPress.WP.PostsPerPage.posts_per_page_posts_per_page
			)
		);

		foreach ( $lists as $list ) {
			$list                         = new self( $list );
			self::$all_lists[ $list->id ] = $list;
		}

		return self::$all_lists;
	}

	/**
	 * @param $list
	 *
	 * @return mixed $list
	 * @throws \Exception
	 */
	public static function create( $name, $slug = null, $settings = array() ) {

		$settings = apply_filters( 'automator_ul_create_user_list_settings', $settings, $name, $slug );

		if ( empty( $slug ) ) {
			$slug = sanitize_title( $name );
		}

		$post = array(
			'post_title'  => \wp_strip_all_tags( $name ),
			'post_status' => 'publish',
			'post_name'   => $slug,
			'post_type'   => self::POST_TYPE,
		);

		$post = apply_filters( 'automator_ul_before_user_list_created', $post, $name, $slug, $settings );

		$list_id = \wp_insert_post( $post );

		\update_post_meta( $list_id, 'settings', $settings );
		\update_post_meta( $list_id, 'users', array() );

		$list_object = self::get( $list_id );

		return apply_filters( 'automator_ul_user_list_created', $list_object, $name, $slug, $settings );
	}

	/**
	 * user_in_list
	 *
	 * @param mixed $user_id
	 * @param mixed $list_id
	 *
	 * @return bool
	 */
	public static function user_in_list( $user_id, $list_id ) {

		$list_object = self::get( $list_id );

		return $list_object->has_user( $user_id );
	}

	/**
	 * get_user_lists
	 *
	 * @param mixed $user_id
	 *
	 * @return array
	 */
	public static function get_user_lists( $user_id ) {

		$user_id = absint( $user_id );

		$lists = self::get_all();

		$user_lists = array();

		foreach ( $lists as $list ) {

			if ( array_key_exists( $user_id, $list->users ) ) {
				$user_lists[ $list->id ] = $list;
			}
		}

		return $user_lists;
	}

	/**
	 * Below are the object attributes and methods
	 */

	public $id;
	public $slug;
	public $name;
	public $settings = array();
	public $users = array();

	/**
	 * __construct
	 *
	 * @param mixed $post
	 *
	 * @return void
	 */
	public function __construct( $post ) {
		$this->id       = $post->ID;
		$this->name     = $post->post_title;
		$this->slug     = $post->post_name;
		$this->settings = \get_post_meta( $post->ID, 'settings', true );
		$this->users    = \get_post_meta( $post->ID, 'users', true );
	}

	/**
	 * save
	 *
	 * @return void
	 */
	public function save() {

		$this->settings = apply_filters( 'automator_ul_update_user_list_settings', $this->settings, $this );
		$this->users    = apply_filters( 'automator_ul_update_user_list_users', $this->users, $this );

		$post = array(
			'ID'         => $this->id,
			'post_title' => \wp_strip_all_tags( $this->name ),
			'post_name'  => $this->slug,
		);

		$post = apply_filters( 'automator_ul_update_user_list_post', $post, $this );

		\wp_update_post( $post );

		\update_post_meta( $this->id, 'settings', $this->settings );
		\update_post_meta( $this->id, 'users', $this->users );

		do_action( 'automator_ul_user_list_updated', $this );

		return $this;
	}

	/**
	 * @param $id
	 *
	 * @return mixed
	 * @throws \Exception
	 */
	public static function delete( $id ) {
		\delete_post_meta( $id, 'settings' );
		\delete_post_meta( $id, 'users' );
		\wp_delete_post( $id );
		do_action( 'automator_ul_user_list_deleted', $id );
	}

	/**
	 * get_users
	 *
	 * @return array
	 */
	public function get_users() {
		return $this->users;
	}

	/**
	 * get_list_users
	 *
	 * @return array
	 */
	public static function get_list_users( $list_id ) {

		$lists = self::get_all();

		return $lists[ $list_id ]->get_users();
	}


	/**
	 * users_count
	 *
	 * @return array
	 */
	public function users_count() {
		return count( $this->users );
	}

	/**
	 * add_user
	 *
	 * @param mixed $user_id
	 */
	public function add_user( $user_id ) {

		if ( ! is_numeric( $user_id ) ) {
			return;
		}

		$user_id = absint( $user_id );

		if ( self::UNSUBSCRIBED_LIST_SLUG === $this->slug ) {
			// This is an attempt to add a user to the unusubscribed list
			self::unsubscribe_from_all( $user_id );

			return true;
		}

		try {
			$this->add_user_record( $user_id );
		} catch ( \Exception $e ) {
			return false;
		}

		$this->maybe_remove_from_unsubscribed( $user_id );

		do_action( 'automator_ul_user_added_to_list', $user_id, $this->id, $this );

		return true;
	}

	/**
	 * add_user_record
	 *
	 * @param int $user_id
	 *
	 * @return void
	 */
	public function add_user_record( $user_id ) {

		if ( isset( $this->users[ $user_id ] ) ) {
			throw new \Exception( __( 'User was already on the list', 'uaul' ) );
		}

		$user_record = array(
			'member_since' => time(),
			'added_by'     => get_current_user_id(),
		);

		$this->users[ $user_id ] = apply_filters( 'automator_ul_user_list_new_user_record', $user_record );

		$this->save();
	}

	/**
	 * has_user
	 *
	 * @param mixed $user_id
	 *
	 * @return bool
	 */
	public function has_user( $user_id ) {
		return array_key_exists( absint( $user_id ), $this->users );
	}

	/**
	 * remove_user
	 */
	public function remove_user( $user_id ) {

		$user_id = absint( $user_id );

		if ( ! isset( $this->users[ $user_id ] ) ) {
			throw new \Exception( __( 'User was not on the list', 'uaul' ) );
		}

		unset( $this->users[ $user_id ] );

		$this->save();

		do_action( 'automator_ul_user_removed_from_list', $user_id, $this->id, $this );

		return true;
	}

	/**
	 * get_id
	 *
	 * @return mixed
	 */
	public function get_id() {
		return $this->id;
	}

	/**
	 * get_name
	 *
	 * @return string
	 */
	public function get_name() {
		return $this->name;
	}

	/**
	 * get_slug
	 *
	 * @return string
	 */
	public function get_slug() {
		return $this->slug;
	}

	/**
	 * get_input_name
	 *
	 * @return string
	 */
	public function get_input_name() {
		return $this->get_slug();
	}

	/**
	 * unsubscribe_from_all
	 *
	 * @param int $user_id
	 *
	 * @return void
	 */
	public static function unsubscribe_from_all( $user_id ) {

		$all_lists = self::get_all();

		foreach ( $all_lists as $list ) {

			if ( self::UNSUBSCRIBED_LIST_SLUG === $list->slug ) {
				continue;
			}

			try {
				$list->remove_user( $user_id );
			} catch ( \Exception $e ) {
				// User was not on the list. Do nothing.
				continue;
			}

		}

		$unsubscribed_list = self::get_unsubscribed_list();
		// We can't use add_user method because it will cause a loop.
		try {
			$unsubscribed_list->add_user_record( $user_id );
		} catch ( \Exception $e ) {
			return false;
		}
	}

	/**
	 * maybe_remove_from_unsubscribed
	 *
	 * @param int $user_id
	 *
	 * @return void
	 */
	public function maybe_remove_from_unsubscribed( $user_id ) {

		$unsubscribed_list = self::get_unsubscribed_list();

		try {
			$unsubscribed_list->remove_user( $user_id );
		} catch ( \Exception $e ) {
			return false;
		}

		return true;
	}

	public static function maybe_remove_deleted_user_from_lists( $user_id, $reassign, $user ) {

		$lists = self::get_user_lists( $user_id );

		foreach ( $lists as $list ) {
			try {
				$list->remove_user( $user_id );
			} catch ( \Exception $e ) {
				// Do nothing.
				continue;
			}
		}
	}
}
