<?php

namespace UncannyAutomator\AddOn\UserLists\Lists;

use UncannyAutomator\AddOn\UserLists\Integration\Settings;

/**
 *
 */
class Rest_Api {


	/**
	 *
	 */
	const ENDPOINT = 'uaul';
	/**
	 *
	 */
	const NONCE = 'wp_rest';

	/**
	 * @param $plugin
	 */
	public function __construct( $plugin ) {
		add_action( 'rest_api_init', array( $this, 'add_custom_users_api' ) );
		add_filter( 'automator_ul_settings_pass_to_js', array( $this, 'js_vars' ) );
	}

	/**
	 * @param $vars
	 *
	 * @return mixed
	 */
	public function js_vars( $vars ) {

		$settings = array(
			'url'   => get_rest_url( null, self::ENDPOINT ),
			'nonce' => wp_create_nonce( self::NONCE ),
		);

		$vars['restApi'] = $settings;

		$vars['lists'] = array_values( User_List::get_all() );

		return $vars;
	}

	/**
	 * @return void
	 */
	public function add_custom_users_api() {
		register_rest_route(
			self::ENDPOINT,
			'/user-lists',
			array(
				'methods'             => 'POST',
				'callback'            => array( $this, 'create_list' ),
				'validate_callback'   => array( $this, 'validate_call' ),
				'permission_callback' => array( $this, 'user_has_permissions' ),
			)
		);

		register_rest_route(
			self::ENDPOINT,
			'/user-lists',
			array(
				'methods'             => 'GET',
				'callback'            => array( $this, 'get_lists' ),
				'validate_callback'   => array( $this, 'validate_call' ),
				'permission_callback' => array( $this, 'user_has_permissions' ),
			)
		);

		register_rest_route(
			self::ENDPOINT,
			'/user-lists',
			array(
				'methods'             => 'DELETE',
				'callback'            => array( $this, 'delete_list' ),
				'validate_callback'   => array( $this, 'validate_call' ),
				'permission_callback' => array( $this, 'user_has_permissions' ),
			)
		);

		register_rest_route(
			self::ENDPOINT,
			'/user-lists',
			array(
				'methods'             => 'PUT',
				'callback'            => array( $this, 'update_list' ),
				'validate_callback'   => array( $this, 'validate_call' ),
				'permission_callback' => array( $this, 'user_has_permissions' ),
			)
		);

		register_rest_route(
			self::ENDPOINT,
			'/user-lists-unsubscribe-link-settings',
			array(
				'methods'             => 'PUT',
				'callback'            => array( $this, 'update_unsubsribe_link_settings' ),
				'validate_callback'   => array( $this, 'validate_call' ),
				'permission_callback' => array( $this, 'user_has_permissions' ),
			)
		);
	}

	/**
	 * @param $request
	 *
	 * @return bool
	 */
	public function validate_call( $request ) {

		$nonce = $request->get_header( 'X-WP-Nonce' );

		if ( wp_verify_nonce( $nonce, 'wp_rest' ) ) {
			return true;
		}

		return false;
	}

	/**
	 * @return bool
	 */
	public function user_has_permissions() {
		return current_user_can( 'edit_users' );
	}

	/**
	 * @param $request
	 *
	 * @return \WP_REST_Response
	 */
	public function create_list( $request ) {

		try {

			$json_params = $request->get_json_params();

			$list = User_List::create( $json_params['name'], $json_params['slug'], $json_params['settings'] );

			return $this->respond_with_data( 'list', $list, 201 );

		} catch ( \Exception $e ) {

			return $this->respond_with_data( 'error', $e->getMessage(), $e->getCode() );
		}

	}

	/**
	 * @param $request
	 *
	 * @return \WP_REST_Response
	 * @throws \Exception
	 */
	public function get_lists( $request ) {

		$lists = User_List::get_all();

		return $this->respond_with_data( 'lists', $lists );
	}

	/**
	 * @param $context
	 * @param $data
	 * @param $status
	 *
	 * @return \WP_REST_Response
	 */
	public function respond_with_data( $context, $data, $status = 200 ) {

		$headers = array(
			'Content-Type' => 'application/json',
		);

		$success = true;

		if ( 'error' === $context ) {
			$success = false;
		}

		$response_data = array(
			'success'   => $success,
			$context    => $data,
			'all_lists' => array_values( User_List::get_all( true ) ),
		);

		return new \WP_REST_Response( $response_data, $status, $headers );
	}

	/**
	 * @param $request
	 *
	 * @return \WP_REST_Response
	 */
	public function delete_list( $request ) {

		try {

			$json_params = $request->get_json_params();

			$list_id = $json_params['id'];

			User_List::delete( $json_params['id'] );

			return $this->respond_with_data( 'id', $list_id, 200 );

		} catch ( \Exception $e ) {

			return $this->respond_with_data( 'error', $e->getMessage(), $e->getCode() );
		}
	}

	/**
	 * @param $request
	 *
	 * @return \WP_REST_Response
	 */
	public function update_list( $request ) {

		try {

			$json_params = $request->get_json_params();

			$list = User_List::get( $json_params['id'] );

			$list->name = $json_params['name'];

			$list->save();

			return $this->respond_with_data( 'list', $list, 200 );

		} catch ( \Exception $e ) {
			return $this->respond_with_data( 'error', $e->getMessage(), $e->getCode() );
		}
	}

	/**
	 * @param $request
	 *
	 * @return \WP_REST_Response
	 */
	public function update_unsubsribe_link_settings( $request ) {

		try {

			$json_params = $request->get_json_params();

			$settings_to_update = array(
				'link_text' => $json_params['linkText'],
				'link_path' => $json_params['linkPath'],
			);

			$response = Settings::update( $settings_to_update );

			return $this->respond_with_data( 'settings', $response, 200 );

		} catch ( \Exception $e ) {
			return $this->respond_with_data( 'error', $e->getMessage(), $e->getCode() );
		}
	}
}
