<?php

namespace UncannyAutomator\AddOn\UserLists\Framework;

use UncannyAutomator\AddOn\UserLists\Licensing\License;
use UncannyAutomator\AddOn\UserLists\Integration\Automator_Integration as Integration;
use UncannyAutomator\AddOn\UserLists\Lists\Rest_Api;
use UncannyAutomator\AddOn\UserLists\Lists\Post_Type;
use UncannyAutomator\AddOn\UserLists\Lists\User_List;
use UncannyAutomator\AddOn\UserLists\User\Profile as User_Profile;
use UncannyAutomator\AddOn\UserLists\User\Users_Page;
use UncannyAutomator\AddOn\UserLists\User\Shortcode;
use UncannyAutomator\AddOn\UserLists\Integration\Settings;

/**
 *
 */
class Plugin {


	/**
	 *
	 */
	const OPTION_VERSION = 'uaul_plugin_version';
	/**
	 * @var
	 */
	private static $version;
	/**
	 * @var bool
	 */
	private static $requirements_met = false;
	/**
	 * @var
	 */
	public $licenses;
	/**
	 * @var
	 */
	private $notifications;

	/**
	 *
	 */
	public function __construct() {

		$this->register_plugin();

		add_action( 'automator_ul_plugin_installed', array( $this, 'setup' ) );
		add_action( 'plugins_loaded', array( $this, 'boot' ) );
		add_action( 'delete_user', array(
			'UncannyAutomator\AddOn\UserLists\Lists\User_List',
			'maybe_remove_deleted_user_from_lists',
		), 10, 3 );

		// Load text domain
		add_action(
			'init',
			array( $this, 'automator_ul_load_textdomain' )
		);
	}

	/**
	 * @return false|mixed|null
	 */
	public static function get_version() {

		// add_option will only take action if there is no option created (plugin is activated for the first time).
		automator_add_option( self::OPTION_VERSION, '0' );

		if ( null === self::$version ) {
			self::$version = automator_get_option( self::OPTION_VERSION, '0' );
		}

		return self::$version;
	}

	/**
	 * Runs only once on plugin's first installation
	 *
	 * @return void
	 */
	public function setup() {

		$default_settings = array(
			'link_text' => __( 'Unsubscribe', 'uaul' ),
			'link_path' => '/manage-emails/',
		);

		$post_details = array(
			'post_title'     => __( 'Manage emails', 'uaul' ),
			'post_content'   => '[automator_ul/]',
			'post_type'      => 'page',
			'post_status'    => 'publish',
			'comment_status' => 'closed',
			'post_name'      => 'manage-emails',
		);

		$post_id = wp_insert_post( $post_details );

		if ( ! is_wp_error( $post_id ) ) {

			$link     = get_permalink( $post_id );
			$home_url = get_home_url();

			$default_settings['link_path'] = str_replace( $home_url, '', $link );
		}

		//Add default Unsubscribed list
		User_List::get_unsubscribed_list();

		Settings::update( $default_settings );
	}

	/**
	 * @return void
	 */
	public function register_plugin() {
		Post_Type::register();
		new Plugin_Lifecycle_Events();
	}

	/**
	 * @return void
	 */
	public function boot() {

		do_action( 'automator_ul_before_plugin_loaded' );

		$this->load_classes();
	}

	/**
	 * @return void
	 */
	public function automator_ul_load_textdomain() {
		$locale = determine_locale();

		/**
		 * Filter to adjust the Uncanny Automator locale to use for translations.
		 *
		 * Note: the first-loaded translation file overrides any following ones if the same translation is present.
		 *
		 * Locales found in:
		 *      - WP_LANG_DIR/uncanny-automator-user-lists/uaul-LOCALE.mo
		 *      - WP_LANG_DIR/plugins/uaul-LOCALE.mo
		 */
		$locale = apply_filters( 'plugin_locale', $locale, 'uaul' );

		unload_textdomain( 'uaul' );
		load_textdomain( 'uaul', WP_LANG_DIR . '/uncanny-automator-user-lists/uaul-' . $locale . '.mo' );
		load_plugin_textdomain( 'uaul', false, plugin_basename( dirname( UAUL_PLUGIN_FILE ) ) . '/languages' );
	}

	/**
	 * @return void
	 */
	private function load_classes() {

		new License();
		add_action(
			'admin_notices',
			function () {
				// Check if AUTOMATOR_PLUGIN_VERSION is defined and if it's less than 6.0
				if ( defined( 'AUTOMATOR_PLUGIN_VERSION' ) && version_compare( AUTOMATOR_PLUGIN_VERSION, '6.0', '<' ) ) {
					// Display the admin notice with a link to the plugins page
					echo '<div class="notice notice-error">
            <p><strong>' . UAUL_ITEM_NAME . ':</strong> You must upgrade the Automator plugin to at least version 6.0 or later to activate this add-on. 
            Please <a href="' . esc_url( admin_url( 'plugins.php?s=Uncanny+Automator' ) ) . '">update the Uncanny Automator plugin</a> to continue.</p>
          </div>';
				}
			}
		);
		if ( defined( 'AUTOMATOR_PLUGIN_VERSION' ) && version_compare( AUTOMATOR_PLUGIN_VERSION, '6.0', '<' ) ) {
			return;
		}

		if ( ! $this->requirements_met() ) {
			new Admin_Notifications( $this->notifications );

			return;
		}

		new Integration( $this );
		new Rest_Api( $this );

		new User_Profile( $this );
		new Users_Page( $this );
		// new Post_Editor_Metabox();
		// new Maybe_Restrict_Post();

		new Shortcode();
		// new Blocks();
		// new Updater( $this );
		new Admin_Notifications( $this->notifications );

		do_action( 'automator_ul_plugin_loaded' );
	}

	/**
	 * @return bool
	 */
	private function requirements_met() {

		if ( ! $this->automator_active() ) {
			return false;
		}

		if ( ! $this->automator_pro_active() ) {
			return false;
		}

		$message = sprintf( '<strong>%s</strong>', __( "Uncanny Automator - User Lists Addon doesn't work with your current plugin license.", 'uaul' ) );
		$kb_link = sprintf( ' <a href="%s">%s</a>', 'https://automatorplugin.com/knowledge-base/installing-the-user-lists-addon/', __( 'Click here for more info', 'uaul' ) );

		$license_check_response = $this->check_license();

		if ( 403 === $license_check_response ) {
			$this->add_notification(
				'notice notice-error',
				$message
			);
		}

		if ( 402 === $license_check_response ) {
			$this->add_notification(
				'notice notice-error',
				$message . $kb_link
			);
		}

		return 200 === $license_check_response;
	}

	/**
	 * @return bool
	 */
	private function automator_active() {
		return defined( 'AUTOMATOR_PLUGIN_VERSION' );
	}

	/**
	 * @return bool
	 */
	private function automator_pro_active() {
		return defined( 'AUTOMATOR_PRO_PLUGIN_VERSION' );
	}

	/**
	 * @return int
	 */
	private function check_license() {
		return License::uaul_available();
	}

	/**
	 * @param $class
	 * @param $message
	 *
	 * @return void
	 */
	private function add_notification( $class, $message ) {
		$this->notifications[] = array(
			'class'   => $class,
			'message' => $message,
		);
	}
}
