<?php

namespace Uncanny_Automator_Pro\Integrations\GravityForms\Utilities;

use GFAPI;

/**
 * Class GF_Time_Handler
 * Utility class for handling Gravity Forms time field operations
 *
 * @package Uncanny_Automator_Pro
 */
class Time_Handler {

	/**
	 * Format time fields in the form input values
	 *
	 * @param array $input_values The form input values
	 * @param int   $form_id     The Gravity Form ID
	 *
	 * @return array Modified input values with formatted time fields
	 */
	public function format_time_fields( $input_values, $form_id ) {

		$form = GFAPI::get_form( $form_id );

		foreach ( $form['fields'] as $field ) {
			if ( 'time' === $field->type ) {
				$input_values = $this->format_single_time_field( $input_values, $field->id );
			}
		}

		return $input_values;
	}

	/**
	 * Format a single time field from its components
	 *
	 * @param array $input_values The form input values
	 * @param int   $field_id    The field ID to format
	 *
	 * @return array Modified input values with the formatted time field
	 */
	private function format_single_time_field( $input_values, $field_id ) {

		if ( ! $this->has_valid_time_components( $input_values, $field_id ) ) {
			return $input_values;
		}

		$formatted_time = $this->combine_time_components(
			$input_values[ $field_id . '.1' ],
			$input_values[ $field_id . '.2' ],
			$input_values[ $field_id . '.3' ]
		);

		// Set the formatted time value and remove components
		$input_values[ $field_id ] = $formatted_time;
		unset( $input_values[ $field_id . '.1' ] );
		unset( $input_values[ $field_id . '.2' ] );
		unset( $input_values[ $field_id . '.3' ] );

		return $input_values;
	}

	/**
	 * Check if all required time components exist
	 *
	 * @param array $input_values The form input values
	 * @param int   $field_id    The field ID to check
	 *
	 * @return bool Whether all time components exist
	 */
	private function has_valid_time_components( $input_values, $field_id ) {

		return isset( $input_values[ $field_id . '.1' ] )
			&& isset( $input_values[ $field_id . '.2' ] )
			&& isset( $input_values[ $field_id . '.3' ] );
	}

	/**
	 * Combine time components into formatted time string
	 *
	 * @param string $hours   The hours component
	 * @param string $minutes The minutes component
	 * @param string $ampm    The AM/PM component
	 *
	 * @return string Formatted time string (HH:MM AM/PM)
	 */
	private function combine_time_components( $hours, $minutes, $ampm ) {

		$hours   = str_pad( $hours, 2, '0', STR_PAD_LEFT );
		$minutes = str_pad( $minutes, 2, '0', STR_PAD_LEFT );

		return sprintf( '%s:%s %s', $hours, $minutes, $ampm );
	}
}
