<?php

namespace Uncanny_Automator_Pro;

use Uncanny_Automator\Recipe;
use Uncanny_Automator\Ws_Form_Lite_Helpers;

/**
 * Class WSF_ANON_SPECIFIC_FIELD_VALUE
 *
 * @package Uncanny_Automator_Pro
 */
class WSF_ANON_SPECIFIC_FIELD_VALUE {

	use Recipe\Triggers;

	/**
	 * Set up Automator trigger constructor.
	 */
	public function __construct() {
		if ( ! class_exists( 'Uncanny_Automator\Ws_Form_Lite_Helpers' ) ) {
			return;
		}
		$this->set_helper( new Ws_Form_Lite_Helpers() );
		$this->set_tokens_class( new Ws_Form_Lite_Pro_Tokens() );
		$this->setup_trigger();
	}

	/**
	 * Define and register the trigger by pushing it into the Automator object
	 */
	public function setup_trigger() {
		$this->set_integration( 'WSFORMLITE' );
		$this->set_trigger_code( 'WSFORM_ANON_FROM_FIELDVALUE' );
		$this->set_trigger_meta( 'WSFORM_FORMS' );
		$this->set_is_login_required( false );
		$this->set_is_pro( true );
		$this->set_trigger_type( 'anonymous' );
		$this->set_support_link( Automator()->get_author_support_link( $this->trigger_code, 'integration/ws-form/' ) );
		$this->set_sentence(
			sprintf(
				// translators: %1$s The form ID. %2$s The specific value. %3$s The specific field.
				esc_html_x( '{{A form:%1$s}} is submitted with {{a specific value:%2$s}} in {{a specific field:%3$s}}', 'Ws Form Lite', 'uncanny-automator-pro' ),
				$this->get_trigger_meta(),
				'SPECIFIC_VALUE' . ':' . $this->get_trigger_meta(),
				'SPECIFIC_FIELD' . ':' . $this->get_trigger_meta()
			)
		);
		// Non-active state sentence to show
		$this->set_readable_sentence( esc_attr_x( '{{A form}} is submitted with {{a specific value}} in {{a specific field}}', 'Ws Form Lite', 'uncanny-automator-pro' ) );
		// Which do_action() fires this trigger.
		$this->set_action_hook( 'wsf_submit_post_complete' );
		$this->set_options_callback( array( $this, 'load_options' ) );
		$this->register_trigger();
	}

	/**
	 * @return array
	 */
	public function load_options() {
		return Automator()->utilities->keep_order_of_options(
			array(
				'options_group' => array(
					$this->get_trigger_meta() => array(
						$this->get_helper()->get_ws_all_forms(
							null,
							$this->get_trigger_meta(),
							array(
								'is_ajax'      => true,
								'target_field' => 'SPECIFIC_FIELD',
								'endpoint'     => 'select_form_fields_WSFORMS',
							)
						),
						Automator()->helpers->recipe->field->select(
							array(
								'option_code' => 'SPECIFIC_FIELD',
								'label'       => esc_attr_x( 'Field', 'Ws Form Lite', 'uncanny-automator-pro' ),
							)
						),
						Automator()->helpers->recipe->field->text(
							array(
								'option_code' => 'SPECIFIC_VALUE',
								'label'       => esc_attr_x( 'Value', 'Ws Form Lite', 'uncanny-automator-pro' ),
							)
						),
					),
				),
			)
		);
	}

	/**
	 * Validate the trigger.
	 *
	 * @param $args
	 *
	 * @return bool
	 */
	protected function validate_trigger( ...$args ) {
		list( $hook_args ) = array_shift( $args );

		if ( empty( $hook_args->form_id ) ) {
			return false;
		}

		return true;
	}

	/**
	 * Prepare to run the trigger.
	 *
	 * @param $data
	 *
	 * @return void
	 */
	public function prepare_to_run( $data ) {
		$this->set_conditional_trigger( true );
	}

	/**
	 * Check contact status against the trigger meta
	 *
	 * @param $args
	 */
	public function validate_conditions( ...$args ) {

		$hook_args       = $args[0][0] ?? new \stdClass(); // WS Form passes object accessible in [0][0].
		$recipes         = $this->trigger_recipes();
		$recipes_matched = array();

		// Bail early if no recipes - Triggering from submit form trigger.
		if ( empty( $recipes ) ) {
			return array();
		}

		// All possible recipes that matches.
		foreach ( $recipes as $recipe ) {

			// The triggers of the recipe.
			$triggers = $recipe['triggers'] ?? array();
			foreach ( $triggers as $trigger ) {

				// The selected trigger arguments values.
				$selected_field = $trigger['meta']['SPECIFIC_FIELD'] ?? null;

				// The form values submitted by the user.
				$form_submitted = $hook_args->meta ?? array();
				$form_submitted = $form_submitted[ $selected_field ] ?? null;

				$user_submitted_form_id     = $hook_args->form_id ?? null;
				$user_submitted_field_id    = isset( $form_submitted['id'] ) ? sprintf( "field_%s", $form_submitted['id'] ) : null;
				$user_submitted_field_value = $form_submitted['value'] ?? null;

				// Guard clause.
				if ( is_null( $user_submitted_form_id ) || is_null( $user_submitted_field_id ) || is_null( $user_submitted_field_value ) ) {
					continue;
				}

				if ( is_array( $user_submitted_field_value ) ) {
					$user_submitted_field_value = implode( ',', $user_submitted_field_value );
				}

				// Find all recipes that match the selected trigger arguments values.
				$match = $this->find_all( $recipes )
					->where( array( $this->get_trigger_meta(), 'SPECIFIC_FIELD', 'SPECIFIC_VALUE' ) )
					->match( array( $user_submitted_form_id, $user_submitted_field_id, $user_submitted_field_value ) ) // phpcs:ignore Uncanny_Automator.PHP.ForbiddenPHP8Features.ForbiddenPHP8Feature
					->format( array( 'intval', 'trim', 'trim' ) )
					->get();

				if ( empty( $match ) ) {
					continue;
				}

				$recipes_matched[] = $match;

			}
		}

		return self::merge_arrays( $recipes_matched, true );
	}

	/**
	 * Method parse_additional_tokens.
	 *
	 * @param $parsed
	 * @param $args
	 * @param $trigger
	 *
	 * @return array
	 */
	public function parse_additional_tokens( $parsed, $args, $trigger ) {

		return $this->get_tokens_class()->wsform_field_specific_tokens( $parsed, $args, $trigger );
	}

	/**
	 * @param ...$args
	 *
	 * @return bool
	 */
	public function do_continue_anon_trigger( ...$args ) {
		return true;
	}

	/**
	* Merge nested associative arrays into one.
	*
	* @param array $arrays Input arrays
	* @param bool  $overwrite If true, later keys overwrite earlier ones
	* @return array
	*/
	private static function merge_arrays( array $arrays, bool $overwrite = false ): array {

		if ( empty( $arrays ) ) {
			return array();
		}

		$result = array();

		foreach ( $arrays as $sub_array ) {

			if ( ! is_array( $sub_array ) ) {
				continue;
			}

			foreach ( $sub_array as $k => $v ) {
				if ( $overwrite || ! array_key_exists( $k, $result ) ) {
					$result[ $k ] = $v;
				}
			}
		}

		return $result;
	}
}
