<?php

namespace Uncanny_Automator_Pro;

use Uncanny_Automator\Recipe\Action_Tokens;

/**
 * Class WP_SETTAXONOMY
 *
 * @package Uncanny_Automator_Pro
 */
class WP_SETTAXONOMY {

	use Action_Tokens;

	/**
	 * Integration code
	 *
	 * @var string
	 */
	public static $integration = 'WP';

	/**
	 * @var string
	 */
	private $action_code;
	/**
	 * @var string
	 */
	private $action_meta;

	/**
	 * Set up Automator action constructor.
	 */
	public function __construct() {

		$this->action_code = 'SETTAXONOMY';
		$this->action_meta = 'WPSETTAXONOMY';

		if ( Automator()->helpers->recipe->is_edit_page() ) {
			add_action( 'wp_loaded', array( $this, 'plugins_loaded' ), 99 );
		} else {
			$this->define_action();
		}
	}

	/**
	 * @return void
	 */
	public function plugins_loaded() {
		$this->define_action();
	}

	/**
	 * Define and register the action by pushing it into the Automator object
	 */
	public function define_action() {

		$action = array(
			'author'             => Automator()->get_author_name( $this->action_code ),
			'support_link'       => Automator()->get_author_support_link( $this->action_code, 'integration/wordpress-core/' ),
			'is_pro'             => true,
			'integration'        => self::$integration,
			'code'               => $this->action_code,
			'requires_user'      => false,
			'sentence'           => sprintf(
			/* translators: Action - WordPress Core */
				esc_html_x( 'Add {{a taxonomy:%2$s}} {{term:%3$s}} to {{a post:%1$s}} in {{a post type:%4$s}}', 'WP', 'uncanny-automator-pro' ),
				$this->action_meta,
				'WPTAXONOMIES:' . $this->action_meta,
				'WPTAXONOMYTERM:' . $this->action_meta,
				'WPSPOSTTYPES:' . $this->action_meta
			),
			/* translators: Action - WordPress Core */
			'select_option_name' => esc_html_x( 'Add {{a taxonomy}} {{term}} to {{a post}} in {{a post type}}', 'WP', 'uncanny-automator-pro' ),
			'priority'           => 11,
			'accepted_args'      => 3,
			'execution_function' => array( $this, 'set_taxonomy' ),
			'options_callback'   => array( $this, 'load_options' ),
		);


		$this->set_action_tokens(
			array(
				'TERM_ID'      => array(
					'name' => esc_html_x( 'Term ID', 'WP', 'uncanny-automator-pro' ),
					'type' => 'int',
				),
				'TERM'         => array(
					'name' => esc_html_x( 'Term', 'WP', 'uncanny-automator-pro' ),
					'type' => 'text',
				),
				'TERM_SLUG'    => array(
					'name' => esc_html_x( 'Term slug', 'WP', 'uncanny-automator-pro' ),
					'type' => 'text',
				),
				'TAXONOMY'     => array(
					'name' => esc_html_x( 'Taxonomy', 'WP', 'uncanny-automator-pro' ),
					'type' => 'text',
				),
			),
			$this->action_code
		);

		Automator()->register->action( $action );
	}

	/**
	 * @return array
	 */
	public function load_options() {

		$options = Automator()->utilities->keep_order_of_options(
			array(
				'options_group' => array(
					$this->action_meta => array(
						Automator()->helpers->recipe->wp->options->pro->all_wp_post_types_set_taxonomy(
							esc_html_x( 'Post type', 'WP', 'uncanny-automator-pro' ),
							'WPSPOSTTYPES',
							array(
								'token'        => false,
								'is_ajax'      => true,
								'target_field' => $this->action_meta,
								'is_any'       => false,
								'endpoint'     => 'select_all_post_of_selected_post_type',
							)
						),
						Automator()->helpers->recipe->field->select_field_ajax(
							$this->action_meta,
							esc_html_x( 'Post', 'WP', 'uncanny-automator-pro' ),
							array(),
							'',
							'',
							false,
							true,
							array(
								'target_field' => 'WPTAXONOMIES',
								'endpoint'     => 'select_post_type_taxonomies_SELECTEDTAXONOMY',
							)
						),
						Automator()->helpers->recipe->wp->options->pro->all_wp_taxonomy(
							esc_html_x( 'Taxonomy', 'WP', 'uncanny-automator-pro' ),
							'WPTAXONOMIES',
							array(
								'token'        => false,
								'is_ajax'      => true,
								'target_field' => 'WPTAXONOMYTERM',
								'is_any'       => false,
								'endpoint'     => 'select_all_terms_of_SELECTEDTAXONOMY',
							)
						),
						array(
							'option_code'           => 'WPTAXONOMYTERM',
							'label'                 => esc_html_x( 'Taxonomy term', 'WP', 'uncanny-automator-pro' ),
							'input_type'            => 'text',
							'supports_custom_value' => true,
							'required'              => true,
							'description'           => esc_html_x( 'Enter term name, slug, or ID. New terms will be created automatically.', 'WP', 'uncanny-automator-pro' ),
						),
					),
				),
			)
		);

		return $options;
	}

	/**
	 * Get available tokens for this action.
	 *
	 * @return array
	 */
	private function get_action_tokens() {
		return array(
			'TERM_ID'      => array(
				'name' => esc_html_x( 'Term ID', 'WP', 'uncanny-automator-pro' ),
				'type' => 'int',
			),
			'TERM'         => array(
				'name' => esc_html_x( 'Term', 'WP', 'uncanny-automator-pro' ),
				'type' => 'text',
			),
			'TERM_SLUG'    => array(
				'name' => esc_html_x( 'Term slug', 'WP', 'uncanny-automator-pro' ),
				'type' => 'text',
			),
			'TAXONOMY'     => array(
				'name' => esc_html_x( 'Taxonomy', 'WP', 'uncanny-automator-pro' ),
				'type' => 'text',
			),
		);
	}

	/**
	 * Set taxonomy.
	 *
	 * @param $user_id
	 * @param $action_data
	 * @param $recipe_id
	 * @param $args
	 */
	public function set_taxonomy( $user_id, $action_data, $recipe_id, $args ) {



		$post_type     = Automator()->parse->text( $action_data['meta']['WPSPOSTTYPES'], $recipe_id, $user_id, $args );
		$post_id       = Automator()->parse->text( $action_data['meta'][ $this->action_meta ], $recipe_id, $user_id, $args );
		$post_taxonomy = Automator()->parse->text( $action_data['meta']['WPTAXONOMIES'], $recipe_id, $user_id, $args );
		$post_term     = Automator()->parse->text( $action_data['meta']['WPTAXONOMYTERM'], $recipe_id, $user_id, $args );

		// Get or create the term
		$term_id = $this->get_or_create_term( $post_term, $post_taxonomy );

		if ( is_wp_error( $term_id ) ) {
			Automator()->complete_action( $user_id, $action_data, $recipe_id, $term_id->get_error_message() );
			return;
		}

		// Get term details for tokens
		$term = get_term( $term_id, $post_taxonomy );
		$term_name = $term && ! is_wp_error( $term ) ? $term->name : $post_term;
		$term_slug = $term && ! is_wp_error( $term ) ? $term->slug : sanitize_title( $post_term );

		if ( intval( '-1' ) === intval( $post_id ) ) {

			$args = array(
				'post_type'      => $post_type,
				'orderby'        => 'ID',
				'post_status'    => 'publish',
				'order'          => 'DESC',
				'posts_per_page' => 9999,
				// phpcs:ignore WordPress.WP.PostsPerPage.posts_per_page_posts_per_page
			);

			$result = get_posts( $args );
			if ( ! empty( $result ) ) {
				foreach ( $result as $post ) {
					if ( 'category' === $post_taxonomy ) {
						wp_set_post_categories( $post->ID, array( $term_id ), true );
					} else {
						wp_set_post_terms( $post->ID, array( $term_id ), $post_taxonomy, true );
					}
				}
			}
			
			// Set tokens for multiple posts
			$this->hydrate_tokens( array(
				'TERM_ID'      => $term_id,
				'TERM'         => $term_name,
				'TERM_SLUG'    => $term_slug,
				'TAXONOMY'     => $post_taxonomy,
			) );
			Automator()->complete_action( $user_id, $action_data, $recipe_id );

			return;
		}

		if ( 'category' === $post_taxonomy ) {

			wp_set_post_categories( $post_id, array( $term_id ), true );

		} else {

			wp_set_post_terms( $post_id, array( $term_id ), $post_taxonomy, true );

		}

		// Set tokens for single post
		$this->hydrate_tokens( array(
			'TERM_ID'      => $term_id,
			'TERM'         => $term_name,
			'TERM_SLUG'    => $term_slug,
			'TAXONOMY'     => $post_taxonomy,
		) );
		Automator()->complete_action( $user_id, $action_data, $recipe_id );
	}



	/**
	 * Get existing term or create new one if it doesn't exist.
	 *
	 * @param string $term_value Term name, slug, or ID.
	 * @param string $taxonomy   Taxonomy name.
	 *
	 * @return int|WP_Error Term ID on success, WP_Error on failure.
	 */
	private function get_or_create_term( $term_value, $taxonomy ) {
		// If it's numeric, try to get by ID first
		if ( is_numeric( $term_value ) ) {
			$term = get_term( (int) $term_value, $taxonomy );
			if ( ! is_wp_error( $term ) && $term ) {
				return $term->term_id;
			}
		}

		// Try to get by slug
		$term = get_term_by( 'slug', $term_value, $taxonomy );
		if ( $term && ! is_wp_error( $term ) ) {
			return $term->term_id;
		}

		// Try to get by name
		$term = get_term_by( 'name', $term_value, $taxonomy );
		if ( $term && ! is_wp_error( $term ) ) {
			return $term->term_id;
		}

		// Term doesn't exist, create it
		$result = wp_insert_term( $term_value, $taxonomy );

		if ( is_wp_error( $result ) ) {
			return $result;
		}

		return $result['term_id'];
	}


}
