<?php

namespace Uncanny_Automator_Pro;

/**
 * Class WC_USER_ORDER_STATUS
 *
 * @package Uncanny_Automator_Pro
 */
class WC_USER_ORDER_STATUS extends Action_Condition {

	/**
	 * Method define_condition
	 *
	 * @return void
	 */
	public function define_condition() {

		$this->integration  = 'WC';
		$this->name         = esc_html_x( "{{An order}} {{is/is not}} in {{a specific status}}", 'WC', 'uncanny-automator-pro' );
		$this->code         = 'USER_ORDER_STATUS';
		$this->dynamic_name = sprintf(
			/* translators: %1$s: order, %2$s: criteria, %3$s: status */
			esc_html_x( "{{An order:%1\$s}} {{is/is not:%2\$s}} in {{a specific status:%3\$s}}", 'WC', 'uncanny-automator-pro' ),
			'ORDER',
			'CRITERIA',
			'STATUS'
		);
		$this->is_pro        = true;
		$this->requires_user = false;
	}

	/**
	 * Method fields
	 *
	 * @return array
	 */
	public function fields() {

		$order_statuses = $this->get_woo_order_statuses();

		return array(
			$this->field->text(
				array(
					'option_code' => 'ORDER',
					'label'       => esc_html_x( 'Order', 'WC', 'uncanny-automator-pro' ),
					'required'    => true,
					'input_type'  => 'text',
					'placeholder' => '',
					'description' => esc_html_x( 'Enter the order ID or use a token', 'WC', 'uncanny-automator-pro' ),
					'supports_tokens' => true,
				)
			),
			$this->field->select_field_args(
				array(
					'option_code' => 'CRITERIA',
					'label'       => esc_html_x( 'Criteria', 'WC', 'uncanny-automator-pro' ),
					'required'    => true,
					'options_show_id'     => false,
					'options'     => array(
						array(
							'value' => 'is',
							'text'  => esc_html_x( 'is', 'WC', 'uncanny-automator-pro' ),
						),
						array(
							'value' => 'is_not',
							'text'  => esc_html_x( 'is not', 'WC', 'uncanny-automator-pro' ),
						),
					),
				)
			),
			$this->field->select_field_args(
				array(
					'option_code'           => 'STATUS',
					'label'                 => esc_html_x( 'Order status', 'WC', 'uncanny-automator-pro' ),
					'required'              => true,
					'options'               => $order_statuses,
					'supports_custom_value' => true,
				)
			),
		);
	}

	/**
	 * Get WooCommerce order statuses
	 *
	 * @return array
	 */
	private function get_woo_order_statuses() {
		if ( ! function_exists( 'wc_get_order_statuses' ) ) {
			return array();
		}

		$wc_statuses = wc_get_order_statuses();
		$return      = array();

		foreach ( $wc_statuses as $status_key => $status_label ) {
			$return[] = array(
				'value' => $status_key,
				'text'  => $status_label,
			);
		}

		return $return;
	}

	/**
	 * Evaluate_condition
	 *
	 * Has to use the $this->condition_failed( $message ); method if the condition is not met.
	 *
	 * @return void
	 */
	public function evaluate_condition() {

		$order_id = $this->get_parsed_option( 'ORDER' );
		$criteria = $this->get_parsed_option( 'CRITERIA' );
		$status   = $this->get_parsed_option( 'STATUS' );
		$status   = str_replace( 'wc-', '', $status );

		// Validate order ID
		if ( empty( $order_id ) || ! is_numeric( $order_id ) ) {
			$this->condition_failed( esc_html_x( 'Invalid order ID provided.', 'WC', 'uncanny-automator-pro' ) );
			return;
		}

		// Get the order
		$order = wc_get_order( $order_id );
		if ( ! $order ) {
			$this->condition_failed( esc_html_x( 'Order not found.', 'WC', 'uncanny-automator-pro' ) );
			return;
		}

		// Get the current order status
		$current_status = $order->get_status();

		// Evaluate the condition
		$condition_met = false;
		if ( 'is' === $criteria ) {
			$condition_met = ( $current_status === $status );
		} elseif ( 'is_not' === $criteria ) {
			$condition_met = ( $current_status !== $status );
		}

		// If the condition is not met, send an error message and mark the condition as failed.
		if ( false === $condition_met ) {

			$message = $this->generate_error_message( $current_status, $status, $criteria );

			$this->condition_failed( $message );

		}

		// If the condition is met, do nothing.
	}

	/**
	 * Check if WooCommerce dependencies are active
	 *
	 * @return bool
	 */
	protected function is_dependency_active() {
		return class_exists( 'WooCommerce' ) && function_exists( 'wc_get_order_statuses' );
	}

	/**
	 * Generate_error_message
	 *
	 * @param string $current_status
	 * @param string $expected_status
	 * @param string $criteria
	 *
	 * @return string
	 */
	public function generate_error_message( $current_status, $expected_status, $criteria ) {

		$readable_current_status  = wc_get_order_status_name( $current_status );
		$readable_expected_status = $this->get_option( 'STATUS_readable' );

		if ( 'is' === $criteria ) {
			return sprintf(
				/* translators: 1: Current order status, 2: Expected order status */
				esc_html_x( 'Order status is "%1$s" but expected "%2$s"', 'WC', 'uncanny-automator-pro' ),
				$readable_current_status,
				$readable_expected_status
			);
		} else {
			return sprintf(
				/* translators: 1: Current order status, 2: Expected order status */
				esc_html_x( 'Order status is "%1$s" but should not be "%2$s"', 'WC', 'uncanny-automator-pro' ),
				$readable_current_status,
				$readable_expected_status
			);
		}
	}
}
