<?php

namespace Uncanny_Automator_Pro\Integrations\WooCommerce_Bookings;

use Uncanny_Automator_Pro\Action_Condition;

/**
 * Class WC_BOOKINGS_PRODUCT_TAG
 *
 * @package Uncanny_Automator_Pro
 */
class WC_BOOKINGS_PRODUCT_TAG extends Action_Condition {

	/**
	 * Define_condition
	 *
	 * @return void
	 */
	public function define_condition() {

		$this->integration = 'WC_BOOKINGS';
		$this->name        = esc_html_x( 'A booking product {{contains/does not contain}} {{a specific product tag}}', 'WooCommerce Bookings', 'uncanny-automator-pro' );
		$this->code        = 'PRODUCT_TAG';
		// translators: %1$s: Criteria (contains/does not contain), %2$s: Product tags
		$this->dynamic_name  = sprintf(
			esc_html_x( 'A booking product {{contains/does not contain:%1$s}} {{a specific product tag:%2$s}}', 'WooCommerce Bookings', 'uncanny-automator-pro' ),
			'CRITERIA',
			'TAGS'
		);
		$this->is_pro        = true;
		$this->requires_user = false;
	}

	/**
	 * fields
	 *
	 * @return array
	 */
	public function fields() {

		$tags_options = $this->get_product_tags_options();

		return array(
			$this->field->text(
				array(
					'option_code' => 'PRODUCT',
					'label' => esc_html_x( 'Product', 'WooCommerce Bookings', 'uncanny-automator-pro' ),
					'required' => true,
					'input_type' => 'text',
					'placeholder' => '',
					'description' => esc_html_x( 'Enter the product ID or use a token', 'WooCommerce Bookings', 'uncanny-automator-pro' ),
					'supports_tokens' => true,
				)
			),
			$this->field->select_field_args(
				array(
					'option_code' => 'CRITERIA',
					'label' => esc_html_x( 'Criteria', 'WooCommerce Bookings', 'uncanny-automator-pro' ),
					'required' => true,
					'options' => array(
						array(
							'value' => 'contains',
							'text' => esc_html_x( 'contains', 'WooCommerce Bookings', 'uncanny-automator-pro' ),
						),
						array(
							'value' => 'not_contains',
							'text' => esc_html_x( 'does not contain', 'WooCommerce Bookings', 'uncanny-automator-pro' ),
						),
					),
					'options_show_id' => false,
					'supports_custom_value' => false,
				)
			),
			$this->field->select_field_args(
				array(
					'option_code' => 'TAGS',
					'label' => esc_html_x( 'Product tags', 'WooCommerce Bookings', 'uncanny-automator-pro' ),
					'required' => true,
					'options' => $tags_options,
					'supports_multiple_values' => true,
					'options_show_id' => false,
				)
			),
			$this->field->select_field_args(
				array(
					'option_code' => 'MATCH_TYPE',
					'label' => esc_html_x( 'Match type', 'WooCommerce Bookings', 'uncanny-automator-pro' ),
					'required' => true,
					'options' => array(
						array(
							'value' => 'any',
							'text' => esc_html_x( 'Any of the selected tags', 'WooCommerce Bookings', 'uncanny-automator-pro' ),
						),
						array(
							'value' => 'all',
							'text' => esc_html_x( 'All of the selected tags', 'WooCommerce Bookings', 'uncanny-automator-pro' ),
						),
					),
					'options_show_id' => false,
					'supports_custom_value' => false,
				)
			),
		);
	}

	/**
	 * Get product tags options
	 *
	 * @return array
	 */
	private function get_product_tags_options() {
		$tags = get_terms(
			array(
				'taxonomy' => 'product_tag',
				'hide_empty' => false,
			)
		);

		$return = array();

		if ( ! is_wp_error( $tags ) ) {
			foreach ( $tags as $tag ) {
				$return[] = array(
					'value' => $tag->term_id,
					'text' => $tag->name,
				);
			}
		}

		return $return;
	}

	/**
	 * Evaluate_condition
	 *
	 * @return void
	 */
	public function evaluate_condition() {

		$product_id    = $this->get_parsed_option( 'PRODUCT' );
		$criteria      = $this->get_parsed_option( 'CRITERIA' );
		$selected_tags = $this->get_option( 'TAGS' );
		$match_type    = $this->get_parsed_option( 'MATCH_TYPE' );

		// Validate product ID
		if ( empty( $product_id ) || ! is_numeric( $product_id ) ) {
			$this->condition_failed( esc_html_x( 'Invalid product ID provided.', 'WooCommerce Bookings', 'uncanny-automator-pro' ) );
			return;
		}

		// Get product tags
		$product_tags = \wp_get_post_terms( $product_id, 'product_tag', array( 'fields' => 'ids' ) );

		if ( is_wp_error( $product_tags ) ) {
			$this->condition_failed( esc_html_x( 'Error retrieving product tags.', 'WooCommerce Bookings', 'uncanny-automator-pro' ) );
			return;
		}

		// Convert selected tags to array if it's a string
		if ( is_string( $selected_tags ) ) {
			$selected_tags = array( $selected_tags );
		}

		// Convert to integers for comparison
		$selected_tags = array_map( 'intval', $selected_tags );
		$product_tags  = array_map( 'intval', $product_tags );

		// Check if product has the required tags
		$has_required_tags = false;

		if ( 'any' === $match_type ) {
			// Check if product has any of the selected tags
			$has_required_tags = ! empty( array_intersect( $product_tags, $selected_tags ) );
		} elseif ( 'all' === $match_type ) {
			// Check if product has all of the selected tags
			$has_required_tags = count( array_intersect( $product_tags, $selected_tags ) ) === count( $selected_tags );
		}

		// Evaluate the condition based on criteria
		$condition_met = false;
		if ( 'contains' === $criteria ) {
			$condition_met = $has_required_tags;
		} elseif ( 'not_contains' === $criteria ) {
			$condition_met = ! $has_required_tags;
		}

		// If the condition is not met, send an error message
		if ( false === $condition_met ) {
			$message = $this->generate_error_message( $criteria, $match_type, $selected_tags );
			$this->condition_failed( $message );
		}
	}

	/**
	 * Check if WooCommerce Bookings dependencies are active
	 *
	 * @return bool
	 */
	protected function is_dependency_active() {
		return class_exists( 'WooCommerce' ) && class_exists( 'WC_Bookings' );
	}

	/**
	 * Generate_error_message
	 *
	 * @param string $criteria
	 * @param string $match_type
	 * @param array $selected_tags
	 *
	 * @return string
	 */
	public function generate_error_message( $criteria, $match_type, $selected_tags ) {

		$tag_names = array();
		foreach ( $selected_tags as $tag_id ) {
			$tag = \get_term( $tag_id, 'product_tag' );
			if ( $tag && ! is_wp_error( $tag ) ) {
				$tag_names[] = $tag->name;
			}
		}

		$tags_text  = implode( ', ', $tag_names );
		$match_text = ( 'any' === $match_type ) ? esc_html_x( 'any of', 'WooCommerce Bookings', 'uncanny-automator-pro' ) : esc_html_x( 'all of', 'WooCommerce Bookings', 'uncanny-automator-pro' );

		if ( 'contains' === $criteria ) {
			return sprintf(
				/* translators: 1: Match type, 2: Tag names */
				esc_html_x( 'Booking product does not contain %1$s the tags: %2$s', 'WooCommerce Bookings', 'uncanny-automator-pro' ),
				$match_text,
				$tags_text
			);
		} else {
			return sprintf(
				/* translators: 1: Match type, 2: Tag names */
				esc_html_x( 'Booking product contains %1$s the tags: %2$s', 'WooCommerce Bookings', 'uncanny-automator-pro' ),
				$match_text,
				$tags_text
			);
		}
	}
}

new WC_BOOKINGS_PRODUCT_TAG();
