<?php

namespace Uncanny_Automator_Pro\Integrations\WooCommerce_Bookings;

use Uncanny_Automator_Pro\Action_Condition;

/**
 * Class WC_BOOKINGS_PRODUCT_CATEGORY
 *
 * @package Uncanny_Automator_Pro
 */
class WC_BOOKINGS_PRODUCT_CATEGORY extends Action_Condition {

	/**
	 * Define_condition
	 *
	 * @return void
	 */
	public function define_condition() {

		$this->integration = 'WC_BOOKINGS';
		$this->name        = esc_html_x( 'A booking product {{contains/does not contain}} {{a specific product category}}', 'WooCommerce Bookings', 'uncanny-automator-pro' );
		$this->code        = 'PRODUCT_CATEGORY';
		// translators: %1$s: Criteria (contains/does not contain), %2$s: Product categories
		$this->dynamic_name  = sprintf(
			esc_html_x( 'A booking product {{contains/does not contain:%1$s}} {{a specific product category:%2$s}}', 'WooCommerce Bookings', 'uncanny-automator-pro' ),
			'CRITERIA',
			'CATEGORIES'
		);
		$this->is_pro        = true;
		$this->requires_user = false;
	}

	/**
	 * fields
	 *
	 * @return array
	 */
	public function fields() {

		$categories_options = $this->get_product_categories_options();

		return array(
			$this->field->text(
				array(
					'option_code' => 'PRODUCT',
					'label' => esc_html_x( 'Product', 'WooCommerce Bookings', 'uncanny-automator-pro' ),
					'required' => true,
					'input_type' => 'text',
					'placeholder' => '',
					'description' => esc_html_x( 'Enter the product ID or use a token', 'WooCommerce Bookings', 'uncanny-automator-pro' ),
					'supports_tokens' => true,
				)
			),
			$this->field->select_field_args(
				array(
					'option_code' => 'CRITERIA',
					'label' => esc_html_x( 'Criteria', 'WooCommerce Bookings', 'uncanny-automator-pro' ),
					'required' => true,
					'options' => array(
						array(
							'value' => 'contains',
							'text' => esc_html_x( 'contains', 'WooCommerce Bookings', 'uncanny-automator-pro' ),
						),
						array(
							'value' => 'not_contains',
							'text' => esc_html_x( 'does not contain', 'WooCommerce Bookings', 'uncanny-automator-pro' ),
						),
					),
					'options_show_id' => false,
					'supports_custom_value' => false,
				)
			),
			$this->field->select_field_args(
				array(
					'option_code' => 'CATEGORIES',
					'label' => esc_html_x( 'Product categories', 'WooCommerce Bookings', 'uncanny-automator-pro' ),
					'required' => true,
					'options' => $categories_options,
					'supports_multiple_values' => true,
					'options_show_id' => false,
				)
			),
			$this->field->select_field_args(
				array(
					'option_code' => 'MATCH_TYPE',
					'label' => esc_html_x( 'Match type', 'WooCommerce Bookings', 'uncanny-automator-pro' ),
					'required' => true,
					'options' => array(
						array(
							'value' => 'any',
							'text' => esc_html_x( 'Any of the selected categories', 'WooCommerce Bookings', 'uncanny-automator-pro' ),
						),
						array(
							'value' => 'all',
							'text' => esc_html_x( 'All of the selected categories', 'WooCommerce Bookings', 'uncanny-automator-pro' ),
						),
					),
					'options_show_id' => false,
					'supports_custom_value' => false,
				)
			),
		);
	}

	/**
	 * Get product categories options
	 *
	 * @return array
	 */
	private function get_product_categories_options() {
		$categories = get_terms(
			array(
				'taxonomy' => 'product_cat',
				'hide_empty' => false,
			)
		);

		$return = array();

		if ( ! is_wp_error( $categories ) ) {
			foreach ( $categories as $category ) {
				$return[] = array(
					'value' => $category->term_id,
					'text' => $category->name,
				);
			}
		}

		return $return;
	}

	/**
	 * Evaluate_condition
	 *
	 * @return void
	 */
	public function evaluate_condition() {

		$product_id          = $this->get_parsed_option( 'PRODUCT' );
		$criteria            = $this->get_parsed_option( 'CRITERIA' );
		$selected_categories = $this->get_option( 'CATEGORIES' );
		$match_type          = $this->get_parsed_option( 'MATCH_TYPE' );

		// Validate product ID
		if ( empty( $product_id ) || ! is_numeric( $product_id ) ) {
			$this->condition_failed( esc_html_x( 'Invalid product ID provided.', 'WooCommerce Bookings', 'uncanny-automator-pro' ) );
			return;
		}

		// Get product categories
		$product_categories = \wp_get_post_terms( $product_id, 'product_cat', array( 'fields' => 'ids' ) );

		if ( is_wp_error( $product_categories ) ) {
			$this->condition_failed( esc_html_x( 'Error retrieving product categories.', 'WooCommerce Bookings', 'uncanny-automator-pro' ) );
			return;
		}

		// Convert selected categories to array if it's a string
		if ( is_string( $selected_categories ) ) {
			$selected_categories = array( $selected_categories );
		}

		// Convert to integers for comparison
		$selected_categories = array_map( 'intval', $selected_categories );
		$product_categories  = array_map( 'intval', $product_categories );

		// Check if product has the required categories
		$has_required_categories = false;

		if ( 'any' === $match_type ) {
			// Check if product has any of the selected categories
			$has_required_categories = ! empty( array_intersect( $product_categories, $selected_categories ) );
		} elseif ( 'all' === $match_type ) {
			// Check if product has all of the selected categories
			$has_required_categories = count( array_intersect( $product_categories, $selected_categories ) ) === count( $selected_categories );
		}

		// Evaluate the condition based on criteria
		$condition_met = false;
		if ( 'contains' === $criteria ) {
			$condition_met = $has_required_categories;
		} elseif ( 'not_contains' === $criteria ) {
			$condition_met = ! $has_required_categories;
		}

		// If the condition is not met, send an error message
		if ( false === $condition_met ) {
			$message = $this->generate_error_message( $criteria, $match_type, $selected_categories );
			$this->condition_failed( $message );
		}
	}

	/**
	 * Check if WooCommerce Bookings dependencies are active
	 *
	 * @return bool
	 */
	protected function is_dependency_active() {
		return class_exists( 'WooCommerce' ) && class_exists( 'WC_Bookings' );
	}

	/**
	 * Generate_error_message
	 *
	 * @param string $criteria
	 * @param string $match_type
	 * @param array $selected_categories
	 *
	 * @return string
	 */
	public function generate_error_message( $criteria, $match_type, $selected_categories ) {

		$category_names = array();
		foreach ( $selected_categories as $category_id ) {
			$category = \get_term( $category_id, 'product_cat' );
			if ( $category && ! is_wp_error( $category ) ) {
				$category_names[] = $category->name;
			}
		}

		$categories_text = implode( ', ', $category_names );
		$match_text      = ( 'any' === $match_type ) ? esc_html_x( 'any of', 'WooCommerce Bookings', 'uncanny-automator-pro' ) : esc_html_x( 'all of', 'WooCommerce Bookings', 'uncanny-automator-pro' );

		if ( 'contains' === $criteria ) {
			return sprintf(
				/* translators: 1: Match type, 2: Category names */
				esc_html_x( 'Booking product does not contain %1$s the categories: %2$s', 'WooCommerce Bookings', 'uncanny-automator-pro' ),
				$match_text,
				$categories_text
			);
		} else {
			return sprintf(
				/* translators: 1: Match type, 2: Category names */
				esc_html_x( 'Booking product contains %1$s the categories: %2$s', 'WooCommerce Bookings', 'uncanny-automator-pro' ),
				$match_text,
				$categories_text
			);
		}
	}
}

new WC_BOOKINGS_PRODUCT_CATEGORY();
