<?php

namespace Uncanny_Automator_Pro\Integrations\WooCommerce_Bookings;

use Uncanny_Automator_Pro\Action_Condition;

/**
 * Class WC_BOOKINGS_ORDER_STATUS
 *
 * @package Uncanny_Automator_Pro
 */
class WC_BOOKINGS_ORDER_STATUS extends Action_Condition {

	/**
	 * Define_condition
	 *
	 * @return void
	 */
	public function define_condition() {

		$this->integration = 'WC_BOOKINGS';
		$this->name        = esc_html_x( "A booking's order status {{is/is not}} {{a specific status}}", 'WooCommerce Bookings', 'uncanny-automator-pro' );
		$this->code        = 'ORDER_STATUS';
		// translators: %1$s: Criteria (is/is not), %2$s: Status
		$this->dynamic_name  = sprintf(
			esc_html_x( "A booking's order status {{is/is not:%1\$s}} {{a specific status:%2\$s}}", 'WooCommerce Bookings', 'uncanny-automator-pro' ),
			'CRITERIA',
			'STATUS'
		);
		$this->is_pro        = true;
		$this->requires_user = false;
	}

	/**
	 * fields
	 *
	 * @return array
	 */
	public function fields() {

		$order_statuses = $this->get_woo_order_statuses();

		return array(
			$this->field->text(
				array(
					'option_code' => 'BOOKING_ID',
					'label' => esc_html_x( 'Booking ID', 'WooCommerce Bookings', 'uncanny-automator-pro' ),
					'required' => true,
					'input_type' => 'text',
					'placeholder' => '',
					'description' => esc_html_x( 'Enter the booking ID or use a token', 'WooCommerce Bookings', 'uncanny-automator-pro' ),
					'supports_tokens' => true,
				)
			),
			$this->field->select_field_args(
				array(
					'option_code' => 'CRITERIA',
					'label' => esc_html_x( 'Criteria', 'WooCommerce Bookings', 'uncanny-automator-pro' ),
					'required' => true,
					'options' => array(
						array(
							'value' => 'is',
							'text' => esc_html_x( 'is', 'WooCommerce Bookings', 'uncanny-automator-pro' ),
						),
						array(
							'value' => 'is_not',
							'text' => esc_html_x( 'is not', 'WooCommerce Bookings', 'uncanny-automator-pro' ),
						),
					),
					'options_show_id' => false,
					'supports_custom_value' => false,
				)
			),
			$this->field->select_field_args(
				array(
					'option_code' => 'STATUS',
					'label' => esc_html_x( 'Order status', 'WooCommerce Bookings', 'uncanny-automator-pro' ),
					'required' => true,
					'options' => $order_statuses,
				)
			),
		);
	}

	/**
	 * Get WooCommerce order statuses
	 *
	 * @return array
	 */
	private function get_woo_order_statuses() {
		if ( ! function_exists( '\wc_get_order_statuses' ) ) {
			return array();
		}

		$wc_statuses = \wc_get_order_statuses();
		$return      = array();

		foreach ( $wc_statuses as $status_key => $status_label ) {
			$return[] = array(
				'value' => $status_key,
				'text' => $status_label,
			);
		}

		return $return;
	}

	/**
	 * Evaluate_condition
	 *
	 * @return void
	 */
	public function evaluate_condition() {

		$booking_id = $this->get_parsed_option( 'BOOKING_ID' );
		$criteria   = $this->get_parsed_option( 'CRITERIA' );
		$status     = $this->get_parsed_option( 'STATUS' );
		$status     = str_replace( 'wc-', '', $status );

		// Validate booking ID
		if ( empty( $booking_id ) || ! is_numeric( $booking_id ) ) {
			$this->condition_failed( esc_html_x( 'Invalid booking ID provided.', 'WooCommerce Bookings', 'uncanny-automator-pro' ) );
			return;
		}

		// Get the booking
		$booking = \get_wc_booking( $booking_id );
		if ( ! $booking ) {
			$this->condition_failed( esc_html_x( 'Booking not found.', 'WooCommerce Bookings', 'uncanny-automator-pro' ) );
			return;
		}

		// Get the order ID from the booking
		$order_id = $booking->get_order_id();
		if ( ! $order_id ) {
			$this->condition_failed( esc_html_x( 'Booking has no associated order.', 'WooCommerce Bookings', 'uncanny-automator-pro' ) );
			return;
		}

		// Get the order
		$order = \wc_get_order( $order_id );
		if ( ! $order ) {
			$this->condition_failed( esc_html_x( 'Order not found.', 'WooCommerce Bookings', 'uncanny-automator-pro' ) );
			return;
		}

		// Get the current order status
		$current_status = $order->get_status();

		// Evaluate the condition
		$condition_met = false;
		if ( 'is' === $criteria ) {
			$condition_met = ( $current_status === $status );
		} elseif ( 'is_not' === $criteria ) {
			$condition_met = ( $current_status !== $status );
		}

		// If the condition is not met, send an error message and mark the condition as failed.
		if ( false === $condition_met ) {

			$message = $this->generate_error_message( $current_status, $status, $criteria );

			$this->condition_failed( $message );

		}

		// If the condition is met, do nothing.
	}

	/**
	 * Check if WooCommerce Bookings dependencies are active
	 *
	 * @return bool
	 */
	protected function is_dependency_active() {
		return class_exists( 'WooCommerce' ) && class_exists( 'WC_Bookings' ) && function_exists( 'wc_get_order_statuses' ) && function_exists( '\get_wc_booking' );
	}

	/**
	 * Generate_error_message
	 *
	 * @param string $current_status
	 * @param string $expected_status
	 * @param string $criteria
	 *
	 * @return string
	 */
	public function generate_error_message( $current_status, $expected_status, $criteria ) {

		$readable_current_status  = \wc_get_order_status_name( $current_status );
		$readable_expected_status = $this->get_option( 'STATUS_readable' );

		if ( 'is' === $criteria ) {
			return sprintf(
				/* translators: 1: Current order status, 2: Expected order status */
				esc_html_x( 'Booking order status is "%1$s" but expected "%2$s"', 'WooCommerce Bookings', 'uncanny-automator-pro' ),
				$readable_current_status,
				$readable_expected_status
			);
		} else {
			return sprintf(
				/* translators: 1: Current order status, 2: Expected order status */
				esc_html_x( 'Booking order status is "%1$s" but should not be "%2$s"', 'WooCommerce Bookings', 'uncanny-automator-pro' ),
				$readable_current_status,
				$readable_expected_status
			);
		}
	}
}

new WC_BOOKINGS_ORDER_STATUS();
