<?php

namespace Uncanny_Automator_Pro\Integrations\WooCommerce_Bookings;

use Uncanny_Automator_Pro\Action_Condition;

/**
 * Class WC_BOOKINGS_DATE_RANGE
 *
 * @package Uncanny_Automator_Pro
 */
class WC_BOOKINGS_DATE_RANGE extends Action_Condition {

	/**
	 * Define_condition
	 *
	 * @return void
	 */
	public function define_condition() {

		$this->integration = 'WC_BOOKINGS';
		$this->name        = esc_html_x( 'A booking {{is/is not}} scheduled between {{a specific start date}} and {{a specific end date}}', 'WooCommerce Bookings', 'uncanny-automator-pro' );
		$this->code        = 'DATE_RANGE';
		// translators: %1$s: Criteria (is/is not), %2$s: Start date, %3$s: End date
		$this->dynamic_name  = sprintf(
			esc_html_x( 'A booking {{is/is not:%1$s}} scheduled between {{a specific start date:%2$s}} and {{a specific end date:%3$s}}', 'WooCommerce Bookings', 'uncanny-automator-pro' ),
			'CRITERIA',
			'START_DATE',
			'END_DATE'
		);
		$this->is_pro        = true;
		$this->requires_user = false;
	}

	/**
	 * fields
	 *
	 * @return array
	 */
	public function fields() {

		return array(
			$this->field->text(
				array(
					'option_code' => 'BOOKING_ID',
					'label' => esc_html_x( 'Booking ID', 'WooCommerce Bookings', 'uncanny-automator-pro' ),
					'required' => true,
					'input_type' => 'text',
					'placeholder' => '',
					'description' => esc_html_x( 'Enter the booking ID or use a token', 'WooCommerce Bookings', 'uncanny-automator-pro' ),
					'supports_tokens' => true,
				)
			),
			$this->field->select_field_args(
				array(
					'option_code' => 'CRITERIA',
					'label' => esc_html_x( 'Criteria', 'WooCommerce Bookings', 'uncanny-automator-pro' ),
					'required' => true,
					'options' => array(
						array(
							'value' => 'is',
							'text' => esc_html_x( 'is', 'WooCommerce Bookings', 'uncanny-automator-pro' ),
						),
						array(
							'value' => 'is_not',
							'text' => esc_html_x( 'is not', 'WooCommerce Bookings', 'uncanny-automator-pro' ),
						),
					),
					'options_show_id' => false,
					'supports_custom_value' => false,
					'show_label_in_sentence' => false,
				)
			),
			$this->field->text(
				array(
					'option_code' => 'START_DATE',
					'label' => esc_html_x( 'Start date', 'WooCommerce Bookings', 'uncanny-automator-pro' ),
					'required' => true,
					'input_type' => 'date',
					'description' => esc_html_x( 'Select the start date and time for the range', 'WooCommerce Bookings', 'uncanny-automator-pro' ),
				)
			),
			$this->field->text(
				array(
					'option_code' => 'END_DATE',
					'label' => esc_html_x( 'End date', 'WooCommerce Bookings', 'uncanny-automator-pro' ),
					'required' => true,
					'input_type' => 'date',
					'description' => esc_html_x( 'Select the end date and time for the range', 'WooCommerce Bookings', 'uncanny-automator-pro' ),
				)
			),
		);
	}

	/**
	 * Evaluate_condition
	 *
	 * @return void
	 */
	public function evaluate_condition() {

		$booking_id = $this->get_parsed_option( 'BOOKING_ID' );
		$criteria   = $this->get_parsed_option( 'CRITERIA' );
		$start_date = $this->get_parsed_option( 'START_DATE' );
		$end_date   = $this->get_parsed_option( 'END_DATE' );

		// Validate booking ID
		if ( empty( $booking_id ) || ! is_numeric( $booking_id ) ) {
			$this->condition_failed( esc_html_x( 'Invalid booking ID provided.', 'WooCommerce Bookings', 'uncanny-automator-pro' ) );
			return;
		}

		// Get the booking
		$booking = \get_wc_booking( $booking_id );
		if ( ! $booking ) {
			$this->condition_failed( esc_html_x( 'Booking not found.', 'WooCommerce Bookings', 'uncanny-automator-pro' ) );
			return;
		}

		// Get booking start and end times (these return timestamps)
		$booking_start_timestamp = $booking->get_start();
		$booking_end_timestamp   = $booking->get_end();

		if ( ! $booking_start_timestamp || ! $booking_end_timestamp ) {
			$this->condition_failed( esc_html_x( 'Booking has no valid start or end time.', 'WooCommerce Bookings', 'uncanny-automator-pro' ) );
			return;
		}

		// Convert timestamps to DateTime objects
		$booking_start = new \DateTime( '@' . $booking_start_timestamp );
		$booking_end   = new \DateTime( '@' . $booking_end_timestamp );

		// Parse the date range
		$range_start = new \DateTime( $start_date );
		$range_end   = new \DateTime( $end_date );

		// Check if booking overlaps with the date range
		// Booking overlaps if: booking_start < range_end AND booking_end > range_start
		$booking_overlaps = ( $booking_start < $range_end ) && ( $booking_end > $range_start );

		// Evaluate the condition
		$condition_met = false;
		if ( 'is' === $criteria ) {
			$condition_met = $booking_overlaps;
		} elseif ( 'is_not' === $criteria ) {
			$condition_met = ! $booking_overlaps;
		}

		// If the condition is not met, send an error message
		if ( false === $condition_met ) {
			$message = $this->generate_error_message( $booking_start, $booking_end, $range_start, $range_end, $criteria );
			$this->condition_failed( $message );
		}
	}

	/**
	 * Check if WooCommerce Bookings dependencies are active
	 *
	 * @return bool
	 */
	protected function is_dependency_active() {
		return class_exists( 'WooCommerce' ) && class_exists( 'WC_Bookings' );
	}

	/**
	 * Generate_error_message
	 *
	 * @param \DateTime $booking_start
	 * @param \DateTime $booking_end
	 * @param \DateTime $range_start
	 * @param \DateTime $range_end
	 * @param string $criteria
	 *
	 * @return string
	 */
	public function generate_error_message( $booking_start, $booking_end, $range_start, $range_end, $criteria ) {

		$booking_start_formatted = $booking_start->format( get_option( 'date_format' ) . ' ' . get_option( 'time_format' ) );
		$booking_end_formatted   = $booking_end->format( get_option( 'date_format' ) . ' ' . get_option( 'time_format' ) );
		$range_start_formatted   = $range_start->format( get_option( 'date_format' ) . ' ' . get_option( 'time_format' ) );
		$range_end_formatted     = $range_end->format( get_option( 'date_format' ) . ' ' . get_option( 'time_format' ) );

		if ( 'is' === $criteria ) {
			return sprintf(
				/* translators: 1: Booking start, 2: Booking end, 3: Range start, 4: Range end */
				esc_html_x( 'Booking scheduled from %1$s to %2$s does not overlap with the range %3$s to %4$s', 'WooCommerce Bookings', 'uncanny-automator-pro' ),
				$booking_start_formatted,
				$booking_end_formatted,
				$range_start_formatted,
				$range_end_formatted
			);
		} else {
			return sprintf(
				/* translators: 1: Booking start, 2: Booking end, 3: Range start, 4: Range end */
				esc_html_x( 'Booking scheduled from %1$s to %2$s overlaps with the range %3$s to %4$s', 'WooCommerce Bookings', 'uncanny-automator-pro' ),
				$booking_start_formatted,
				$booking_end_formatted,
				$range_start_formatted,
				$range_end_formatted
			);
		}
	}
}

new WC_BOOKINGS_DATE_RANGE();
