<?php

namespace Uncanny_Automator_Pro\Integrations\Github;

/**
 * Trigger: An issue is created in {a repository}
 *
 * @package Uncanny_Automator_Pro\Integrations\Github
 *
 * @property Github_Pro_App_Helpers $helpers
 * @property Github_Pro_Webhooks $webhooks
 */
class ISSUE_CREATED_IN_REPO extends \Uncanny_Automator\Recipe\App_Trigger {

	/**
	 * Setup the trigger.
	 *
	 * @return void
	 */
	protected function setup_trigger() {
		$this->set_integration( 'GITHUB' );
		$this->set_trigger_code( 'ISSUE_CREATED_IN_REPO' );
		$this->set_trigger_meta( 'ISSUE_CREATED_IN_REPO_META' );
		$this->set_sentence(
			sprintf(
				// translators: %1$s: Repository name.
				esc_attr_x( 'An issue is created in {{a repository:%1$s}}', 'GitHub', 'uncanny-automator-pro' ),
				$this->get_trigger_meta()
			)
		);
		$this->set_readable_sentence( esc_attr_x( 'An issue is created in {{a repository}}', 'GitHub', 'uncanny-automator-pro' ) );
		$this->add_action( 'automator_github_webhook_received' );
		$this->set_action_args_count( 2 );
		$this->set_trigger_type( 'anonymous' );
		$this->set_is_login_required( false );
		$this->set_is_pro( true );
	}

	/**
	 * Define trigger options.
	 *
	 * @return array
	 */
	public function options() {
		return array(
			$this->helpers->get_webhook_repo_option_config( $this->get_trigger_meta(), 'issues' ),
		);
	}

	/**
	 * Validate the trigger.
	 *
	 * @param array $trigger
	 * @param array $hook_args
	 *
	 * @return bool
	 */
	public function validate( $trigger, $hook_args ) {
		if ( ! isset( $trigger['meta'][ $this->get_trigger_meta() ], $hook_args[0], $hook_args[1] ) ) {
			return false;
		}

		// Get the repository ID, payload, and event type.
		$repo_id = $trigger['meta'][ $this->get_trigger_meta() ];
		$payload = $hook_args[0];  // GitHub payload
		$event   = $hook_args[1];  // Event type

		// Validate repository and event and action.
		$repo_matches   = $this->webhooks->webhook_matches_repository( $payload, $repo_id );
		$event_matches  = 'issues' === $event;
		$action_matches = $this->webhooks->webhook_matches_action( $payload, 'opened' );

		return $repo_matches && $event_matches && $action_matches;
	}

	/**
	 * Define tokens.
	 *
	 * @param array $trigger
	 * @param array $tokens
	 *
	 * @return array
	 */
	public function define_tokens( $trigger, $tokens ) {
		$trigger_tokens = array_merge(
			GitHub_Pro_Tokens::get_repository_token_definitions(),
			GitHub_Pro_Tokens::get_issue_token_definitions(),
			GitHub_Pro_Tokens::get_sender_token_definitions()
		);

		return array_merge( $tokens, $trigger_tokens );
	}

	/**
	 * Hydrate tokens.
	 *
	 * @param array $completed_trigger
	 * @param array $hook_args
	 *
	 * @return array
	 */
	public function hydrate_tokens( $completed_trigger, $hook_args ) {
		$payload = $hook_args[0];  // GitHub payload

		// Set the webhooks reference for the token helper.
		GitHub_Pro_Tokens::set_webhooks( $this->webhooks );

		return array_merge(
			GitHub_Pro_Tokens::hydrate_repository_tokens( $payload ),
			GitHub_Pro_Tokens::hydrate_issue_tokens( $payload ),
			GitHub_Pro_Tokens::hydrate_sender_tokens( $payload )
		);
	}
}
