<?php

namespace Uncanny_Automator_Pro\Integrations\Github;

use Uncanny_Automator\App_Integrations\App_Webhooks;
use Exception;

/**
 * Class Github_Pro_Webhooks
 *
 * @package Uncanny_Automator
 *
 * @property Github_Pro_App_Helpers $helpers
 * @property Github_Pro_Api_Caller $api
 */
class Github_Pro_Webhooks extends App_Webhooks {

	/**
	 * The option name for the webhooks config.
	 *
	 * @var string
	 */
	const MANAGER_OPTION = 'github_webhooks_manager';

	/**
	 * Get the webhooks config.
	 *
	 * @return array
	 */
	public function get_webhook_manager_config() {
		$config = automator_get_option( self::MANAGER_OPTION, array() );

		if ( empty( $config ) ) {
			$repos = $this->api->get_user_repos();
			if ( empty( $repos ) ) {
				return array();
			}

			$config = array();

			foreach ( $repos as $repo ) {
				// Add repo config using the new data structure.
				$config[ $repo['id'] ] = array(
					'id'           => $repo['id'],
					'name'         => $repo['name'],
					'owner'        => $repo['owner'],
					'admin'        => $repo['admin'],
					'hook_id'      => null, // Will get set when generated.
					'endpoint'     => $this->get_webhook_endpoint(),
					'secret'       => $this->get_webhook_key( true ),
					'events'       => array(),
					'url'          => $this->get_webhook_url(),
					'connected_at' => null,
				);
			}

			// Sort repositories by owner first, then by name for consistent grouping
			uasort(
				$config,
				function ( $a, $b ) {
					$owner_compare = strcmp( $a['owner'], $b['owner'] );
					if ( 0 !== $owner_compare ) {
						return $owner_compare;
					}
					return strcmp( $a['name'], $b['name'] );
				}
			);

			// Update the option.
			automator_update_option( self::MANAGER_OPTION, $config );
		}

		return $config;
	}

	/**
	 * Get the repo config by repo ID.
	 *
	 * @param int|string $repo_id The ID of the repo.
	 *
	 * @return array
	 */
	public function get_repo_config( $repo_id ) {
		$config = $this->get_webhook_manager_config();
		return $config[ $repo_id ] ?? array();
	}

	/**
	 * Update the repo config.
	 *
	 * @param int $repo_id The ID of the repo.
	 * @param array $repo The repo config.
	 */
	public function update_repo_config( $repo_id, $repo ) {
		$config             = $this->get_webhook_manager_config();
		$config[ $repo_id ] = $repo;
		automator_update_option( self::MANAGER_OPTION, $config );
		$this->manage_webhooks_enabled_status( $config );
	}

	/**
	 * Manage the webhooks enabled status.
	 *
	 * @param array $config The config.
	 */
	private function manage_webhooks_enabled_status( $config ) {
		$enabled = false;
		foreach ( $config as $repo ) {
			if ( ! empty( $repo['events'] ) && ! empty( $repo['hook_id'] ) ) {
				$enabled = true;
				break;
			}
		}
		$this->store_webhooks_enabled_status( $enabled );
	}

	////////////////////////////////////////////////////////////
	// Handle requests
	////////////////////////////////////////////////////////////

	/**
	 * Validate the webhook request.
	 *
	 * @param WP_REST_Request $request The request object.
	 *
	 * @return bool
	 * @throws Exception If the webhook authorization is invalid.
	 */
	protected function validate_webhook( $request ) {

		$event   = $request->get_header( 'X-GitHub-Event' );
		$hook_id = $request->get_header( 'X-GitHub-Hook-ID' );
		$repo_id = $request->get_header( 'X-GitHub-Hook-Installation-Target-ID' );
		$config  = $this->get_repo_config( $repo_id );

		// If the repo is not configured, return false.
		if ( empty( $config ) ) {
			throw new Exception( esc_html_x( 'Repo not configured for Automator', 'GitHub', 'uncanny-automator-pro' ) );
		}

		// If the hook ID is not the same as the configured hook ID, return false.
		if ( (string) $hook_id !== (string) $config['hook_id'] ) {
			throw new Exception( esc_html_x( 'Hook ID does not match configured hook ID', 'GitHub', 'uncanny-automator-pro' ) );
		}

		// Validate the GitHub signature.
		$this->validate_github_signature( $request, $config['secret'] );

		// Check for the initial webhook creation event ping event.
		if ( 'ping' === $event ) {
			// Throw exception with success message to validate but stop processing.
			throw new Exception( esc_html_x( 'Ping event received', 'GitHub', 'uncanny-automator-pro' ) );
		}

		// If the event is not the same as the configured event, return false.
		if ( ! in_array( $event, $config['events'], true ) ) {
			throw new Exception( esc_html_x( 'Event does not match allowed events', 'GitHub', 'uncanny-automator-pro' ) );
		}

		return true;
	}

	/**
	 * Set the shutdown data.
	 * - Override to use structured webhook data.
	 *
	 * @param WP_REST_Request $request The WP_REST_Request object.
	 *
	 * @return array
	 */
	protected function set_shutdown_data( $request ) {
		return array(
			'action_name'   => $this->get_do_action_name(),
			'action_params' => array(
				$this->get_decoded_request_body(), // First param: GitHub payload
				$request->get_header( 'X-GitHub-Event' ), // Second param: event type
			),
		);
	}

	/**
	 * Validate GitHub webhook signature.
	 *
	 * @param WP_REST_Request $request
	 * @param string $secret
	 *
	 * @return void
	 * @throws Exception
	 */
	private function validate_github_signature( $request, $secret ) {
		// Secret should always be set in the config.
		if ( empty( $secret ) ) {
			throw new Exception( esc_html_x( 'Secret is required', 'GitHub', 'uncanny-automator-pro' ) );
		}

		// Validate the request signature.
		$body      = $request->get_body();
		$signature = $request->get_header( 'X-Hub-Signature-256' );

		if ( empty( $signature ) ) {
			throw new Exception( esc_html_x( 'Unauthorized request signature', 'GitHub', 'uncanny-automator-pro' ) );
		}

		$expected_signature = 'sha256=' . hash_hmac( 'sha256', $body, $secret );
		if ( ! hash_equals( $expected_signature, $signature ) ) {
			throw new Exception( esc_html_x( 'Invalid request signature', 'GitHub', 'uncanny-automator-pro' ) );
		}
	}

	////////////////////////////////////////////////////////////
	// Recipe trigger validation helpers.
	////////////////////////////////////////////////////////////

	/**
	 * Check if webhook matches a specific repository
	 *
	 * @param array $data
	 * @param string|int $repo_id
	 *
	 * @return bool
	 */
	public function webhook_matches_repository( $data, $repo_id ) {
		$webhook_repo_id = $data['repository']['id'] ?? null;
		return (string) $webhook_repo_id === (string) $repo_id;
	}

	/**
	 * Check if webhook matches a specific action
	 *
	 * @param array $data
	 * @param string $action
	 *
	 * @return bool
	 */
	public function webhook_matches_action( $data, $action ) {
		$data_action = $data['action'] ?? null;
		return $action === $data_action;
	}

	/**
	 * Get formatted rich text content from GitHub webhook payload
	 *
	 * Handles GitHub-specific content like issue bodies, pull request descriptions,
	 * and release notes that contain markdown formatting. Converts markdown to HTML
	 * for proper rendering in email templates and other HTML contexts.
	 *
	 * @param array  $data The webhook payload data
	 * @param string $path Dot notation path (e.g., 'issue.body', 'pull_request.body')
	 * @param array  $options Optional formatting options
	 *
	 * @return string The formatted HTML content
	 */
	public function get_rich_text_value( $data, $path, $options = array() ) {
		// Get the raw value first using the parent's method
		$raw_value = $this->get_payload_value( $data, $path );

		if ( empty( $raw_value ) || ! is_string( $raw_value ) ) {
			return '';
		}

		// Use the markdown parser service to convert the content
		$parser          = new \Uncanny_Automator\Services\Markdown\Markdown_Parser();
		$formatted_value = $parser->parse( $raw_value, $options );

		/**
		 * Filter the final GitHub rich text HTML output.
		 *
		 * Allows advanced users to make final modifications to the processed
		 * HTML content before it's returned as a token value.
		 *
		 * @since 5.1
		 *
		 * @param string $formatted_value The processed HTML content
		 * @param string $raw_value       The original markdown content
		 * @param string $path            The data path being processed
		 * @param array  $data            The full webhook payload data
		 * @param array  $options         The formatting options used
		 *
		 * @return string Modified HTML content
		 */
		$formatted_value = apply_filters( 'automator_github_rich_text_html', $formatted_value, $raw_value, $path, $data );

		return $formatted_value;
	}
}
