<?php

namespace Uncanny_Automator_Pro\Integrations\Github;

/**
 * Class Github_Pro_App_Helpers
 *
 * Extends the free version of the Github app helpers to include:
 * - Pro webhook events.
 * - Pro webhook repo options.
 * - Pro webhook repo option config.
 *
 * @property Github_Pro_Api_Caller $api
 * @property Github_Pro_Webhooks $webhooks
 */
class Github_Pro_App_Helpers extends \Uncanny_Automator\Integrations\Github\Github_App_Helpers {

	////////////////////////////////////////////////////////////
	// Recipe UI helper methods
	////////////////////////////////////////////////////////////

	/**
	 * Get repository options.
	 *
	 * @param string $event
	 *
	 * @return array
	 */
	public function get_webhook_repo_options( $event = 'all' ) {
		$webhook_config = $this->webhooks->get_webhook_manager_config();
		$options        = array();
		foreach ( $webhook_config as $repo_id => $repo ) {
			// Ensure the webhook has been connected.
			if ( empty( $repo['events'] ) || empty( $repo['hook_id'] ) ) {
				continue;
			}

			// Ensure the event is in the webhook's events.
			if ( 'all' !== $event && ! in_array( $event, $repo['events'], true ) ) {
				continue;
			}

			$options[] = array(
				'text'  => $repo['owner'] . ' : ' . $repo['name'],
				'value' => $repo_id,
			);
		}

		return $options;
	}

	/**
	 * Get repository option config.
	 *
	 * @param string $option_code
	 * @param string $event
	 *
	 * @return array
	 */
	public function get_webhook_repo_option_config( $option_code, $event ) {
		return array(
			'input_type'      => 'select',
			'option_code'     => $option_code,
			'label'           => esc_html_x( 'Repository', 'GitHub', 'uncanny-automator-pro' ),
			'required'        => true,
			'options'         => $this->get_webhook_repo_options( $event ),
			'options_show_id' => false,
			'relevant_tokens' => array(),
		);
	}

	/**
	 * Get event options.
	 *
	 * @return array
	 */
	public function get_event_options() {
		$events = array(
			array(
				'text'  => esc_html_x( 'Push', 'GitHub', 'uncanny-automator-pro' ),
				'value' => 'push',
			),
			array(
				'text'  => esc_html_x( 'Pull request', 'GitHub', 'uncanny-automator-pro' ),
				'value' => 'pull_request',
			),
			array(
				'text'  => esc_html_x( 'Issues', 'GitHub', 'uncanny-automator-pro' ),
				'value' => 'issues',
			),
			array(
				'text'  => esc_html_x( 'Issue comment', 'GitHub', 'uncanny-automator-pro' ),
				'value' => 'issue_comment',
			),
			array(
				'text'  => esc_html_x( 'Pull request review', 'GitHub', 'uncanny-automator-pro' ),
				'value' => 'pull_request_review',
			),
			array(
				'text'  => esc_html_x( 'Release', 'GitHub', 'uncanny-automator-pro' ),
				'value' => 'release',
			),

			array(
				'text'  => esc_html_x( 'Delete', 'GitHub', 'uncanny-automator-pro' ),
				'value' => 'delete',
			),
			array(
				'text'  => esc_html_x( 'Fork', 'GitHub', 'uncanny-automator-pro' ),
				'value' => 'fork',
			),
			array(
				'text'  => esc_html_x( 'Star', 'GitHub', 'uncanny-automator-pro' ),
				'value' => 'star',
			),
			array(
				'text'  => esc_html_x( 'Commit comment', 'GitHub', 'uncanny-automator-pro' ),
				'value' => 'commit_comment',
			),
			array(
				'text'  => esc_html_x( 'Workflow run', 'GitHub', 'uncanny-automator-pro' ),
				'value' => 'workflow_run',
			),
		);

		/**
		 * Filter GitHub webhook events.
		 *
		 * @link https://docs.github.com/en/developers/webhooks-and-events/webhooks/webhook-events-and-payloads
		 *
		 * @param array $events
		 * @property text - The text to display for the event.
		 * @property value - The value to use for the event.
		 *
		 * @return array
		 *
		 * @example:
		 * add_filter( 'automator_github_pro_webhook_events', function( $events ) {
		 *     $events[] = array(
		 *         'text'  => esc_html_x( 'Team member added', 'GitHub', 'uncanny-automator-pro' ),
		 *         'value' => 'team_add',
		 *     );
		 *     return $events;
		 * } );
		 */
		$events = apply_filters( 'automator_github_pro_webhook_events', $events );

		return $events;
	}

	/**
	 * Get event action options.
	 *
	 * @return array
	 */
	public function get_event_action_options_ajax() {

		Automator()->utilities->verify_nonce();

		$values = automator_filter_has_var( 'values', INPUT_POST )
			? automator_filter_input_array( 'values', INPUT_POST )
			: array();

		$actions = $this->get_event_actions_options( $values['EVENT_TYPE'] ?? '' );
		$options = array(
			array(
				'text'  => esc_html_x( 'Any action', 'GitHub', 'uncanny-automator-pro' ),
				'value' => '-1',
			),
		);

		// Merge in any found actions.
		if ( ! empty( $actions ) ) {
			// Add the actions for the event type.
			$options = array_merge( $options, $actions );
		}

		wp_send_json(
			array(
				'success' => true,
				'options' => $options,
			)
		);
	}

	/**
	 * Get event actions options by event type.
	 *
	 * @param string $event
	 *
	 * @return array
	 */
	public function get_event_actions_options( $event ) {

		// Bail if empty or invalid type.
		if ( empty( $event ) || ! is_string( $event ) ) {
			return array();
		}

		// Build filterable event type action options.
		switch ( $event ) {
			case 'release':
				$actions = array(
					array(
						'text'  => esc_html_x( 'Published', 'GitHub', 'uncanny-automator-pro' ),
						'value' => 'published',
					),
					array(
						'text'  => esc_html_x( 'Unpublished', 'GitHub', 'uncanny-automator-pro' ),
						'value' => 'unpublished',
					),
					array(
						'text'  => esc_html_x( 'Created', 'GitHub', 'uncanny-automator-pro' ),
						'value' => 'created',
					),
					array(
						'text'  => esc_html_x( 'Edited', 'GitHub', 'uncanny-automator-pro' ),
						'value' => 'edited',
					),
					array(
						'text'  => esc_html_x( 'Deleted', 'GitHub', 'uncanny-automator-pro' ),
						'value' => 'deleted',
					),
					array(
						'text'  => esc_html_x( 'Prereleased', 'GitHub', 'uncanny-automator-pro' ),
						'value' => 'prereleased',
					),
					array(
						'text'  => esc_html_x( 'Released', 'GitHub', 'uncanny-automator-pro' ),
						'value' => 'released',
					),
				);
				break;
			case 'workflow_run':
				$actions = array(
					array(
						'text'  => esc_html_x( 'Completed', 'GitHub', 'uncanny-automator-pro' ),
						'value' => 'completed',
					),
					array(
						'text'  => esc_html_x( 'Requested', 'GitHub', 'uncanny-automator-pro' ),
						'value' => 'requested',
					),
					array(
						'text'  => esc_html_x( 'In progress', 'GitHub', 'uncanny-automator-pro' ),
						'value' => 'in_progress',
					),
				);
				break;
			default:
				$actions = array();
				break;
		}

		/**
		 * Filter GitHub webhook event type actions.
		 *
		 * @link https://docs.github.com/en/developers/webhooks-and-events/webhooks/webhook-events-and-payloads
		 *
		 * @param array $actions
		 * @property text - The text to display for the event.
		 * @property value - The value to use for the event.
		 *
		 * @return array
		 *
		 * @example:
		 * add_filter( 'automator_github_pro_webhook_event_actions', function( $actions, $event ) {
		 *     if ( 'workflow_run' === $event ) {
		 *         $actions[] = array(
		 *             'text'  => esc_html_x( 'Completed', 'GitHub', 'uncanny-automator-pro' ),
		 *             'value' => 'completed',
		 *         );
		 *     }
		 *     return $actions;
		 * }, 10, 2 );
		 * } );
		 */
		return apply_filters( 'automator_github_pro_webhook_event_actions', $actions, $event );
	}
}
