<?php

namespace Uncanny_Automator_Pro\Integrations\Github;

use Uncanny_Automator\Integrations\Github\Github_Api_Caller;
use Exception;

/**
 * Class Github_Pro_Api_Caller
 *
 * Extends the free version of the Github API caller to include:
 * - webhook creation and updates.
 *
 * @package Uncanny_Automator
 *
 * @property Github_Pro_App_Helpers $helpers
 * @property Github_Pro_Webhooks $webhooks
 */
class Github_Pro_Api_Caller extends Github_Api_Caller {

	/**
	 * Create a webhook.
	 *
	 * @param int $repo_id The ID of the repo.
	 * @param array $events The events to subscribe to.
	 *
	 * @return array
	 * @throws Exception
	 */
	public function create_webhook( $repo_id, $events ) {

		if ( empty( $events ) ) {
			throw new Exception( esc_html_x( 'Please select at least one event to subscribe your webhook to.', 'GitHub', 'uncanny-automator-pro' ) );
		}

		$repo = $this->get_repo_config( $repo_id );

		// Build request args.
		$args = array(
			'action'      => 'create_webhook',
			'owner'       => $repo['owner'],
			'repo'        => $repo['name'],
			'webhook_url' => $repo['url'],
			'events'      => wp_json_encode( $events ),
			'secret'      => $repo['secret'],
		);

		// Make the API request.
		$response = $this->api_request( $args );
		if ( 201 !== $response['statusCode'] ) {
			throw new Exception( esc_html( $response['data']['message'] ) );
		}

		// Update the repo config.
		$repo['hook_id']      = $response['data']['id'];
		$repo['events']       = $events;
		$repo['connected_at'] = time();
		$this->webhooks->update_repo_config( $repo_id, $repo );

		return $repo;
	}

	/**
	 * Update a webhook.
	 *
	 * @param int $repo_id The ID of the repo.
	 * @param array $events The events to subscribe to.
	 *
	 * @return array
	 * @throws Exception
	 */
	public function update_webhook( $repo_id, $events ) {

		if ( empty( $events ) ) {
			throw new Exception( esc_html_x( 'Please select at least one event to subscribe your webhook to.', 'GitHub', 'uncanny-automator-pro' ) );
		}

		$repo = $this->get_repo_config( $repo_id );

		// Compare the events to the existing events
		$existing_events = $repo['events'];
		$new_events      = $events;

		sort( $new_events );
		sort( $existing_events );

		if ( $new_events === $existing_events ) {
			throw new Exception( esc_html_x( 'No changes to the webhook events.', 'GitHub', 'uncanny-automator-pro' ) );
		}

		// Build request args.
		$args = array(
			'action'      => 'update_webhook',
			'webhook_id'  => $repo['hook_id'],
			'owner'       => $repo['owner'],
			'repo'        => $repo['name'],
			'webhook_url' => $repo['url'],
			'events'      => wp_json_encode( $events ),
			'secret'      => $repo['secret'],
		);

		// Make the API request.
		$response = $this->api_request( $args );
		if ( 200 !== $response['statusCode'] ) {
			throw new Exception( esc_html( $response['data']['message'] ) );
		}

		// Update the repo config.
		$repo['events'] = $events;
		$this->webhooks->update_repo_config( $repo_id, $repo );

		return $repo;
	}

	/**
	 * Delete a webhook.
	 *
	 * @param int $repo_id The ID of the repo.
	 *
	 * @return array
	 * @throws Exception
	 */
	public function delete_webhook( $repo_id ) {
		$repo = $this->get_repo_config( $repo_id );
		// Build request args.
		$args = array(
			'action'     => 'delete_webhook',
			'owner'      => $repo['owner'],
			'repo'       => $repo['name'],
			'webhook_id' => $repo['hook_id'],
		);

		// Make the API request.
		$response = $this->api_request( $args );
		if ( 204 !== $response['statusCode'] ) {
			throw new Exception( esc_html( $response['data']['message'] ) );
		}

		// Update the repo config.
		$repo['hook_id']      = null;
		$repo['connected_at'] = null;
		$repo['events']       = array();
		$this->webhooks->update_repo_config( $repo_id, $repo );

		return $repo;
	}

	/**
	 * Get the repo config by repo ID.
	 *
	 * @param int $repo_id The ID of the repo.
	 *
	 * @return array
	 * @throws Exception
	 */
	private function get_repo_config( $repo_id ) {
		$repo = $this->webhooks->get_repo_config( $repo_id );
		if ( empty( $repo ) ) {
			throw new Exception( esc_html_x( 'Repo not found.', 'GitHub', 'uncanny-automator-pro' ) );
		}

		return $repo;
	}
}
