<?php

namespace Uncanny_Automator_Pro\Integrations\Easy_Digital_Downloads;

/**
 * Class EDD_USER_SUBSCRIPTION_EXPIRES
 *
 * @package Uncanny_Automator_Pro
 */
class EDD_USER_SUBSCRIPTION_EXPIRES extends \Uncanny_Automator\Recipe\Trigger {

	/**
	 * @return mixed|void
	 */
	protected function setup_trigger() {

		if ( ! class_exists( 'EDD_Recurring' ) ) {
			return;
		}

		$this->set_integration( 'EDD' );
		$this->set_trigger_code( 'EDDR_SUBSCRIPTION_EXPIRES' );
		$this->set_trigger_meta( 'EDDR_PRODUCTS' );
		$this->set_is_pro( true );
		$this->set_sentence(
			sprintf(
			// translators: %1$s: Download
				esc_html_x( "A user's subscription to {{a download:%1\$s}} expires", 'Easy Digital Downloads', 'uncanny-automator-pro' ),
				$this->get_trigger_meta()
			)
		);
		$this->set_readable_sentence( esc_html_x( "A user's subscription to {{a download}} expires", 'Easy Digital Downloads', 'uncanny-automator-pro' ) );
		$this->add_action( 'edd_subscription_expired', 10, 2 );
	}

	/**
	 * Options.
	 *
	 * @return array
	 */
	public function options() {
		return array(
			$this->get_item_helpers()->all_edd_downloads( esc_html_x( 'Download', 'Easy Digital Downloads', 'uncanny-automator' ), $this->get_trigger_meta(), true, true, true ),
		);
	}

	/**
	 * @return bool
	 */
	public function validate( $trigger, $hook_args ) {
		if ( ! isset( $trigger['meta'][ $this->get_trigger_meta() ] ) ) {
			return false;
		}

		$selected_product_id = $trigger['meta'][ $this->get_trigger_meta() ];
		$subscription_object = $hook_args[1];

		if ( intval( '-1' ) !== intval( $selected_product_id ) && absint( $selected_product_id ) !== absint( $subscription_object->product_id ) ) {
			return false;
		}

		// Set user ID for the trigger from customer data
		$user_id = 0;
		if ( ! empty( $subscription_object->customer_id ) ) {
			$customer = \edd_get_customer( $subscription_object->customer_id );
			if ( $customer && ! empty( $customer->user_id ) ) {
				$user_id = $customer->user_id;
			}
		}

		// Fallback to current user ID if no customer user ID found
		if ( ! $user_id ) {
			$user_id = get_current_user_id();
		}

		if ( $user_id ) {
			$this->set_user_id( $user_id );
		}

		return true;
	}

	/**
	 * define_tokens
	 *
	 * @param mixed $tokens
	 * @param mixed $trigger - options selected in the current recipe/trigger
	 *
	 * @return array
	 */
	public function define_tokens( $trigger, $tokens ) {

		$tokens[] = array(
			'tokenId'   => 'EDDR_EXPIRATION_DATE',
			'tokenName' => esc_html_x( 'Expiration date', 'Easy Digital Downloads', 'uncanny-automator-pro' ),
			'tokenType' => 'date',
		);

		$tokens[] = array(
			'tokenId'   => 'EDD_CUSTOMER_ID',
			'tokenName' => esc_html_x( 'Customer ID', 'Easy Digital Downloads', 'uncanny-automator-pro' ),
			'tokenType' => 'text',
		);

		$tokens[] = array(
			'tokenId'   => 'EDD_CUSTOMER_EMAIL',
			'tokenName' => esc_html_x( 'Customer email', 'Easy Digital Downloads', 'uncanny-automator-pro' ),
			'tokenType' => 'text',
		);

		$tokens[] = array(
			'tokenId'   => 'EDD_CUSTOMER_NAME',
			'tokenName' => esc_html_x( 'Customer name', 'Easy Digital Downloads', 'uncanny-automator-pro' ),
			'tokenType' => 'text',
		);

		$tokens[] = array(
			'tokenId'   => 'EDD_CUSTOMER_STATUS',
			'tokenName' => esc_html_x( 'Customer status', 'Easy Digital Downloads', 'uncanny-automator-pro' ),
			'tokenType' => 'text',
		);

		$tokens[] = array(
			'tokenId'   => 'EDD_CUSTOMER_USER_ID',
			'tokenName' => esc_html_x( 'Customer user ID', 'Easy Digital Downloads', 'uncanny-automator-pro' ),
			'tokenType' => 'text',
		);

		return $tokens;
	}

	/**
	 * hydrate_tokens
	 *
	 * @param $trigger
	 * @param $hook_args
	 *
	 * @return array
	 */
	public function hydrate_tokens( $trigger, $hook_args ) {

		$subscription_object = $hook_args[1];

		if ( ! class_exists( '\EDD_Payment' ) ) {
			return;
		}

		$payment = new \EDD_Payment( $subscription_object->parent_payment_id );

		// Get customer information.
		$customer_id      = '';
		$customer_email   = '';
		$customer_name    = '';
		$customer_status  = '';
		$customer_user_id = '';

		if ( ! empty( $subscription_object->customer_id ) ) {
			$customer = \edd_get_customer( $subscription_object->customer_id );
			if ( $customer ) {
				$customer_id      = $customer->id;
				$customer_email   = $customer->email;
				$customer_name    = $customer->name;
				$customer_status  = $customer->status;
				$customer_user_id = $customer->user_id;
			}
		}

		$token_values = array(
			'EDDR_EXPIRATION_DATE'          => date_i18n( get_option( 'date_format' ), strtotime( $subscription_object->expiration ) ),
			'EDDR_PRODUCTS_DISCOUNT_CODES'  => $payment->discounts,
			'EDDR_PRODUCTS'                 => get_the_title( $subscription_object->product_id ),
			'EDDR_PRODUCTS_ID'              => $subscription_object->product_id,
			'EDDR_PRODUCTS_URL'             => get_permalink( $subscription_object->product_id ),
			'EDDR_PRODUCTS_THUMB_ID'        => get_post_thumbnail_id( $subscription_object->product_id ),
			'EDDR_PRODUCTS_THUMB_URL'       => get_the_post_thumbnail_url( $subscription_object->product_id ),
			'EDDR_PRODUCTS_ORDER_DISCOUNTS' => number_format( $payment->discounted_amount, 2 ),
			'EDDR_PRODUCTS_ORDER_SUBTOTAL'  => number_format( $payment->subtotal, 2 ),
			'EDDR_PRODUCTS_ORDER_TAX'       => number_format( $payment->tax, 2 ),
			'EDDR_PRODUCTS_ORDER_TOTAL'     => number_format( $payment->total, 2 ),
			'EDDR_PRODUCTS_PAYMENT_METHOD'  => $payment->gateway,
			'EDD_CUSTOMER_ID'               => $customer_id,
			'EDD_CUSTOMER_EMAIL'            => $customer_email,
			'EDD_CUSTOMER_NAME'             => $customer_name,
			'EDD_CUSTOMER_STATUS'           => $customer_status,
			'EDD_CUSTOMER_USER_ID'          => $customer_user_id,
		);

		return $token_values;
	}
}
