<?php

namespace Uncanny_Automator_Pro\Integrations\Easy_Digital_Downloads;

use EDD_Payment;
use Uncanny_Automator\Recipe\Trigger;

/**
 * Class EDD_PRODPURCHWITHDISCOUNTCODE
 *
 * @package Uncanny_Automator_Pro
 */
class EDD_PRODPURCHWITHDISCOUNTCODE extends Trigger {

	/**
	 * @return mixed|void
	 */
	protected function setup_trigger() {
		$this->set_integration( 'EDD' );
		$this->set_trigger_code( 'EDDPRODPURCHDISCOUNT' );
		$this->set_trigger_meta( 'EDDDISCOUNTCODE' );
		$this->set_is_pro( true );
		$this->set_sentence(
			// translators: %1$s: Download, %2$s: Discount code
			sprintf( esc_html_x( '{{A product:%1$s}} is purchased with {{a discount code:%2$s}}', 'Easy Digital Downloads', 'uncanny-automator-pro' ), 'EDDPRODUCTS:' . $this->get_trigger_meta(), 'EDDDISCOUNTCODE:' . $this->get_trigger_meta() )
		);
		$this->set_readable_sentence( esc_html_x( '{{A product}} is purchased with {{a discount code}}', 'Easy Digital Downloads', 'uncanny-automator-pro' ) );
		$this->add_action( 'edd_complete_purchase', 10, 3 );
	}

	/**
	 * Options.
	 *
	 * @return array
	 */
	public function options() {
		return array(
			$this->get_item_helpers()->all_edd_downloads( esc_html_x( 'Download', 'Easy Digital Downloads', 'uncanny-automator' ), 'EDDPRODUCTS' ),
			$this->get_item_helpers()->all_edd_discount_codes( esc_html_x( 'Discount code', 'Easy Digital Downloads', 'uncanny-automator' ), 'EDDDISCOUNTCODE' ),
		);
	}

	/**
	 * @param $trigger
	 * @param $hook_args
	 *
	 * @return bool
	 */
	public function validate( $trigger, $hook_args ) {
		if ( ! isset( $hook_args[0], $hook_args[1], $hook_args[2] ) ) {
			return false;
		}

		$order_id = $hook_args[0];
		$payment  = $hook_args[1];
		$customer = $hook_args[2];

		if ( ! $payment instanceof EDD_Payment ) {
			return false;
		}

		$cart_items = edd_get_payment_meta_cart_details( $order_id );

		if ( empty( $cart_items ) ) {
			return false;
		}

		if ( 'none' === $payment->user_info['discount'] ) {
			return false;
		}

		// Get required values from trigger
		$required_discount_code = $trigger['meta']['EDDDISCOUNTCODE'] ?? '';
		$required_product       = $trigger['meta']['EDDPRODUCTS'] ?? '';

		$code_id = edd_get_discount_id_by_code( $payment->user_info['discount'] );

		// Check if discount code matches
		if ( '-1' !== $required_discount_code && absint( $required_discount_code ) !== absint( $code_id ) ) {
			return false;
		}

		// Check if product matches
		if ( '-1' !== $required_product ) {
			$product_found = false;
			foreach ( $cart_items as $item ) {
				if ( absint( $required_product ) === absint( $item['id'] ) ) {
					$product_found = true;
					break;
				}
			}
			if ( ! $product_found ) {
				return false;
			}
		}

		// Set the correct user ID from customer data
		$user_id = 0;
		if ( ! empty( $payment->customer_id ) ) {
			$customer_obj = edd_get_customer( $payment->customer_id );
			if ( $customer_obj && ! empty( $customer_obj->user_id ) ) {
				$user_id = $customer_obj->user_id;
			}
		}

		// Fallback to current user ID if no customer user ID found
		if ( ! $user_id ) {
			$user_id = get_current_user_id();
		}

		if ( $user_id ) {
			$this->set_user_id( $user_id );
		}

		return true;
	}

	/**
	 * @param $trigger
	 * @param $tokens
	 *
	 * @return array|array[]
	 */
	public function define_tokens( $trigger, $tokens ) {
		$trigger_tokens = array(
			array(
				'tokenId'   => 'EDDPRODUCTS',
				'tokenName' => esc_html_x( 'Download title', 'Easy Digital Downloads', 'uncanny-automator-pro' ),
				'tokenType' => 'text',
			),
			array(
				'tokenId'   => 'EDDPRODUCTS_ID',
				'tokenName' => esc_html_x( 'Download ID', 'Easy Digital Downloads', 'uncanny-automator-pro' ),
				'tokenType' => 'int',
			),
			array(
				'tokenId'   => 'EDDPRODUCTS_URL',
				'tokenName' => esc_html_x( 'Download URL', 'Easy Digital Downloads', 'uncanny-automator-pro' ),
				'tokenType' => 'url',
			),
			array(
				'tokenId'   => 'EDDPRODUCTS_THUMB_ID',
				'tokenName' => esc_html_x( 'Download featured image ID', 'Easy Digital Downloads', 'uncanny-automator-pro' ),
				'tokenType' => 'int',
			),
			array(
				'tokenId'   => 'EDDPRODUCTS_THUMB_URL',
				'tokenName' => esc_html_x( 'Download featured image URL', 'Easy Digital Downloads', 'uncanny-automator-pro' ),
				'tokenType' => 'url',
			),
			array(
				'tokenId'   => 'EDDPRODUCTS_DISCOUNT_CODES',
				'tokenName' => esc_html_x( 'Discount codes used', 'Easy Digital Downloads', 'uncanny-automator-pro' ),
				'tokenType' => 'text',
			),
			array(
				'tokenId'   => 'EDDDISCOUNTCODE',
				'tokenName' => esc_html_x( 'Discount code', 'Easy Digital Downloads', 'uncanny-automator-pro' ),
				'tokenType' => 'text',
			),
			array(
				'tokenId'   => 'EDD_CUSTOMER_ID',
				'tokenName' => esc_html_x( 'Customer ID', 'Easy Digital Downloads', 'uncanny-automator-pro' ),
				'tokenType' => 'int',
			),
			array(
				'tokenId'   => 'EDD_CUSTOMER_EMAIL',
				'tokenName' => esc_html_x( 'Customer email', 'Easy Digital Downloads', 'uncanny-automator-pro' ),
				'tokenType' => 'text',
			),
			array(
				'tokenId'   => 'EDD_CUSTOMER_NAME',
				'tokenName' => esc_html_x( 'Customer name', 'Easy Digital Downloads', 'uncanny-automator-pro' ),
				'tokenType' => 'text',
			),
			array(
				'tokenId'   => 'EDD_CUSTOMER_STATUS',
				'tokenName' => esc_html_x( 'Customer status', 'Easy Digital Downloads', 'uncanny-automator-pro' ),
				'tokenType' => 'text',
			),
			array(
				'tokenId'   => 'EDD_CUSTOMER_USER_ID',
				'tokenName' => esc_html_x( 'Customer user ID', 'Easy Digital Downloads', 'uncanny-automator-pro' ),
				'tokenType' => 'int',
			),
		);

		return array_merge( $tokens, $trigger_tokens );
	}

	/**
	 * hydrate_tokens
	 *
	 * @param $trigger
	 * @param $hook_args
	 *
	 * @return array
	 */
	public function hydrate_tokens( $trigger, $hook_args ) {
		$order_id = $hook_args[0];
		$payment  = $hook_args[1];
		$customer = $hook_args[2];

		if ( ! $payment instanceof EDD_Payment ) {
			return array(
				'EDDPRODUCTS'                => '',
				'EDDPRODUCTS_ID'             => '',
				'EDDPRODUCTS_URL'            => '',
				'EDDPRODUCTS_THUMB_ID'       => '',
				'EDDPRODUCTS_THUMB_URL'      => '',
				'EDDPRODUCTS_DISCOUNT_CODES' => '',
				'EDDPRODUCTS_LICENSE_KEY'    => '',
				'EDDPRODUCTS_ORDER_DISCOUNTS'=> '',
				'EDDPRODUCTS_ORDER_SUBTOTAL' => '',
				'EDDPRODUCTS_ORDER_TAX'      => '',
				'EDDPRODUCTS_ORDER_TOTAL'    => '',
				'EDDPRODUCTS_PAYMENT_METHOD' => '',
				'EDDDISCOUNTCODE'            => '',
				'EDD_CUSTOMER_ID'            => '',
				'EDD_CUSTOMER_EMAIL'         => '',
				'EDD_CUSTOMER_NAME'          => '',
				'EDD_CUSTOMER_STATUS'        => '',
				'EDD_CUSTOMER_USER_ID'       => '',
			);
		}

		$cart_items = edd_get_payment_meta_cart_details( $order_id );

		// Get the order object and its discount
		$order          = edd_get_order( $order_id );
		$total_discount = $order ? $order->discount : 0;

		// Process all items to create CSV strings
		$product_names = array();
		$product_ids   = array();
		$product_urls  = array();

		foreach ( $cart_items as $item ) {
			$product_names[] = $item['name'];
			$product_ids[]   = $item['id'];
			$product_urls[]  = get_permalink( $item['id'] );
		}

		$license_key = '';
		// Check if get_licenses already exists in Automator free.
		if ( isset( Automator()->helpers->recipe->edd->options ) &&
			method_exists( Automator()->helpers->recipe->edd->options, 'get_licenses' ) ) {
			$license_key = Automator()->helpers->recipe->edd->options->get_licenses( $order_id );
		}

		// Get product thumbnails
		$product_thumb_ids  = array();
		$product_thumb_urls = array();

		foreach ( $cart_items as $item ) {
			$product_thumb_ids[]  = get_post_thumbnail_id( $item['id'] );
			$product_thumb_urls[] = get_the_post_thumbnail_url( $item['id'] );
		}

		// Get customer information.
		$customer_id      = '';
		$customer_email   = '';
		$customer_name    = '';
		$customer_status  = '';
		$customer_user_id = '';

		if ( ! empty( $payment->customer_id ) ) {
			$customer = edd_get_customer( $payment->customer_id );
			if ( $customer ) {
				$customer_id      = $customer->id;
				$customer_email   = $customer->email;
				$customer_name    = $customer->name;
				$customer_status  = $customer->status;
				$customer_user_id = $customer->user_id;
			}
		}

		return array(
			'EDDPRODUCTS'                 => implode( ', ', $product_names ),
			'EDDPRODUCTS_ID'              => implode( ', ', $product_ids ),
			'EDDPRODUCTS_URL'             => implode( ', ', $product_urls ),
			'EDDPRODUCTS_THUMB_ID'        => implode( ', ', $product_thumb_ids ),
			'EDDPRODUCTS_THUMB_URL'       => implode( ', ', $product_thumb_urls ),
			'EDDPRODUCTS_DISCOUNT_CODES'  => $payment->user_info['discount'],
			'EDDPRODUCTS_LICENSE_KEY'     => $license_key,
			'EDDPRODUCTS_ORDER_DISCOUNTS' => number_format( $total_discount, 2 ),
			'EDDPRODUCTS_ORDER_SUBTOTAL'  => number_format( $payment->subtotal, 2 ),
			'EDDPRODUCTS_ORDER_TAX'       => number_format( $payment->tax, 2 ),
			'EDDPRODUCTS_ORDER_TOTAL'     => number_format( $payment->total, 2 ),
			'EDDPRODUCTS_PAYMENT_METHOD'  => $payment->gateway,
			'EDDDISCOUNTCODE'             => $payment->user_info['discount'],
			'EDD_CUSTOMER_ID'             => $customer_id,
			'EDD_CUSTOMER_EMAIL'          => $customer_email,
			'EDD_CUSTOMER_NAME'           => $customer_name,
			'EDD_CUSTOMER_STATUS'         => $customer_status,
			'EDD_CUSTOMER_USER_ID'        => $customer_user_id,
		);
	}
}
