<?php
/**
 * Creates the settings page
 */

namespace Uncanny_Automator_Pro\Integrations\Asana;

use Uncanny_Automator_Pro\Settings\App_Integration_Pro_Settings;
use Uncanny_Automator\Settings\Premium_Integration_Webhook_Settings;

use Exception;

/**
 * Asana_Pro_Settings
 *
 * Pro settings for Asana integration
 * Hooks into the existing free Asana settings page to add Pro functionality
 *
 * @package Uncanny_Automator_Pro\Integrations\Asana\Settings
 *
 * @property Asana_Pro_Api_Caller $api
 * @property Asana_Pro_App_Helpers $helpers
 * @property Asana_Pro_Webhooks $webhooks
 *
 */
class Asana_Pro_Settings extends App_Integration_Pro_Settings {

	use Premium_Integration_Webhook_Settings;

	/**
	 * Output webhook manager after connected panel content
	 *
	 * @return void
	 */
	public function output_after_connected_panel_content() {
		$this->output_webhook_manager();
	}

	/**
	 * Perform cleanup before disconnection
	 * Delete all registered webhooks and register Asana-specific options for automatic cleanup
	 *
	 * @param array $response The current response array
	 * @param array $data The posted data
	 * @param object $base_settings_object The base settings object
	 *
	 * @return array Modified response array
	 */
	public function before_disconnect_pro_cleanup( $response, $data, $base_settings_object ) {
		try {
			// Get webhook manager config to find all registered webhooks
			$config = $this->webhooks->get_webhook_manager_config();

			if ( ! empty( $config ) ) {
				// Loop through and delete all registered webhooks
				foreach ( $config as $project ) {
					if ( ! empty( $project['hook_id'] ) ) {
						try {
							$this->api->delete_webhook( $project['id'] );
						} catch ( Exception $e ) {
							// Log error but continue with other webhooks
							automator_log(
								'Failed to delete webhook for project ' . $project['name'] . ': ' . $e->getMessage(),
								'Asana Pro Webhook Cleanup',
								AUTOMATOR_DEBUG_MODE,
								'asana'
							);
						}
					}
				}
			}
		} catch ( Exception $e ) {
			// Log error but don't fail the disconnection
			automator_log(
				'Error during disconnect cleanup: ' . $e->getMessage(),
				'Asana Pro Disconnect Cleanup',
				AUTOMATOR_DEBUG_MODE,
				'asana'
			);
		}

		// Register manager config option.
		$base_settings_object->register_option( $this->webhooks->get_const( 'MANAGER_OPTION' ) );

		return $response;
	}

	/**
	 * Handle the create webhook request.
	 *
	 * @param array $response - The response array.
	 * @param array $data     - The data passed from the REST request.
	 *
	 * @return array
	 */
	protected function handle_create_webhook( $response, $data ) {
		$project_id = $this->maybe_get_posted_row_id( $data );
		$events     = $this->get_webhook_data_events( $data );

		try {
			$project = $this->api->create_webhook( $project_id, $events );
		} catch ( Exception $e ) {
			$response['alert'] = $this->get_error_alert(
				$e->getMessage(),
			);
			return $response;
		}

		// Update the response with handshake information.
		$response['alert'] = $this->get_success_alert(
			sprintf(
				// translators: %s is the project name
				esc_html_x( 'Webhook created successfully for %s.', 'Asana', 'uncanny-automator-pro' ),
				esc_html( $project['name'] )
			),
		);

		return $this->get_updated_manager_data( $response );
	}

	/**
	 * Handle the edit webhook request.
	 *
	 * @param array $response - The response array.
	 * @param array $data     - The data passed from the REST request.
	 *
	 * @return array
	 */
	protected function handle_edit_webhook( $response, $data ) {
		$project_id = $this->maybe_get_posted_row_id( $data );
		$events     = $this->get_webhook_data_events( $data );

		try {
			$project = $this->api->update_webhook( $project_id, $events );
		} catch ( Exception $e ) {
			$response['alert'] = $this->get_error_alert(
				$e->getMessage(),
			);
			return $response;
		}

		// Update the response.
		$response['alert'] = $this->get_success_alert(
			sprintf(
				// translators: %s is the project name
				esc_html_x( 'Webhook updated successfully for %s.', 'Asana', 'uncanny-automator-pro' ),
				esc_html( $project['name'] )
			),
		);

		return $this->get_updated_manager_data( $response );
	}

	/**
	 * Handle the delete webhook request.
	 *
	 * @param array $response - The response array.
	 * @param array $data     - The data passed from the REST request.
	 *
	 * @return array
	 */
	protected function handle_delete_webhook( $response, $data ) {
		$project_id = $this->maybe_get_posted_row_id( $data );

		// Delete the webhook.
		try {
			$project = $this->api->delete_webhook( $project_id );
		} catch ( Exception $e ) {
			$response['alert'] = $this->get_error_alert(
				$e->getMessage(),
			);
			return $response;
		}

		// Update the response.
		$response['alert'] = $this->get_success_alert(
			sprintf(
				// translators: %s is the project name
				esc_html_x( 'Webhook deleted successfully for %s.', 'Asana', 'uncanny-automator-pro' ),
				esc_html( $project['name'] )
			),
		);

		// Update the response.
		return $this->get_updated_manager_data( $response );
	}

	/**
	 * Get the events from the data.
	 *
	 * @param array $data The data passed from the REST request.
	 * @return array The events.
	 */
	private function get_webhook_data_events( $data ) {
		$all_events = $data['all-events'] ?? false;
		$events     = $all_events
			? $this->get_event_data()['values']
			: array_keys( array_filter( $this->get_array_data( 'events', $data ) ) );

		return $events;
	}

	/**
	 * Get the updated manager data.
	 *
	 * @param array $response The response array.
	 *
	 * @return array
	 */
	private function get_updated_manager_data( $response ) {
		$response['data'] = $this->get_webhook_manager_rows(
			$this->webhooks->get_webhook_manager_config()
		);
		return $response;
	}

	////////////////////////////////////////////////////
	// Integration specific template methods
	////////////////////////////////////////////////////

	/**
	 * Output the webhook manager.
	 *
	 * @return void
	 */
	public function output_webhook_manager() {

		// Title and description.
		$this->output_panel_subtitle( esc_html_x( 'Asana Webhooks', 'Asana', 'uncanny-automator' ) );
		$this->output_subtle_panel_paragraph( esc_html_x( "To use Asana triggers in your recipes, you'll need to authorize webhooks for each project with the events you want to listen for.", 'Asana', 'uncanny-automator-pro' ) );
		$this->output_subtle_panel_paragraph( esc_html_x( "You don't need to authorize projects to use actions.", 'Asana', 'uncanny-automator-pro' ) );

		// Get our webhook manager config.
		$config = $this->webhooks->get_webhook_manager_config();
		if ( empty( $config ) ) {
			$this->alert_html(
				array(
					'type'    => 'warning',
					'heading' => esc_html_x( 'No projects found.', 'Asana', 'uncanny-automator-pro' ),
					'class'   => 'uap-spacing-bottom',
				)
			);
			return;
		}

		$this->output_settings_table(
			$this->get_webhook_manager_columns(),
			$this->get_webhook_manager_rows( $config ),
		);
	}

	/**
	 * Get the linked accounts columns for component.
	 *
	 * @return array
	 */
	private function get_webhook_manager_columns() {
		return array(
			array(
				'key'    => 'project',
				'header' => esc_html_x( 'Project', 'Asana', 'uncanny-automator-pro' ),
			),
			array(
				'key'    => 'status',
				'header' => esc_html_x( 'Status', 'Asana', 'uncanny-automator-pro' ),
			),
			array(
				'key'    => 'action',
				'header' => esc_html_x( 'Action', 'Asana', 'uncanny-automator-pro' ),
			),
		);
	}

	/**
	 * Get the webhook manager rows for component.
	 *
	 * @param array $config
	 *
	 * @return array
	 */
	private function get_webhook_manager_rows( $config ) {
		$rows       = array();
		$workspaces = array();

		foreach ( $config as $project ) {
			$show_workspace = ! in_array( $project['workspace_id'], $workspaces, true );
			$rows[]         = $this->get_webhook_manager_row( $project, $show_workspace );
			if ( $show_workspace ) {
				$workspaces[] = $project['workspace_id'];
			}
		}

		return $rows;
	}

	/**
	 * Get the webhook manager formatted row for component.
	 *
	 * @param array $project
	 * @param bool $show_workspace Whether to show the workspace name
	 *
	 * @return array
	 */
	private function get_webhook_manager_row( $project, $show_workspace = true ) {

		$is_connected = ! empty( $project['hook_id'] );

		// For confirmation messages, use the project name
		$project_name = $project['name'];

		// Base button configuration
		$base_button = array(
			'name'               => 'automator_action',
			'row-submission'     => true,
			'size'               => 'small',
			'type'               => 'submit',
			'needs-confirmation' => true,
			'integration-id'     => $this->get_pro_id(),
		);

		// Build actions based on connection status
		$actions = array();

		if ( $is_connected ) {
			// Edit button
			$actions[] = array(
				'type' => 'button',
				'data' => $base_button + array(
					'value'                     => 'edit_webhook',
					'label'                     => esc_html_x( 'Edit', 'Asana', 'uncanny-automator-pro' ),
					'icon'                      => array(
						'id'   => 'pencil',
						'size' => 'xsmall',
					),
					'confirmation-heading'      => esc_html_x( 'Edit webhook events', 'Asana', 'uncanny-automator-pro' ),
					'confirmation-content'      => sprintf(
						// translators: %s is the project name
						esc_html_x( 'Select the events from the %s project you want to listen for:', 'Asana', 'uncanny-automator-pro' ),
						esc_html( $project_name )
					),
					'confirmation-fields'       => wp_json_encode( $this->get_event_switches( $project['events'] ) ),
					'confirmation-button-label' => esc_html_x( 'Edit webhook', 'Asana', 'uncanny-automator-pro' ),
				),
			);

			// Remove button
			$actions[] = array(
				'type' => 'button',
				'data' => $base_button + array(
					'value'                     => 'delete_webhook',
					'label'                     => esc_html_x( 'Remove', 'Asana', 'uncanny-automator-pro' ),
					'icon'                      => array(
						'id'   => 'trash',
						'size' => 'xsmall',
					),
					'color'                     => 'danger',
					'confirmation-heading'      => esc_html_x( 'Remove Webhook', 'Asana', 'uncanny-automator-pro' ),
					'confirmation-content'      => sprintf(
						// translators: %s is the project name
						esc_html_x( 'Are you sure you want to remove the webhook for the %s project?', 'Asana', 'uncanny-automator-pro' ),
						esc_html( $project_name )
					),
					'confirmation-button-label' => esc_html_x( 'Remove webhook', 'Asana', 'uncanny-automator-pro' ),
				),
			);
		} else {
			// Connect button
			$actions[] = array(
				'type' => 'button',
				'data' => $base_button + array(
					'value'                     => 'create_webhook',
					'label'                     => esc_html_x( 'Connect', 'Asana', 'uncanny-automator-pro' ),
					'icon'                      => array(
						'id'   => 'bolt',
						'size' => 'xsmall',
					),
					'confirmation-heading'      => esc_html_x( 'Connect Webhook', 'Asana', 'uncanny-automator-pro' ),
					'confirmation-content'      => sprintf(
						// translators: %s is the project name
						esc_html_x( 'Select the events from the %s project you want to listen for:', 'Asana', 'uncanny-automator-pro' ),
						esc_html( $project_name )
					),
					'confirmation-fields'       => wp_json_encode( $this->get_event_switches() ),
					'confirmation-button-label' => esc_html_x( 'Connect webhook', 'Asana', 'uncanny-automator-pro' ),
				),
			);
		}

		// Build project column based on whether this is the first row for the workspace
		$project_text = array();
		if ( $show_workspace ) {
			$project_text[] = array(
				'type' => 'text',
				'data' => '**' . $project['workspace_name'] . ' :**',
			);
			$project_text[] = array(
				'type' => 'text',
				'data' => $project['name'],
			);
		} else {
			$project_text[] = array(
				'type' => 'text',
				'data' => $project['name'],
			);
		}

		return array(
			'id'      => $project['id'],
			'columns' => array(
				'project' => array(
					'options' => $project_text,
				),
				'status'  => array(
					'options' => array(
						array(
							'type' => 'text',
							'data' => ! empty( $project['hook_id'] )
								? esc_html_x( 'Connected', 'Asana', 'uncanny-automator-pro' )
								: esc_html_x( 'Not connected', 'Asana', 'uncanny-automator-pro' ),
						),
					),
				),
				'action'  => array(
					'options' => $actions,
					'layout'  => 'horizontal',
				),
			),
		);
	}

	/**
	 * Get the event switches.
	 *
	 * @param array|null $current_events Current events for the webhook (when editing), null for new connections
	 * @return array
	 */
	private function get_event_switches( $current_events = null ) {
		$event_data       = $this->get_event_data();
		$event_options    = $event_data['options'];
		$all_event_values = $event_data['values'];

		// For new connections (null), default to all events enabled
		// For existing connections, check if all events are currently selected
		$current_events = $current_events ?? array();
		$has_all_events = empty( $current_events ) || empty( array_diff( $all_event_values, $current_events ) );

		// All events switch
		$all_events_switch = array(
			'type'               => 'switch',
			'name'               => 'all-events',
			'checked'            => $has_all_events,
			'label-on'           => esc_html_x( 'All events', 'Asana', 'uncanny-automator-pro' ),
			'label-off'          => esc_html_x( 'All events', 'Asana', 'uncanny-automator-pro' ),
			'label-placement'    => 'right',
			'data-state-control' => 'all-events',
		);

		// Individual event switches
		$individual_switches = array();
		foreach ( $event_options as $event ) {
			$individual_switches[] = array(
				'type'            => 'switch',
				'name'            => 'events[' . $event['value'] . ']',
				'checked'         => in_array( $event['value'], $current_events, true ),
				'value'           => $event['value'],
				'label-on'        => $event['text'],
				'label-off'       => $event['text'],
				'label-placement' => 'right',
			);
		}

		// Section for individual switches, only visible when all_events is not checked
		$section = array(
			'type'       => 'section',
			'id'         => 'asana-events-section',
			'state'      => 'all-events',
			'show-when'  => '0', // Show when all_events is unchecked (value '0')
			'components' => $individual_switches,
			'class'      => 'uap-bordered-section',
		);

		return array_merge( array( $all_events_switch, $section ) );
	}

	/**
	 * Get event data.
	 *
	 * @return array Array with full 'options' and 'values' keys
	 */
	private function get_event_data() {
		static $event_data = null;

		if ( null === $event_data ) {
			$event_options = $this->helpers->get_event_options();
			// Filter out any "all events" option from individual switches
			$filtered_options = array_filter(
				$event_options,
				function ( $option ) {
					return 'all' !== $option['value'];
				}
			);

			$event_data = array(
				'options' => $filtered_options,
				'values'  => wp_list_pluck( $filtered_options, 'value' ),
			);
		}

		return $event_data;
	}
}
