<?php

namespace Uncanny_Automator_Pro\Integrations\Asana;

/**
 * Class Asana_Pro_App_Helpers
 *
 * Extends the free version of the Asana app helpers to include:
 * - Pro webhook events.
 * - Pro webhook project options.
 * - Pro webhook project option config.
 *
 * @property Asana_Pro_Api_Caller $api
 * @property Asana_Pro_Webhooks $webhooks
 */
class Asana_Pro_App_Helpers extends \Uncanny_Automator\Integrations\Asana\Asana_App_Helpers {

	////////////////////////////////////////////////////////////
	// Recipe UI helper methods
	////////////////////////////////////////////////////////////

	/**
	 * Get workspace option config for webhooks.
	 *
	 * @param string $option_code
	 *
	 * @return array
	 */
	public function get_webhook_workspace_option_config( $option_code ) {
		return array(
			'input_type'      => 'select',
			'option_code'     => $option_code,
			'label'           => esc_html_x( 'Workspace', 'Asana', 'uncanny-automator' ),
			'placeholder'     => esc_html_x( 'Select a workspace', 'Asana', 'uncanny-automator' ),
			'required'        => true,
			'options'         => array(),
			'options_show_id' => false,
			'relevant_tokens' => array(),
			'description'     => sprintf(
				// translators: %s is common text with a link to the Asana settings page.
				esc_html_x( 'Only workspaces with at least one connected project webhook will be shown. %s', 'Asana', 'uncanny-automator-pro' ),
				$this->get_formatted_manage_webhooks_link()
			),
			'ajax'            => array(
				'endpoint' => 'automator_asana_get_webhook_workspace_options',
				'event'    => 'on_load',
			),
		);
	}

	/**
	 * Get webhook workspace options AJAX.
	 *
	 * @return void
	 */
	public function get_webhook_workspace_options_ajax() {
		Automator()->utilities->verify_nonce();

		// Get all workspaces that have projects with connected webhooks.
		$webhook_config = $this->webhooks->get_webhook_manager_config();
		$options        = array();

		foreach ( $webhook_config as $project ) {
			if ( empty( $project['events'] ) || empty( $project['hook_id'] ) ) {
				continue;
			}

			if ( isset( $options[ $project['workspace_id'] ] ) ) {
				continue;
			}

			// Add the workspace to the options.
			$options[ $project['workspace_id'] ] = array(
				'text'  => $project['workspace_name'],
				'value' => $project['workspace_id'],
			);
		}

		wp_send_json(
			array(
				'success' => true,
				'options' => array_values( $options ),
			)
		);
	}

	/**
	 * Get project option config for webhooks.
	 *
	 * @param string $option_code
	 * @param string $parent_code
	 * @param string $event
	 *
	 * @return array
	 */
	public function get_webhook_project_option_config( $option_code, $parent_code, $event ) {

		$description = 'all' !== $event
			? sprintf(
				// translators: %1$s is the event name, %2$s is common text with a link to the Asana settings page.
				esc_html_x( 'Only projects with a connected webhook for %1$s will be shown. %2$s', 'Asana', 'uncanny-automator-pro' ),
				$event,
				$this->get_formatted_manage_webhooks_link()
			)
			: sprintf(
				// translators: %s is common text with a link to the Asana settings page.
				esc_html_x( 'Only projects with a connected webhook will be shown. %s', 'Asana', 'uncanny-automator-pro' ),
				$this->get_formatted_manage_webhooks_link()
			);

		return array(
			'input_type'      => 'select',
			'option_code'     => $option_code,
			'label'           => esc_html_x( 'Project', 'Asana', 'uncanny-automator-pro' ),
			'required'        => true,
			'options'         => array(),
			'options_show_id' => false,
			'relevant_tokens' => array(),
			'description'     => $description,
			'ajax'            => array(
				'endpoint'      => 'automator_asana_get_webhook_project_options',
				'event'         => 'parent_fields_change',
				'listen_fields' => array( $parent_code ),
			),
		);
	}

	/**
	 * Get webhook project options AJAX.
	 *
	 * @return void
	 */
	public function get_webhook_project_options_ajax() {
		Automator()->utilities->verify_nonce();

		$options   = array();
		$event     = $this->get_event_filter_from_ajax();
		$workspace = $this->get_workspace_from_ajax();
		$options   = $this->get_webhook_project_options( $workspace, $event );

		wp_send_json(
			array(
				'success' => true,
				'options' => $options,
			)
		);
	}

	/**
	 * Get task option config for webhooks.
	 *
	 * @param string $option_code
	 * @param string $parent_code
	 * @param string $event
	 *
	 * @return array
	 */
	public function get_webhook_task_option_config( $option_code, $parent_code ) {
		// Get default from base version.
		$config = $this->get_task_option_config( $option_code );

		// Override the AJAX endpoint.
		$config['ajax']['endpoint']      = 'automator_asana_get_webhook_task_options';
		$config['ajax']['listen_fields'] = array( $parent_code );
		return $config;
	}

	/**
	 * Get webhook task options AJAX.
	 *
	 * @return void
	 */
	public function get_webhook_task_options_ajax() {
		Automator()->utilities->verify_nonce();
		$project_id = $this->get_project_from_ajax();
		$tasks      = $this->api->get_project_tasks( $project_id, $this->is_ajax_refresh() );

		// Prepend "Any task" option.
		$options = array(
			array(
				'text'  => esc_html_x( 'Any task', 'Asana', 'uncanny-automator-pro' ),
				'value' => '-1',
			),
		);

		wp_send_json(
			array(
				'success' => true,
				'options' => array_merge( $options, $tasks ),
			)
		);
	}

	/**
	 * Get event options.
	 *
	 * @return array
	 */
	public function get_event_options() {
		return array(
			array(
				'text'  => esc_html_x( 'All events', 'Asana', 'uncanny-automator-pro' ),
				'value' => 'all',
			),
			array(
				'text'  => esc_html_x( 'Task added', 'Asana', 'uncanny-automator-pro' ),
				'value' => 'task.added',
			),
			array(
				'text'  => esc_html_x( 'Task changed', 'Asana', 'uncanny-automator-pro' ),
				'value' => 'task.changed',
			),
			array(
				'text'  => esc_html_x( 'Comment added', 'Asana', 'uncanny-automator-pro' ),
				'value' => 'story.added',
			),
			array(
				'text'  => esc_html_x( 'Approval status changed', 'Asana', 'uncanny-automator-pro' ),
				'value' => 'task.status_changed',
			),
		);
	}

	/**
	 * Get a formatted link to the Asana settings page.
	 *
	 * @return string
	 */
	public function get_formatted_manage_webhooks_link() {
		return sprintf(
			// translators: %s is a link to the Asana settings page.
			esc_html_x( 'Manage webhooks in your %s.', 'Asana', 'uncanny-automator-pro' ),
			sprintf(
				// translators: %1$s is a link to the Asana settings page, %2$s is the text "Asana settings".
				'<a href="%1$s" target="_blank">%2$s</a>',
				esc_url( $this->get_settings_page_url() ),
				esc_html_x( 'Asana settings', 'Asana', 'uncanny-automator-pro' )
			)
		);
	}

	/**
	 * Get the event filter from the AJAX request.
	 *
	 * @return string
	 */
	private function get_event_filter_from_ajax() {
		$option_code = automator_filter_has_var( 'group_id', INPUT_POST )
			? automator_filter_input( 'group_id', INPUT_POST )
			: false;

		// Map the option code to the event.
		$triggers = array(
			'COMMENT_ADDED_TO_TASK_META'   => 'story.added',
			'TASK_CREATED_IN_PROJECT_META' => 'task.added',
			'TASK_UPDATED_IN_PROJECT_META' => 'task.changed',
			'TASK_STATUS_CHANGED_META'     => 'task.status_changed',
		);

		return isset( $triggers[ $option_code ] )
			? $triggers[ $option_code ]
			: 'all';
	}

	/**
	 * Get project options for webhooks.
	 *
	 * @param string $workspace_id
	 * @param string $event
	 *
	 * @return array
	 */
	public function get_webhook_project_options( $workspace_id, $event = 'all' ) {
		$webhook_config = $this->webhooks->get_webhook_manager_config();
		$options        = array();
		foreach ( $webhook_config as $project_id => $project ) {

			// Ensure the project is in the workspace.
			if ( $workspace_id !== $project['workspace_id'] ) {
				continue;
			}

			// Ensure the webhook has been connected.
			if ( empty( $project['events'] ) || empty( $project['hook_id'] ) ) {
				continue;
			}

			// Ensure the event is in the webhook's events.
			if ( 'all' !== $event && ! in_array( $event, $project['events'], true ) ) {
				continue;
			}

			$options[] = array(
				'text'  => $project['name'],
				'value' => $project_id,
			);
		}

		return $options;
	}

	/**
	 * Get webhook project custom fields AJAX.
	 *
	 * @return void
	 */
	public function get_webhook_project_custom_fields_ajax() {
		Automator()->utilities->verify_nonce();

		$project_id    = $this->get_project_from_ajax();
		$custom_fields = $this->api->get_project_custom_fields( $project_id, $this->is_ajax_refresh() );

		wp_send_json(
			array(
				'success' => true,
				'options' => $custom_fields,
			)
		);
	}

	////////////////////////////////////////////////////////////
	// Trigger custom field helper methods.
	////////////////////////////////////////////////////////////

	/**
	 * Extract custom field GIDs from webhook data.
	 *
	 * @param array $custom_fields Array of custom fields from webhook
	 * @return array Array of custom field GIDs
	 */
	public function get_custom_field_gids_from_webhook( $custom_fields ) {
		if ( empty( $custom_fields ) ) {
			return array();
		}

		return wp_list_pluck( $custom_fields, 'gid' );
	}

	/**
	 * Extract custom field value from task custom fields array.
	 *
	 * @param array  $custom_fields Array of custom fields from task details
	 * @param string $field_gid     GID of the custom field to extract
	 * @return string Custom field value or empty string
	 */
	public function get_custom_field_value_from_task( $custom_fields, $field_gid ) {
		if ( empty( $custom_fields ) || empty( $field_gid ) ) {
			return '';
		}

		foreach ( $custom_fields as $field ) {
			if ( $field['gid'] === $field_gid ) {
				// Handle enum fields (dropdown/multi-select)
				if ( ! empty( $field['enum_value']['name'] ) ) {
					return $field['enum_value']['name'];
				}
				// Handle text fields
				if ( ! empty( $field['text_value'] ) ) {
					return $field['text_value'];
				}
				// Handle number fields
				if ( ! empty( $field['number_value'] ) ) {
					return $field['number_value'];
				}
				// Field found but no value
				return '';
			}
		}

		return '';
	}

	/**
	 * Format custom field GIDs for token display.
	 *
	 * @param array $custom_fields Array of custom fields from webhook
	 * @return string Comma-separated list of GIDs
	 */
	public function format_custom_field_gids_for_tokens( $custom_fields ) {
		if ( empty( $custom_fields ) ) {
			return '';
		}

		$gids = $this->get_custom_field_gids_from_webhook( $custom_fields );
		return implode( ', ', $gids );
	}

	/**
	 * Override standard task fields for Pro triggers with limited set.
	 *
	 * @return array Array of standard field definitions with value and text keys
	 */
	public function get_standard_task_fields() {
		return array(
			array(
				'value' => 'name',
				'text'  => esc_html_x( 'Task name', 'Asana', 'uncanny-automator-pro' ),
			),
			array(
				'value' => 'notes',
				'text'  => esc_html_x( 'Task description', 'Asana', 'uncanny-automator-pro' ),
			),
			array(
				'value' => 'due_on',
				'text'  => esc_html_x( 'Due date', 'Asana', 'uncanny-automator-pro' ),
			),
		);
	}
}
