<?php

namespace Uncanny_Automator_Pro\Integrations\Asana;

use Uncanny_Automator\Integrations\Asana\Asana_Api_Caller;
use Exception;

/**
 * Class Asana_Pro_Api_Caller
 *
 * Extends the free version of the Asana API caller to include:
 * - webhook creation and updates.
 *
 * @package Uncanny_Automator
 *
 * @property Asana_Pro_Webhooks $webhooks
 */
class Asana_Pro_Api_Caller extends Asana_Api_Caller {

	/**
	 * Create a webhook.
	 *
	 * @param int $project_id The ID of the project.
	 * @param array $events The events to subscribe to.
	 *
	 * @return array
	 * @throws Exception
	 */
	public function create_webhook( $project_id, $events ) {

		if ( empty( $events ) ) {
			throw new Exception( esc_html_x( 'Please select at least one event to subscribe your webhook to.', 'Asana', 'uncanny-automator-pro' ) );
		}

		$project = $this->get_project_config( $project_id );

		// Check if Asana webhooks are enabled.
		$webhooks_enabled = $this->webhooks->get_webhooks_enabled_status();

		// If webhooks are not enabled, enable them before creating the first webhook.
		if ( ! $webhooks_enabled ) {
			$this->webhooks->store_webhooks_enabled_status( true );
		}

		// Build request args for Asana webhook creation.
		$args = array(
			'action'   => 'create_webhook',
			'resource' => $project['id'],
			'target'   => $project['url'],
			'filters'  => wp_json_encode( $this->convert_events_to_filters( $events ) ),
		);

		// Make the API request.
		$response = $this->api_request( $args );

		if ( 201 !== $response['statusCode'] ) {

			// If endpoints were not previously enabled, disable again.
			if ( ! $webhooks_enabled ) {
				$this->webhooks->store_webhooks_enabled_status( false );
			}

			throw new Exception( esc_html( $response['data']['message'] ) );
		}

		// Update the project config with webhook data.
		$project['hook_id']      = $response['data']['data']['gid'];
		$project['events']       = $events;
		$project['connected_at'] = time();
		$project['secret']       = $response['data']['X-Hook-Secret'] ?? null;
		$this->webhooks->update_project_config( $project_id, $project );

		return $project;
	}

	/**
	 * Update a webhook.
	 *
	 * @param int $project_id The ID of the project.
	 * @param array $events The events to subscribe to.
	 *
	 * @return array
	 * @throws Exception
	 */
	public function update_webhook( $project_id, $events ) {

		if ( empty( $events ) ) {
			throw new Exception( esc_html_x( 'Please select at least one event to subscribe your webhook to.', 'Asana', 'uncanny-automator-pro' ) );
		}

		$project = $this->get_project_config( $project_id );

		// Compare the events to the existing events
		$existing_events = $project['events'];
		$new_events      = $events;

		sort( $new_events );
		sort( $existing_events );

		if ( $new_events === $existing_events ) {
			throw new Exception( esc_html_x( 'No changes to the webhook events.', 'Asana', 'uncanny-automator-pro' ) );
		}

		// Build request args for Asana webhook update.
		$args = array(
			'action'     => 'update_webhook',
			'webhook_id' => $project['hook_id'],
			'filters'    => wp_json_encode( $this->convert_events_to_filters( $events ) ),
		);

		// Make the API request.
		$response = $this->api_request( $args );
		if ( 200 !== $response['statusCode'] ) {
			throw new Exception( esc_html( $response['data']['message'] ) );
		}

		// Update the project config.
		$project['events'] = $events;
		$this->webhooks->update_project_config( $project_id, $project );

		return $project;
	}

	/**
	 * Delete a webhook.
	 *
	 * @param int $project_id The ID of the project.
	 *
	 * @return array
	 * @throws Exception
	 */
	public function delete_webhook( $project_id ) {
		$project = $this->get_project_config( $project_id );

		// Build request args for Asana webhook deletion.
		$args = array(
			'action'     => 'delete_webhook',
			'webhook_id' => $project['hook_id'],
		);

		// Make the API request.
		$response = $this->api_request( $args );

		if ( 200 !== $response['statusCode'] ) {
			// Check if we got a message else generate message with status code.
			$message = $response['data']['message'] ?? esc_html_x( 'Unknown error :', 'Asana', 'uncanny-automator-pro' ) . ' ' . $response['statusCode'];
			throw new Exception( esc_html( $message ) );
		}

		// Update the project config.
		$project['hook_id']      = null;
		$project['connected_at'] = null;
		$project['events']       = array();
		$project['secret']       = null;
		$this->webhooks->update_project_config( $project_id, $project );

		return $project;
	}

	/**
	 * Get task.
	 *
	 * @param string $task_id The task ID.
	 *
	 * @return array
	 */
	public function get_task( $task_id ) {
		if ( empty( $task_id ) ) {
			return array();
		}

		try {
			$args = array(
				'action'  => 'get_task',
				'task_id' => $task_id,
			);

			$response = $this->api_request( $args );

			if ( 200 !== $response['statusCode'] ) {
				return array();
			}

			return $response['data']['data'] ?? array();
		} catch ( Exception $e ) {
			return array();
		}
	}

	/**
	 * Convert events to Asana webhook filters.
	 *
	 * @param array $events The events array.
	 *
	 * @return array
	 */
	private function convert_events_to_filters( $events ) {
		$filters = array();

		foreach ( $events as $event ) {
			switch ( $event ) {
				case 'task.added':
					$filters[] = array(
						'resource_type' => 'task',
						'action'        => 'added',
					);
					break;
				case 'task.changed':
					$filters[] = array(
						'resource_type' => 'task',
						'action'        => 'changed',
						'fields'        => array( 'name', 'notes', 'due_on', 'custom_fields' ),
					);
					break;
				case 'story.added':
					$filters[] = array(
						'resource_type' => 'story',
						'action'        => 'added',
					);
					break;
				case 'task.status_changed':
					$filters[] = array(
						'resource_type'    => 'task',
						'resource_subtype' => 'approval',
						'action'           => 'changed',
					);
					break;
			}
		}

		return $filters;
	}

	/**
	 * Get the project config by project ID.
	 *
	 * @param int $project_id The ID of the project.
	 *
	 * @return array
	 * @throws Exception
	 */
	private function get_project_config( $project_id ) {
		$project = $this->webhooks->get_project_config( $project_id );
		if ( empty( $project ) ) {
			throw new Exception( esc_html_x( 'Project not found.', 'Asana', 'uncanny-automator-pro' ) );
		}

		return $project;
	}
}
