<?php // phpcs:ignore WordPress.Files.FileName.InvalidClassFileName

namespace Uncanny_Automator_Pro;

/**
 * Class Ameliabooking_Pro_Helpers
 *
 * @package Uncanny_Automator_Pro
 */
class Ameliabooking_Pro_Helpers {

	/**
	 * Registers the hooks for the AmeliaBookingStatusUpdated and AmeliaBookingCanceled actions.
	 */
	public function __construct() {
		$this->register_hooks();
	}

	/**
	 * Register the hooks.
	 *
	 * @return void
	 */
	public function register_hooks() {

		// Register the hooks for the AmeliaBookingStatusUpdated and AmeliaBookingCanceled actions.
		add_action( 'AmeliaBookingStatusUpdated', array( $this, 'ameliabooking_status_updated' ), 10, 3 );
		add_action( 'AmeliaBookingCanceled', array( $this, 'ameliabooking_status_cancelled' ), 10, 3 );

		// Register hook for event booking status updates
		add_action( 'amelia_after_event_booking_updated', array( $this, 'amelia_event_booking_updated' ), 10, 2 );
	}

	/**
	 * Action hook for AmeliaBookingStatusUpdated.
	 *
	 * @param array $reservation Reservation data.
	 * @param array $bookings Bookings data.
	 * @param mixed $container Container object.
	 *
	 * @return void
	 */
	public function ameliabooking_status_updated( $reservation, $bookings, $container ) {

		// Delegate the action to the automator_ameliabooking_status_updated action hook.
		foreach ( $bookings as $booking ) {
			// Fires the trigger for each booking.
			do_action( 'automator_ameliabooking_status_updated', $reservation, $booking, $container );
		}
	}

	/**
	 * Action hook for AmeliaBookingCanceled.
	 *
	 * @param array $reservation Reservation data.
	 * @param array $bookings Bookings data.
	 * @param mixed $container Container object.
	 *
	 * @return void
	 */
	public function ameliabooking_status_cancelled( $reservation, $bookings, $container ) {

		// Delegate the action to the automator_ameliabooking_status_cancelled action hook.
		foreach ( $bookings as $booking ) {
			// Fires the trigger for each booking.
			do_action( 'automator_ameliabooking_status_cancelled', $reservation, $booking, $container );
		}
	}

	/**
	 * Action hook for amelia_after_event_booking_updated.
	 *
	 * @param array $booking Booking data.
	 * @param array $booking_data Additional booking data.
	 *
	 * @return void
	 */
	public function amelia_event_booking_updated( $booking, $booking_data ) {

		// Get event data using booking ID
		$event_data = $this->get_event_by_booking_id( $booking['id'] );

		if ( ! $event_data ) {
			return;
		}

		// Fire our custom hook with event data included
		do_action( 'automator_amelia_event_booking_status_updated', $booking, $booking_data, $event_data );
	}

	/**
	 * Get event data by booking ID using Amelia's EventRepository.
	 *
	 * @param int $booking_id Booking ID.
	 *
	 * @return array|false
	 */
	private function get_event_by_booking_id( $booking_id ) {

		// Create Amelia container
		if ( ! class_exists( 'AmeliaBooking\Infrastructure\Common\Container' ) ) {
			return false;
		}

		try {
			$container = new \AmeliaBooking\Infrastructure\Common\Container();

			// Get the event repository
			$event_repository = $container->get( 'domain.booking.event.repository' );

			// Get the event using the booking ID
			$event = $event_repository->getByBookingId(
				$booking_id,
				array(
					'fetchEventsTickets'   => true,
					'fetchEventsTags'      => true,
					'fetchEventsProviders' => true,
					'fetchEventsImages'    => true,
				)
			);

			// Convert to array if event exists
			return $event ? $event->toArray() : false;

		} catch ( \Exception $e ) {
			// Fallback to direct database query if container method fails
			return $this->get_event_by_booking_id_fallback( $booking_id );
		}
	}

	/**
	 * Fallback method to get event data via direct database query.
	 *
	 * @param int $booking_id Booking ID.
	 *
	 * @return array|false
	 */
	private function get_event_by_booking_id_fallback( $booking_id ) {
		global $wpdb;

		// Query the Amelia database tables to get event data
		$table_prefix = $wpdb->prefix . 'amelia_';

		$event_data = $wpdb->get_row(
			$wpdb->prepare(
				"SELECT e.* FROM {$table_prefix}events e
			 INNER JOIN {$table_prefix}events_periods ep ON ep.eventId = e.id
			 INNER JOIN {$table_prefix}customer_bookings_to_events_periods cb ON cb.eventPeriodId = ep.id
			 WHERE cb.customerBookingId = %d",
				$booking_id
			),
			ARRAY_A
		);

		return $event_data ?: false;
	}

	/**
	 * Get the WP User ID from the booking.
	 *
	 * @param array $booking Booking data.
	 * @param mixed $container Container object.
	 *
	 * @return bool|int
	 */
	public static function get_single_booking_wp_user_id( $booking, $container ) {

		// Get the customer id.
		$customer_id = $booking['customerId'] ?? null;

		if ( ! $customer_id ) {
			return false;
		}

		// Get the customer data by customer id.
		$external_id = self::get_external_id( $customer_id, $container );

		// Check if external ID is WP User ID.
		if ( $external_id ) {
			$wp_user = get_user_by( 'ID', $external_id );

			return $wp_user->ID ?? false;
		}

		// Get the user data.
		$user_data  = self::get_customer_data( $customer_id, $container );
		$user_email = $user_data['email'] ?? null;
		$wp_user    = get_user_by( 'email', $user_email );

		return $wp_user->ID ?? false;
	}

	/**
	 * Get the WP User ID from the reservation.
	 *
	 * @param array $reservation Reservation data.
	 * @param mixed $container Container object.
	 *
	 * @return bool|int
	 */
	public static function get_reservation_wp_user_id( $reservation, $container ) {

		$user_id = false;

		// Validate the reservation and container params.
		if ( ! is_array( $reservation ) ) {
			return $user_id;
		}

		// Get the customer id.
		$booking     = ! empty( $reservation['bookings'] ) ? $reservation['bookings'][0] : array();
		$customer_id = ! empty( $booking['customerId'] ) ? $booking['customerId'] : false;

		if ( ! $customer_id ) {
			return $user_id;
		}

		// Get the customer data by customer id.
		$external_id = self::get_external_id( $customer_id, $container );

		// Check if external ID is WP User ID.
		if ( $external_id ) {
			$wp_user = get_user_by( 'ID', $external_id );
			$user_id = ! empty( $wp_user ) ? $wp_user->ID : false;
		}

		return $user_id;
	}

	/**
	 * Get the external ID from the customer ID.
	 *
	 * @param int $customer_id Customer ID.
	 * @param mixed $container Container object.
	 *
	 * @return bool|int
	 */
	public static function get_external_id( $customer_id, $container ) {

		$user_data   = self::get_customer_data( $customer_id, $container );
		$external_id = ! empty( $user_data['externalId'] ) ? $user_data['externalId'] : false;

		return $external_id;
	}

	/**
	 * Get the customer data by customer ID.
	 *
	 * @param int $customer_id Customer ID.
	 * @param mixed $container Container object.
	 *
	 * @return array
	 */
	public static function get_customer_data( $customer_id, $container ) {

		if ( ! is_a( $container, 'AmeliaBooking\Infrastructure\Common\Container' ) ) {
			return array();
		}

		try {
			$user_repo = $container->get( 'domain.users.repository' );
			$user      = $user_repo->getById( (int) $customer_id );
			$user_data = ! empty( $user ) ? $user->toArray() : array();

			return $user_data;
		} catch ( \Exception $e ) {
			// Fallback to direct database query if container method fails
			return self::get_customer_data_fallback( $customer_id );
		}
	}

	/**
	 * Fallback method to get customer data via direct database query.
	 *
	 * @param int $customer_id Customer ID.
	 *
	 * @return array
	 */
	private static function get_customer_data_fallback( $customer_id ) {
		global $wpdb;

		$table_name = $wpdb->prefix . 'amelia_users';

		$customer_data = $wpdb->get_row(
			$wpdb->prepare(
				"SELECT * FROM {$table_name} WHERE id = %d",
				$customer_id
			),
			ARRAY_A
		);

		return $customer_data ?: array();
	}

	/**
	 * @param $is_any
	 * @param $is_all
	 *
	 * @return array[]
	 */
	public function get_event_booking_statuses( $is_any = false, $is_all = false ) {
		$status_label = true === $is_any ? esc_attr_x( 'Any status', 'Ameliabooking', 'uncanny-automator-pro' ) : esc_attr_x( 'All statuses', 'Ameliabooking', 'uncanny-automator-pro' );

		return array(
			array(
				'text'  => $status_label,
				'value' => '-1',
			),
			array(
				'text'  => esc_attr_x( 'Approved', 'Ameliabooking', 'uncanny-automator-pro' ),
				'value' => 'approved',
			),
			array(
				'text'  => esc_attr_x( 'Rejected', 'Ameliabooking', 'uncanny-automator-pro' ),
				'value' => 'rejected',
			),
			array(
				'text'  => esc_attr_x( 'No-show', 'Ameliabooking', 'uncanny-automator-pro' ),
				'value' => 'no-show',
			),
		);
	}
}
