<?php

namespace Uncanny_Automator_Pro;

/**
 * Automator Free Version Dependency
 *
 * Hooks into site_transient_update_plugins and checks if Automator free has an update.
 * If it does, prevents Automator Pro from updating and adds a notice below Automator pro plugin row.
 * 
 * Also provides a public "met" method to check if the required Automator free version is met.
 * 
 * @package Uncanny_Automator_Pro
 */
class Automator_Free_Version_Dependency {

    const pro_file = 'uncanny-automator-pro/uncanny-automator-pro.php';

    const free_file = 'uncanny-automator/uncanny-automator.php';

    /**
     * The required Automator free version.
     *
     * @var string
     */
    protected $required_automator_version;

    /**
     * The name of the plugin.
     *
     * @var string
     */
    protected $name;

    /**
     * The slug of the plugin.
     *
     * @var string
     */
    protected $slug;

    /**
     * Whether to show the update notice.
     *
     * @var bool
     */
    protected $show_update_notice = false;

    /**
     * Class constructor
     */
    public function __construct( $version ) {
        $this->required_automator_version = $version;
        $this->name = plugin_basename( self::pro_file );
        $this->slug = basename( dirname( self::pro_file ) );

        $this->register_hooks();
    }

    /**
     * Register WordPress hooks
     */
    protected function register_hooks() {
        // Adjust the update transient to check if there is an update for Automator free version and disable Pro update if needed.
        add_filter( 'site_transient_update_plugins', array( $this, 'check_update_plugins_transient' ), 999 );

        // Add a notice to the Automator Pro plugin row.
        add_action( 'after_plugin_row', array( $this, 'maybe_show_update_notification' ), 10, 2 );
    }

    /**
     * Checks if Automator free version has an update.
     * If it does, we remove any pending Automator Pro updates
     *
     * @param object $transient The transient object.
     *
     * @return object
     */
    public function check_update_plugins_transient( $transient ) {

        if ( ! is_object( $transient ) ) {
            return $transient;
        }

        // If there is no pending Pro update, do nothing
        if ( ! isset( $transient->response[ self::pro_file ] ) ) {
            return $transient;
        }

        // If there is no pending Free update, allow updating Pro.
        if ( ! isset( $transient->response[ self::free_file ] ) ) {
            return $transient;
        }
        
        // Otherwise, prevent Pro from updating.
        unset( $transient->response[ self::pro_file ] );

        // Add a notice below the Automator Pro plugin row.
        $this->show_update_notice = true;

        return $transient;
    }

    /**
	 * Show the update notification on multisite subsites.
	 *
	 * @param string  $file
	 * @param array   $plugin
	 */
	public function maybe_show_update_notification( $file, $plugin ) {

        if ( ! $this->show_update_notice ) {
            return;
        }

        if ( self::pro_file !== $file ) {
            return;
        }

		// Allow single site admins to see that an update is available.
		if ( ! current_user_can( 'activate_plugins' ) ) {
			return;
		}

		// Calculate the correct colspan based on the standard plugin table columns
		$colspan = 3; // Default: checkbox, name, description
		
		// Check if auto-updates column is shown (WordPress 5.5+)
		if ( wp_is_auto_update_enabled_for_type( 'plugin' ) && current_user_can( 'update_plugins' ) ) {
			$colspan = 4; // Add auto-updates column
		}

		$active_class = in_array( $this->name, $this->get_active_plugins(), true ) ? ' active' : '';

        // Simulate the default plugin update notification looks
		printf(
			'<tr class="plugin-update-tr update%s" id="%s-update" data-slug="%s" data-plugin="%s">' .
			'<td colspan="%d" class="plugin-update colspanchange">' .
			'<div class="update-message notice inline notice-warning notice-alt"><p>',
			$active_class,
			esc_attr( $this->slug ),
			esc_attr( $this->slug ),
			esc_attr( self::pro_file ),
			$colspan
		);

		esc_html_e( 'There is a new version of Uncanny Automator Pro available, but Automator Free must be updated first.', 'uncanny-automator-pro' );

		echo '</p></div></td></tr>';

        // Add the "update" class to the Automator Pro plugin row above to remove the default border between rows.
		?>
		<script>
        document.addEventListener('DOMContentLoaded', function() {
            // Add 'update' class to the plugin row above this notice
            var selector = 'tr[data-plugin="<?php echo esc_js( self::pro_file ); ?>"]';
            var row = document.querySelector(selector);
            if (row) {
                row.classList.add('update');
            }
        });
		</script>
        <?php
	}

    /**
	 * Gets the active plugins.
	 *
	 * @return array
	 */
	protected function get_active_plugins() {
		$active_plugins         = (array) get_option( 'active_plugins' );
		$active_network_plugins = (array) get_site_option( 'active_sitewide_plugins' );

		return array_merge( $active_plugins, array_keys( $active_network_plugins ) );
	}

    /**
     * Check if the Automator Free version meets the requirements.
     *
     * @return bool
     */
    public function met() {

        $version = $this->get_automator_free_version();

        if ( version_compare( $version, $this->required_automator_version, '<' ) ) {
            return false;
        }

        return true;
    }

    /**
     * Get the Automator free version.
     *
     * This method retrieves the version of the Automator free plugin.
     * If the plugin is not installed, it returns '0'.
     *
     * We do not use AUTOMATOR_PLUGIN_VERSION constant here because Automator Free might not be loaded yet.
     *
     * @return string The version of the Automator free plugin.
     */
    public function get_automator_free_version() {

        $plugin_data = $this->get_plugin_data( self::free_file );

        return isset( $plugin_data['Version'] ) ? $plugin_data['Version'] : '0';
    }
    
    /**
     * get_plugin_data
     *
     * @return array
     */
    public function get_plugin_data( $file ) {
        require_once ABSPATH . 'wp-admin/includes/plugin.php';

        return get_plugin_data( WP_PLUGIN_DIR . '/' . $file, false, false );
    }

}