<?php
// phpcs:disable Universal.Operators.DisallowStandalonePostIncrementDecrement.PostIncrementFound

namespace Uncanny_Automator_Pro\Integrations\Db_Query\Helpers;

/**
 * Class Sql_Semicolon_Checker
 *
 * Parses an SQL string and determines if there is any semicolon
 * outside of literals, comments, dollar-quotes, or bracketed identifiers.
 *
 * @package YourPluginName
 */
class Sql_Semicolon_Checker_Helper {

	/**
	 * The SQL query being parsed.
	 *
	 * @var string
	 */
	private $query;

	/**
	 * Length of the SQL query.
	 *
	 * @var int
	 */
	private $query_length;

	/**
	 * Current parse position.
	 *
	 * @var int
	 */
	private $position;

	/**
	 * Checks if the query contains any unquoted semicolon.
	 *
	 * @param string $query SQL query to inspect.
	 * @return bool True if an unquoted semicolon is found, false otherwise.
	 */
	public function contains_unquoted_semicolon( $query ) {
		$this->query        = $query;
		$this->query_length = strlen( $query );
		$this->position     = 0;

		while ( $this->position < $this->query_length ) {
			$char = $this->query[ $this->position ];

			switch ( $char ) {
				case '-':
					if ( '-' === $this->peek( 1 ) ) {
						$this->skip_single_line_comment( 2 );
						continue 2;
					}
					break;

				case '#':
					$this->skip_single_line_comment( 1 );
					continue 2;

				case '/':
					if ( '*' === $this->peek( 1 ) ) {
						$this->skip_multi_line_comment();
						continue 2;
					}
					break;

				case '$':
					if ( $this->skip_dollar_quoted() ) {
						continue 2;
					}
					break;

				case '"':
				case "'":
				case '`':
					$this->skip_quoted_literal( $char );
					continue 2;

				case '[':
					$this->skip_bracket_identifier();
					continue 2;

				case ';':
					return true;
			}

			$this->position++;
		}

		return false;
	}

	/**
	 * Peek ahead in the query.
	 *
	 * @param int $offset Number of chars ahead to peek.
	 * @return string|null
	 */
	private function peek( $offset ) {
		$idx = $this->position + $offset;
		return ( $idx < $this->query_length )
			? $this->query[ $idx ]
			: null;
	}

	/**
	 * Skip a single-line comment starting at current position.
	 *
	 * @param int $start_len Length of the comment start sequence.
	 */
	private function skip_single_line_comment( $start_len ) {
		$this->position += $start_len;
		while ( $this->position < $this->query_length
			&& "\n" !== $this->query[ $this->position ]
		) {
			$this->position++;
		}
	}

	/**
	 * Skip a multi-line comment (/* ... *\/).
	 */
	private function skip_multi_line_comment() {
		$this->position += 2;
		while ( $this->position + 1 < $this->query_length ) {
			if (
				'*' === $this->query[ $this->position ]
				&& '/' === $this->peek( 1 )
			) {
				$this->position += 2;
				return;
			}
			$this->position++;
		}
	}

	/**
	 * Attempt to skip a dollar-quoted string.
	 *
	 * @return bool True if skipped, false otherwise.
	 */
	private function skip_dollar_quoted() {
		if ( preg_match(
			'/\A\$(?:[A-Za-z0-9_]+)?\$/',
			substr( $this->query, $this->position ),
			$m
		) ) {
			$tag             = $m[0];
			$this->position += strlen( $tag );
			$end             = strpos( $this->query, $tag, $this->position );
			if ( false === $end ) {
				$this->position = $this->query_length;
				return true;
			}
			$this->position = $end + strlen( $tag );
			return true;
		}

		return false;
	}

	/**
	 * Skip a quoted literal ('...', "..." or `...`).
	 *
	 * @param string $quote The quote character.
	 */
	private function skip_quoted_literal( $quote ) {
		$this->position++;
		while ( $this->position < $this->query_length ) {
			$c = $this->query[ $this->position ];

			if ( $quote === $c ) {
				if (
					'\\' !== $quote
					&& $this->peek( 1 ) === $quote
				) {
					$this->position += 2;
					continue;
				}
				$this->position++;
				break;
			}

			if (
				'\\' === $c
				&& '`' !== $quote
				&& null !== $this->peek( 1 )
			) {
				$this->position += 2;
				continue;
			}

			$this->position++;
		}
	}

	/**
	 * Skip a SQL Server bracketed identifier [ ... ].
	 */
	private function skip_bracket_identifier() {
		$this->position++;
		while (
			$this->position < $this->query_length
			&& ']' !== $this->query[ $this->position ]
		) {
			$this->position++;
		}
		if ( $this->position < $this->query_length ) {
			$this->position++;
		}
	}
}
