<?php
namespace Uncanny_Automator_Pro;

use Uncanny_Automator\Recipe;

/**
 * Class THRIVE_APPRENTICE_USER_COURSE_PROGRESSED
 *
 * @package Uncanny_Automator
 */
class THRIVE_APPRENTICE_USER_COURSE_PROGRESSED {


	use Recipe\Triggers;

	/**
	 * Constant TRIGGER_CODE.
	 *
	 * @var string
	 */
	const TRIGGER_CODE = 'THRIVE_APPRENTICE_USER_COURSE_PROGRESSED';

	/**
	 * Constant TRIGGER_META.
	 *
	 * @var string
	 */
	const TRIGGER_META = 'THRIVE_APPRENTICE_USER_COURSE_PROGRESSED_META';

	/**
	 * The helper instance.
	 *
	 * @var Thrive_Apprentice_Pro_Helpers
	 */
	protected $helper;

	/**
	 * Constructor.
	 */
	public function __construct() {

		$this->set_helper( new Thrive_Apprentice_Pro_Helpers() );

		$this->setup_trigger();
	}

	/**
	 * Define and register the trigger by pushing it into the Automator object.
	 *
	 * @return void.
	 */
	public function setup_trigger() {

		$this->set_integration( 'THRIVE_APPRENTICE' );

		$this->set_trigger_code( self::TRIGGER_CODE );

		$this->set_trigger_meta( self::TRIGGER_META );

		$this->set_is_pro( true );

		$this->set_is_login_required( true );

		// The action hook to attach this trigger into.
		$this->add_action( 'thrive_apprentice_course_progress' );

		$this->set_sentence(
			sprintf(
				// translators: %1$s is the course title.
				esc_html_x( 'A user progresses in {{a course:%1$s}}', 'Thrive Apprentice', 'uncanny-automator-pro' ),
				$this->get_trigger_meta()
			)
		);

		$this->set_readable_sentence(
		/* Translators: Trigger sentence */
			esc_html_x( 'A user progresses in {{a course}}', 'Thrive Apprentice', 'uncanny-automator-pro' )
		);

		// Loads options.
		$this->set_options_callback( array( $this, 'load_options' ) );

		// Register the trigger.
		$this->register_trigger();
	}

	/**
	 * Loads available options for the Trigger.
	 *
	 * @return array The available trigger options.
	 */
	public function load_options() {

		$relevant_tokens                                = $this->get_helper()->get_relevant_tokens_courses();
		$relevant_tokens['COURSE_PERCENTAGE_COMPLETED'] = esc_html_x( 'Course percentage completed', 'Thrive Apprentice', 'uncanny-automator-pro' );

		return Automator()->utilities->keep_order_of_options(
			array(
				'options_group' => array(
					$this->get_trigger_meta() => array(
						array(
							'option_code'     => $this->get_trigger_meta(),
							'required'        => true,
							'label'           => esc_html_x( 'Course', 'Thrive Apprentice', 'uncanny-automator-pro' ),
							'input_type'      => 'select',
							'options'         => $this->get_helper()->get_dropdown_options_courses( true ),
							'relevant_tokens' => $relevant_tokens,
						),
					),
				),
			)
		);
	}

	/**
	 * Validate the trigger.
	 *
	 * @return boolean True.
	 */
	public function validate_trigger( ...$args ) {

		// No need to check anything here.
		return true;
	}

	/**
	 * Prepare to run.
	 *
	 * Sets the conditional trigger to true.
	 *
	 * @return void.
	 */
	public function prepare_to_run( $data ) {

		$this->set_conditional_trigger( true );
	}

	/**
	 * Validates the conditions.
	 *
	 * @param array $args The trigger args.
	 *
	 * @return array The matching recipe and trigger IDs.
	 */
	public function validate_conditions( ...$args ) {

		list( $term, $user ) = $args[0];

		// Handle both array and object cases for term
		$course_id = 0;
		if ( is_object( $term ) && isset( $term->term_id ) ) {
			$course_id = $term->term_id;
		} elseif ( is_array( $term ) ) {
			// Check for both term_id and course_id
			$course_id = isset( $term['course_id'] ) ? $term['course_id'] : ( isset( $term['term_id'] ) ? $term['term_id'] : 0 );
		}

		$matching_recipes_triggers = $this->find_all( $this->trigger_recipes() )
			->where( array( $this->get_trigger_meta() ) )
			->match( array( absint( $course_id ) ) ) //phpcs:ignore Uncanny_Automator.PHP.ForbiddenPHP8Features.ForbiddenPHP8Feature -- method name, not PHP 8 match expression
			->format( array( 'intval' ) )
			->get();

		return $matching_recipes_triggers;
	}

	/**
	 * Parse additional tokens.
	 *
	 * @param mixed $parsed The parsed.
	 * @param mixed $args The arguments.
	 * @param mixed $trigger The trigger.
	 * @return mixed
	 */
	public function parse_additional_tokens( $parsed, $args, $trigger ) {

		$params  = array_shift( $args['trigger_args'] );
		$user_id = isset( $args['trigger_entry']['user_id'] ) ? $args['trigger_entry']['user_id'] : 0;

		// Get course data using helper method
		$course_response = $this->get_helper()->get_course_by_id( $params['course_id'] );
		if ( is_wp_error( $course_response ) ) {
			return $parsed;
		}

		$course_data = json_decode( wp_remote_retrieve_body( $course_response ), true );

		$tva_author = get_term_meta( $params['course_id'], 'tva_author', true );
		$user_data  = ! empty( $tva_author ) && isset( $tva_author['ID'] ) ? get_userdata( $tva_author['ID'] ) : false;

		// Get course progress using helper method
		$progress = $this->get_helper()->get_course_progress_percentage( $user_id, $params['course_id'] );

		$hydrated_tokens = array(
			'COURSE_ID'                   => $params['course_id'],
			'COURSE_TITLE'                => isset( $course_data['name'] ) ? $course_data['name'] : '',
			'COURSE_URL'                  => isset( $course_data['preview_url'] ) ? $course_data['preview_url'] : '',
			'COURSE_AUTHOR'               => is_object( $user_data ) && ! empty( $user_data ) ? $user_data->user_email : '',
			'COURSE_SUMMARY'              => isset( $course_data['excerpt'] ) ? $course_data['excerpt'] : '',
			'COURSE_PERCENTAGE_COMPLETED' => $progress,
		);

		return $parsed + $hydrated_tokens;
	}
}
