<?php

namespace UncannyAutomator\AddOn\CustomUserFields\Integration\Conditions;

use UncannyAutomator\AddOn\CustomUserFields\Fields\Factory as Fields_Factory;

use Uncanny_Automator_Pro\Action_Condition;

/**
 * Class WP_USER_HAS_META_KEY
 *
 * @package Uncanny_Automator_Pro
 */
class User_Meta_Meets_Condition extends Action_Condition {

	/**
	 * Method define_condition
	 *
	 * @return void
	 */
	public function define_condition() {

		$this->integration  = 'CUSTOMUSERFIELDS';
		$this->name         = __( "A user's {{custom user field}} meets a condition", 'uacuf' );
		$this->code         = 'USER_META_MEETS_CONDITION';
		$this->dynamic_name = sprintf(
		/* translators: 1: the meta key, 2: the value to match */
			esc_html__( "A user's {{custom user field:%1\$s}} {{matches:%2\$s}} {{a meta value:%3\$s}}", 'uacuf' ),
			'METAKEY',
			'CRITERIA',
			'VALUE'
		);
		$this->is_pro        = true;
		$this->requires_user = false;
	}

	/**
	 * Method fields
	 *
	 * @return array
	 */
	public function fields() {

		$criteria = array(
			'option_code'            => 'CRITERIA',
			'input_type'             => 'select',
			'label'                  => esc_html__( 'Criteria', 'uacuf' ),
			'show_label_in_sentence' => false,
			'required'               => true,
			'options_show_id'        => false,
			'options'                => apply_filters(
				'automator_pro_wp_token_meets_condition_criteria_options',
				array(
					array(
						'value' => 'is',
						'text'  => esc_html__( 'is', 'uacuf' ),
					),
					array(
						'value' => 'is_not',
						'text'  => esc_html__( 'is not', 'uacuf' ),
					),
					array(
						'value' => 'contains',
						'text'  => esc_html__( 'contains', 'uacuf' ),
					),
					array(
						'value' => 'does_not_contain',
						'text'  => esc_html__( 'does not contain', 'uacuf' ),
					),
					array(
						'value' => 'is_greater_than',
						'text'  => esc_html__( 'is greater than', 'uacuf' ),
					),
					array(
						'value' => 'is_greater_than_or_equal_to',
						'text'  => esc_html__( 'is greater than or equal to', 'uacuf' ),
					),
					array(
						'value' => 'is_less_than',
						'text'  => esc_html__( 'is less than', 'uacuf' ),
					),
					array(
						'value' => 'is_less_than_or_equal_to',
						'text'  => esc_html__( 'is less than or equal to', 'uacuf' ),
					),
					array(
						'value' => 'starts_with',
						'text'  => esc_html__( 'starts with', 'uacuf' ),
					),
					array(
						'value' => 'does_not_start_with',
						'text'  => esc_html__( 'does not start with', 'uacuf' ),
					),
					array(
						'value' => 'ends_with',
						'text'  => esc_html__( 'ends with', 'uacuf' ),
					),
					array(
						'value' => 'does_not_end_with',
						'text'  => esc_html__( 'does not end with', 'uacuf' ),
					),

				)
			),
			'supports_custom_value'  => false,
		);

		return array(
			$this->field->select(
				array(
					'option_code'     => 'METAKEY',
					'label'           => esc_html__( 'Meta key', 'uacuf' ),
					'required'        => true,
					'options'         => $this->load_options(),
					'input_type'      => 'select',
					'default_value'   => null,
					'options_show_id' => false,
				)
			),
			$criteria,
			$this->field->text(
				array(
					'show_label_in_sentence' => false,
					'option_code'            => 'VALUE',
					'label'                  => esc_html__( 'Value', 'uacuf' ),
					'required'               => true,
				)
			),
		);
	}

	/**
	 * @return array
	 */
	public function load_options() {

		$fields = Fields_Factory::get_all();

		$options = array();

		foreach ( $fields as $field ) {
			$text  = $field->get_name();
			$value = $field->get_meta();

			$options[] = array(
				'text'  => $text,
				'value' => $value,
			);

			$value .= '_uacuf_metakey';
			$text  .= ' - ' . _x( 'Meta key', 'Token dropdown identifier', 'uacuf' );

			$options[] = array(
				'text'  => $text,
				'value' => $value,
			);
		}

		return $options;
	}

	/**
	 * Evaluate_condition
	 *
	 * Has to use the $this->condition_failed( $message ); method if the condition is not met.
	 *
	 * @return void
	 * @throws \Exception
	 */
	public function evaluate_condition() {

		$meta_key          = $this->get_parsed_option( 'METAKEY' );
		$meta_key_readable = $this->get_option( 'METAKEY_readable' );

		// If its uacuf meta key AND contains _uacuf_metakey, return the meta key instead of the meta value
		$substring = '_uacuf_metakey';
		if ( strpos( $meta_key, $substring ) && 0 === substr_compare( $meta_key, $substring, - strlen( $substring ) ) ) {
			$meta_key          = str_replace( '_uacuf_metakey', '', $meta_key );
			$meta_key_readable = $meta_key;
		}

		$value_to_match = $this->get_parsed_option( 'VALUE' );

		$criteria = $this->get_option( 'CRITERIA' );

		$user_meta_value = get_user_meta( $this->user_id, $meta_key, true );

		$result = $this->check_condition( $criteria, $user_meta_value, $value_to_match );

		// If the conditions is not met, send an error message and mark the condition as failed.
		if ( false === $result ) {
			$message = $this->generate_error_message( $value_to_match, $user_meta_value, $meta_key_readable );
			$this->condition_failed( $message );
		}
	}

	/**
	 * @param $criteria
	 * @param $user_meta_value
	 * @param $value_to_match
	 *
	 * @return bool
	 */
	public function check_condition( $criteria, $user_meta_value, $value_to_match ) {
		switch ( $criteria ) {
			case 'is':
				$result = $user_meta_value == $value_to_match; //phpcs:ignore WordPress.PHP.StrictComparisons.LooseComparison
				break;
			case 'is_not':
				$result = $user_meta_value != $value_to_match; //phpcs:ignore WordPress.PHP.StrictComparisons.LooseComparison
				break;
			case 'contains':
				if ( is_array( $user_meta_value ) || is_object( $user_meta_value ) ) {
					$user_meta_value = join( ' ', $user_meta_value );
				}
				$result = stripos( $user_meta_value, $value_to_match ) !== false;
				break;
			case 'does_not_contain':
				if ( is_array( $user_meta_value ) || is_object( $user_meta_value ) ) {
					$user_meta_value = join( ' ', $user_meta_value );
				}
				$result = stripos( $user_meta_value, $value_to_match ) === false;
				break;
			case 'is_greater_than':
				$result = floatval( $user_meta_value ) > floatval( $value_to_match );
				break;
			case 'is_greater_than_or_equal_to':
				$result = floatval( $user_meta_value ) >= floatval( $value_to_match );
				break;
			case 'is_less_than':
				$result = floatval( $user_meta_value ) < floatval( $value_to_match );
				break;
			case 'is_less_than_or_equal_to':
				$result = floatval( $user_meta_value ) <= floatval( $value_to_match );
				break;
			case 'starts_with':
				$result = stripos( $user_meta_value, $value_to_match ) === 0;
				break;
			case 'does_not_start_with':
				$result = stripos( $user_meta_value, $value_to_match ) !== 0;
				break;
			case 'ends_with':
				$result = stripos( strrev( $user_meta_value ), strrev( $value_to_match ) ) === 0;
				break;
			case 'does_not_end_with':
				$result = stripos( strrev( $user_meta_value ), strrev( $value_to_match ) ) !== 0;
				break;
			default:
				$result = true;
				break;
		}

		return $result;
	}

	/**
	 * Generate_error_message
	 *
	 * @param $value_to_match
	 * @param $actual_value
	 * @param string $meta_key_readable
	 *
	 * @return string
	 * @throws \Exception
	 */
	public function generate_error_message( $value_to_match, $actual_value, $meta_key_readable = '' ) {

		$readable_criteria = $this->get_option( 'CRITERIA_readable' );
		$readable_meta     = $meta_key_readable;

		$format = __( "User's meta key: \"%1\$s\" \"%2\$s\" \"%3\$s\". Value returned: \"%4\$s\"", 'uacuf' );

		$condition_sentence = sprintf( $format, $readable_meta, $readable_criteria, $value_to_match, $actual_value );

		return __( 'Failed condition: ', 'uacuf' ) . $condition_sentence;
	}
}
