<?php

namespace UncannyAutomator\AddOn\CustomUserFields\Framework;

use UncannyAutomator\AddOn\CustomUserFields\Fields\Rest_Api;
use UncannyAutomator\AddOn\CustomUserFields\Integration\Automator_Integration as Integration;
use UncannyAutomator\AddOn\CustomUserFields\User\Profile as User_Profile;
use UncannyAutomator\AddOn\CustomUserFields\Shortcodes\User_Meta as User_Meta_Shortcode;
use UncannyAutomator\AddOn\CustomUserFields\Licensing\License;
use UncannyAutomator\AddOn\CustomUserFields\Licensing\Updater;
use UncannyAutomator\AddOn\CustomUserFields\Blocks\Blocks;

/**
 *
 */
class Plugin {

	/**
	 * @var
	 */
	private static $version;
	/**
	 * @var bool
	 */
	private static $requirements_met = false;
	/**
	 * @var
	 */
	public $licenses;
	/**
	 *
	 */
	const OPTION_VERSION = 'uacuf_plugin_version';
	/**
	 * @var
	 */
	private $notifications;

	/**
	 *
	 */
	public function __construct() {
		$this->register_plugin();
		add_action( 'plugins_loaded', array( $this, 'boot' ) );
		// Load text domain
		add_action(
			'init',
			array( $this, 'automator_cuf_load_textdomain' )
		);
	}

	/**
	 * @return void
	 */
	public function register_plugin() {
		new Plugin_Lifecycle_Events();
	}

	/**
	 * @return void
	 */
	public function boot() {
		do_action( 'uacuf_before_plugin_loaded' );

		self::$requirements_met = $this->requirements_met();

		$this->load();
	}

	/**
	 * @return false|mixed|null
	 */
	public static function get_version() {

		// add_option will only take action if there is no option created (plugin is activated for the first time).
		automator_add_option( self::OPTION_VERSION, '0' );

		if ( null === self::$version ) {
			self::$version = automator_get_option( self::OPTION_VERSION, '0' );
		}

		return self::$version;
	}

	/**
	 * @return bool
	 */
	private function requirements_met() {

		if ( ! $this->automator_active() ) {
			return false;
		}

		if ( ! $this->automator_pro_active() ) {
			return false;
		}

		$message = __( '<strong>Uncanny Automator - Custom User Fields Addon</strong> doesn\'t work with your current plugin license.', 'uacuf' );

		$kb_link = sprintf( ' <a href="%s">%s</a>', 'https://automatorplugin.com/knowledge-base/installing-the-custom-user-fields-addon/', __( 'Click here for more info', 'uacuf' ) );

		$license_check_response = $this->check_license();

		if ( empty( $license_check_response ) && empty( trim( get_option( 'uap_automator_pro_license_key' ) ) ) ) {
			$message = sprintf(
				__( 'Please enter your <strong>%1$s</strong> license key to use the <strong>%2$s</strong>.', 'uacuf' ),
				'Uncanny Automator Pro',
				'Custom User Fields addon'
			);
			$this->add_notification(
				'notice notice-error',
				$message . $kb_link
			);
		}

		if ( 403 === $license_check_response ) {
			$this->add_notification(
				'notice notice-error',
				$message
			);
		}

		if ( 402 === $license_check_response ) {
			$this->add_notification(
				'notice notice-error',
				$message . $kb_link
			);
		}

		return 200 === $license_check_response;
	}

	/**
	 * @return bool
	 */
	private function automator_active() {
		return defined( 'AUTOMATOR_PLUGIN_VERSION' );
	}

	/**
	 * @return bool
	 */
	private function automator_pro_active() {
		return defined( 'AUTOMATOR_PRO_PLUGIN_VERSION' );
	}

	/**
	 * @return int
	 */
	private function check_license() {
		return License::uacuf_available();
	}

	/**
	 * @return void
	 */
	private function load() {

		new License();

		add_action(
			'admin_notices',
			function () {
				// Check if AUTOMATOR_PLUGIN_VERSION is defined and if it's less than 6.0
				if ( defined( 'AUTOMATOR_PLUGIN_VERSION' ) && version_compare( AUTOMATOR_PLUGIN_VERSION, '6.0', '<' ) ) {
					// Display the admin notice with a link to the plugins page
					echo '<div class="notice notice-error">
            <p><strong>' . UACUF_ITEM_NAME . ':</strong> You must upgrade the Automator plugin to at least version 6.0 or later to activate this add-on. 
            Please <a href="' . esc_url( admin_url( 'plugins.php?s=Uncanny+Automator' ) ) . '">update the Uncanny Automator plugin</a> to continue.</p>
          </div>';
				}
			}
		);
		if ( defined( 'AUTOMATOR_PLUGIN_VERSION' ) && version_compare( AUTOMATOR_PLUGIN_VERSION, '6.0', '<' ) ) {
			return;
		}

		new Admin_Notifications( $this->notifications );

		if ( ! self::$requirements_met ) {
			return;
		}

		new Integration( $this );
		new Rest_Api( $this );
		new User_Profile( $this );
		new User_Meta_Shortcode( $this );
		new Updater( $this );
		new Blocks( $this );

		do_action( 'uacuf_plugin_loaded' );
	}

	/**
	 * @param $class
	 * @param $message
	 *
	 * @return void
	 */
	private function add_notification( $class, $message ) {
		$this->notifications[] = array(
			'class'   => $class,
			'message' => $message,
		);
	}

	/**
	 * @return void
	 */
	public function automator_cuf_load_textdomain() {
		$locale = determine_locale();

		/**
		 * Filter to adjust the Uncanny Automator locale to use for translations.
		 *
		 * Note: the first-loaded translation file overrides any following ones if the same translation is present.
		 *
		 * Locales found in:
		 *      - WP_LANG_DIR/uncanny-automator-custom-user-fields/uacuf-LOCALE.mo
		 *      - WP_LANG_DIR/plugins/uacuf-LOCALE.mo
		 */
		$locale = apply_filters( 'plugin_locale', $locale, 'uacuf' );

		unload_textdomain( 'uacuf' );
		load_textdomain( 'uacuf', WP_LANG_DIR . '/uncanny-automator-custom-user-fields/uacuf-' . $locale . '.mo' );
		load_plugin_textdomain( 'uacuf', false, plugin_basename( dirname( UACUF_PLUGIN_FILE ) ) . '/languages' );
	}
}
