<?php

namespace UncannyAutomator\AddOn\CustomUserFields\Fields;

use UncannyAutomator\AddOn\CustomUserFields\Fields\Factory as Fields_Factory;

/**
 *
 */
class Rest_Api {

	/**
	 *
	 */
	const ENDPOINT = 'uacuf';
	/**
	 *
	 */
	const NONCE = 'wp_rest';

	/**
	 * @param $plugin
	 */
	public function __construct( $plugin ) {
		add_action( 'rest_api_init', array( $this, 'add_custom_users_api' ) );
		add_filter( 'uacuf_settings_pass_to_js', array( $this, 'js_vars' ) );
	}

	/**
	 * @param $vars
	 *
	 * @return mixed
	 */
	public function js_vars( $vars ) {

		$settings = array(
			'url'   => get_rest_url( null, self::ENDPOINT ),
			'nonce' => wp_create_nonce( self::NONCE ),
		);

		$vars['restApi'] = $settings;

		$vars['fields'] = array_values( Fields_Factory::get_all() );

		return $vars;
	}

	/**
	 * @return void
	 */
	public function add_custom_users_api() {
		register_rest_route(
			self::ENDPOINT,
			'/fields',
			array(
				'methods'             => 'POST',
				'callback'            => array( $this, 'create_field' ),
				'validate_callback'   => array( $this, 'validate_call' ),
				'permission_callback' => array( $this, 'user_has_permissions' ),
			)
		);

		register_rest_route(
			self::ENDPOINT,
			'/fields',
			array(
				'methods'             => 'GET',
				'callback'            => array( $this, 'get_fields' ),
				'validate_callback'   => array( $this, 'validate_call' ),
				'permission_callback' => array( $this, 'user_has_permissions' ),
			)
		);

		register_rest_route(
			self::ENDPOINT,
			'/fields',
			array(
				'methods'             => 'DELETE',
				'callback'            => array( $this, 'delete_field' ),
				'validate_callback'   => array( $this, 'validate_call' ),
				'permission_callback' => array( $this, 'user_has_permissions' ),
			)
		);

		register_rest_route(
			self::ENDPOINT,
			'/fields',
			array(
				'methods'             => 'PUT',
				'callback'            => array( $this, 'update_field' ),
				'validate_callback'   => array( $this, 'validate_call' ),
				'permission_callback' => array( $this, 'user_has_permissions' ),
			)
		);
	}

	/**
	 * @param $request
	 *
	 * @return bool
	 */
	public function validate_call( $request ) {

		$nonce = $request->get_header( 'X-WP-Nonce' );

		if ( wp_verify_nonce( $nonce, 'wp_rest' ) ) {
			return true;
		}

		return false;
	}

	/**
	 * @return bool
	 */
	public function user_has_permissions() {
		return current_user_can( 'edit_users' );
	}

	/**
	 * @param $request
	 *
	 * @return \WP_REST_Response
	 */
	public function create_field( $request ) {

		try {

			$json_params = $request->get_json_params();

			$field = Fields_Factory::create( $json_params['type'], $json_params['meta'], $json_params['name'] );

			return $this->respond_with_data( 'field', $field, 201 );

		} catch ( \Exception $e ) {

			return $this->respond_with_data( 'error', $e->getMessage(), $e->getCode() );
		}

	}

	/**
	 * @param $request
	 *
	 * @return \WP_REST_Response
	 * @throws \Exception
	 */
	public function get_fields( $request ) {

		$fields = Fields_Factory::get();

		if ( ! empty( $request->get_param( 'isUacufEntityRecord' ) ) ) {
			return $this->entity_record_fields_response( $fields );
		}

		return $this->respond_with_data( 'fields', $fields );
	}

	/**
	 * @param $context
	 * @param $data
	 * @param $status
	 *
	 * @return \WP_REST_Response
	 */
	public function respond_with_data( $context, $data, $status = 200 ) {

		$headers = array(
			'Content-Type' => 'application/json',
		);

		$success = true;

		if ( 'error' === $context ) {
			$success = false;
		}

		$response_data = array(
			'success'    => $success,
			$context     => $data,
			'all_fields' => array_values( Fields_Factory::get_all() ),
		);

		return new \WP_REST_Response( $response_data, $status, $headers );
	}

	/**
	 * @param array $fields
	 *
	 * @return array
	 */
	public function entity_record_fields_response( $fields ) {

		$data = array();
		foreach ( $fields as $key => $field ) {
			$data[] = array(
				'meta' => $field->get_meta(),
				'name' => $field->get_name(),
			);
		}

		return new \WP_REST_Response( $data, 200 );
	}

	/**
	 * @param $request
	 *
	 * @return \WP_REST_Response
	 */
	public function delete_field( $request ) {

		try {

			$json_params = $request->get_json_params();

			$field_id = Fields_Factory::delete( $json_params['id'] );

			return $this->respond_with_data( 'id', $field_id, 200 );

		} catch ( \Exception $e ) {

			return $this->respond_with_data( 'error', $e->getMessage(), $e->getCode() );
		}
	}

	/**
	 * @param $request
	 *
	 * @return \WP_REST_Response
	 */
	public function update_field( $request ) {

		try {

			$json_params = $request->get_json_params();

			$field = Fields_Factory::update( $json_params );

			return $this->respond_with_data( 'field', $field, 200 );

		} catch ( \Exception $e ) {
			return $this->respond_with_data( 'error', $e->getMessage(), $e->getCode() );
		}
	}
}
