<?php
/**
 * Plugin Name: SW Google Tag Manager
 * Plugin URI:  https://statenweb.com/
 * Description: Adds a single ACF options page (\"GTM\") where you can store your Google Tag Manager container ID and automatically outputs the GTM snippet in <head> and immediately after <body>.
 * Version:     1.0.1
 * Author:      StatenWeb LLC
 * Author URI:  https://statenweb.com/
 * License:     GPL-2.0-or-later
 * License URI: https://www.gnu.org/licenses/gpl-2.0.html
 * Text Domain: sw-gtm
 */

if ( ! defined( 'ABSPATH' ) ) {
    exit; // Exit if accessed directly.
}

class SW_GTM {

    /**
     * Kick things off on the plugins_loaded action so ACF is available first.
     */
    public function __construct() {
        add_action( 'plugins_loaded', array( $this, 'init' ) );
    }

    /**
     * Initialise the plugin – only if ACF Pro (with Options Pages) is present.
     */
    public function init() : void {
        if ( ! function_exists( 'acf_add_options_page' ) ) {
            // ACF Pro (or at least the Options Page module) is not present – bail early.
            return;
        }

        $this->attach_hooks();
    }

    /* -------------------------------------------------------------------------
     *  Public Helpers
     * ---------------------------------------------------------------------- */

    /**
     * Return the normalised GTM container ID (just the hash, no “GTM-”).
     */
    public static function get_id() : string {
        $raw = get_field( 'gtm_id', 'option' );

        if ( empty( $raw ) ) {
            return '';
        }

        // Remove any leading “GTM-” or stray punctuation, leaving only the alphanumeric ID.
        $normalised = preg_replace( '/^GTM[-_\s]*/i', '', trim( $raw ) );
        $normalised = preg_replace( '/[^A-Za-z0-9]/', '', $normalised );

        return strtoupper( 'GTM-' . $normalised );
    }

    /* -------------------------------------------------------------------------
     *  ACF Field Registration
     * ---------------------------------------------------------------------- */

    /**
     * Register the Options Page and the single \"GTM ID\" field.
     */
    public function add_fields() : void {
        // Create an Options Page titled “GTM”.
        acf_add_options_page( array(
            'page_title' => 'Google Tag Manager',
            'menu_title' => 'GTM',
            'menu_slug'  => 'sw-gtm',
            'capability' => 'manage_options',
            'redirect'   => false,
            'position'   => 80,
            'icon_url'   => 'dashicons-tag',
        ) );

        // Register the field group + field.
        acf_add_local_field_group( array(
            'key'                   => 'group_sw_gtm',
            'title'                 => 'Google Tag Manager',
            'menu_order'            => 0,
            'position'              => 'normal',
            'style'                 => 'default',
            'label_placement'       => 'left',
            'instruction_placement' => 'label',
            'active'                => true,
            'fields'                => array(
                array(
                    'key'          => 'field_sw_gtm_id',
                    'label'        => 'GTM ID',
                    'name'         => 'gtm_id',
                    'type'         => 'text',
                    'instructions' => 'Enter your container ID (e.g. GTM‑ABCDE123 or ABCDE123). Any “GTM‑” prefix will be ignored.',
                    'placeholder'  => 'GTM-XXXXX',
                    'wrapper'      => array( 'width' => '40%' ),
                ),
            ),
            'location'              => array(
                array(
                    array(
                        'param'    => 'options_page',
                        'operator' => '==',
                        'value'    => 'sw-gtm',
                    ),
                ),
            ),
        ) );
    }

    /* -------------------------------------------------------------------------
     *  Hooks & Rendering
     * ---------------------------------------------------------------------- */

    /**
     * Attach WP and ACF action hooks.
     */
    private function attach_hooks() : void {
        add_action( 'acf/init',     array( $this, 'add_fields' ) );
        add_action( 'wp_head',      array( $this, 'add_google_tag_manager_head' ), 1 );
        add_action( 'wp_body_open', array( $this, 'add_google_tag_manager' ) );
    }

    /**
     * Output the standard GTM <script> in <head>.
     */
    public function add_google_tag_manager_head() : void {
        $gtm_id = self::get_id();

        if ( ! $gtm_id ) {
            return; // Nothing set – don’t print the snippet.
        }
        ?>
        <!-- Google Tag Manager -->
        <script>(function(w,d,s,l,i){w[l]=w[l]||[];w[l].push({'gtm.start':
        new Date().getTime(),event:'gtm.js'});var f=d.getElementsByTagName(s)[0],
        j=d.createElement(s),dl=l!='dataLayer'?'&l='+l:'';j.async=true;j.src=
        'https://www.googletagmanager.com/gtm.js?id='+i+dl;f.parentNode.insertBefore(j,f);
        })(window,document,'script','dataLayer','<?php echo esc_js( $gtm_id ); ?>');</script>
        <!-- End Google Tag Manager -->
        <?php
    }

    /**
     * Output the <noscript> version immediately after <body>.
     */
    public function add_google_tag_manager() : void {
        $gtm_id = self::get_id();

        if ( ! $gtm_id ) {
            return;
        }
        ?>
        <!-- Google Tag Manager (noscript) -->
        <noscript><iframe src="https://www.googletagmanager.com/ns.html?id=<?php echo esc_attr( $gtm_id ); ?>"
                          height="0" width="0" style="display:none;visibility:hidden"></iframe></noscript>
        <!-- End Google Tag Manager (noscript) -->
        <?php
    }
}

// Instantiate the class.
new SW_GTM();

