<?php

class Smuac_Public {

	public function __construct() {

		add_action(
			'plugins_loaded',
			function() {

				// Only load if WooCommerce is activated
				if ( class_exists( 'woocommerce' ) ) {

					// Load colors
					add_action( 'wp_head', array( $this, 'smuac_custom_color' ) );

					$user_data_current_user_id = get_current_user_id();

					$account_type = get_user_meta( $user_data_current_user_id, 'smuac_account_type', true );
					if ( 'multiaccount' === $account_type ) {
						// for all intents and purposes set current user as the multiaccount parent
						$parent_user_id            = get_user_meta( $user_data_current_user_id, 'smuac_account_parent', true );
						$user_data_current_user_id = $parent_user_id;

						// Mention in order notes that order is placed by multiaccount and point to main accounts
						add_action( 'woocommerce_thankyou', array( $this, 'smuac_multiaccount_order_note' ) );
					}

					$user_data_current_user_b2b = get_user_meta( $user_data_current_user_id, 'smuac_b2buser', true );

					$user_data_current_user_group = get_user_meta( $user_data_current_user_id, 'smuac_customergroup', true );

					// Enqueue resources
					add_action( 'wp_enqueue_scripts', array( $this, 'enqueue_public_resources' ) );

					/*
					 Add items to "My Account" */
					// Add custom items to My account WooCommerce user menu
					add_filter( 'woocommerce_account_menu_items', array( $this, 'smuac_my_account_custom_items' ), 10, 1 );
					// Add custom endpoints
					add_action( 'init', array( $this, 'smuac_custom_endpoints' ) );
					if ( 0 === intval( get_option( 'smuac_force_permalinks_setting', 0 ) ) ) {
						// Add redirects by default to prevent 404 problems
						add_action( 'template_redirect', array( $this, 'smuac_redirects_my_account_default' ) );
						// adds "id" query var to WP list. Makes the query recognizable by wp
						add_filter( 'query_vars', array( $this, 'smuac_add_query_vars_filter' ) );
					}

					if ( 1 === intval( get_option( 'smuac_force_permalinks_flushing_setting', 0 ) ) ) {
						add_action( 'init', array( $this, 'force_permalinks_rewrite' ) );
					}

					if ( 1 === intval( get_option( 'smuac_enable_multiaccounts_setting', 1 ) ) ) {
						/*
						 Multiaccount */
						// Add content to multiaccounts endpoint
						add_action( 'woocommerce_account_multiaccounts_endpoint', array( $this, 'smuac_multiaccounts_endpoint_content' ) );
						add_action( 'woocommerce_account_multiaccount_endpoint', array( $this, 'smuac_multiaccount_endpoint_content' ) );
						// Multiaccount: add "Placed by" column in Orders
						add_filter( 'woocommerce_my_account_my_orders_columns', array( $this, 'smuac_orders_placed_by_column' ) );
						// Add data to "Placed by" column
						add_action( 'woocommerce_my_account_my_orders_column_order-placed-by', array( $this, 'smuac_orders_placed_by_column_content' ) );
						// Add multiaccounts orders to main account order query
						add_filter( 'woocommerce_my_account_my_orders_query', array( $this, 'smuac_add_multiaccounts_orders_to_main_query' ), 10, 1 );
						// Give main account permission to view multiaccount orders
						add_filter( 'user_has_cap', array( $this, 'smuac_give_main_account_view_multiaccount_orders_permission' ), 10, 3 );
						// Give permissions to order again
						add_filter( 'user_has_cap', array( $this, 'smuac_multiaccounts_orderagain_cap' ), 10, 3 );

						// Multiaccount checkout permission validation
						add_action( 'woocommerce_after_checkout_validation', array( $this, 'smuac_multiaccount_checkout_permission_validation' ) );

					}

					/* General */

					if ( 1 === intval( get_option( 'smuac_enable_shopping_lists_setting', 1 ) ) ) {
						/*
						 Shopping list */
						// Add content to shopping lists endpoints
						add_action( 'woocommerce_account_purchase-lists_endpoint', array( $this, 'smuac_shopping_lists_endpoint_content' ) );
						add_action( 'woocommerce_account_purchase-list_endpoint', array( $this, 'smuac_shopping_list_endpoint_content' ) );
						// Add "Save cart as Shopping List" button
						add_action( 'woocommerce_cart_actions', array( $this, 'smuac_shopping_list_cart_button' ) );
					}
				}
			}
		);
	}



	public function smuac_multiaccount_checkout_permission_validation() {
		$user_id = get_current_user_id();
		// check if multiaccount
		$account_type = get_user_meta( $user_id, 'smuac_account_type', true );
		if ( 'multiaccount' === $account_type ) {
			// if it's multiaccount check, if multiaccount has permission to checkout
			$permission_checkout = filter_var( get_user_meta( $user_id, 'smuac_account_permission_buy', true ), FILTER_VALIDATE_BOOLEAN );
			if ( false === $permission_checkout ) {
				wc_add_notice( esc_html__( 'Your account does not have permission to checkout', 'smuac' ), 'error' );
			}
		}
	}


	public function smuac_user_is_in_list( $user_data_current_user_id, $user_data_current_user_b2b, $user_data_current_user_group, $list ) {
		// checks based on user id, b2b status and group, if it's part of an applicable rules list
		$is_in_list = 'no';
		$list_array = explode( ',', $list );
		if ( 0 !== intval( $user_data_current_user_id ) ) {
			if ( in_array( 'everyone_registered', $list_array ) ) {
				return 'yes';
			}
			if ( 'yes' === $user_data_current_user_b2b ) {
				// user is b2b
				if ( in_array( 'everyone_registered_b2b', $list_array ) ) {
					return 'yes';
				}
				if ( in_array( 'group_' . $user_data_current_user_group, $list_array ) ) {
					return 'yes';
				}
			} else {
				// user is b2c
				if ( in_array( 'everyone_registered_b2c', $list_array ) ) {
					return 'yes';
				}
			}
			if ( in_array( 'user_' . $user_data_current_user_id, $list_array ) ) {
				return 'yes';
			}
		} elseif ( 0 === intval( $user_data_current_user_id ) ) {
			if ( in_array( 'user_0', $list_array ) ) {
				return 'yes';
			}
		}

		return $is_in_list;
	}


	// Add item metadata to order
	public function smuac_add_item_metadata_to_order( $item, $cart_item_key, $values, $order ) {

		if ( isset( $values['smuac_bundle_name'] ) ) {
			$item->update_meta_data( esc_html__( 'Bundle name', 'smuac' ), esc_html( $values['smuac_bundle_name'] ) );
			// add products to details string
			$details = '';
			for ( $i = 1; $i <= intval( $values['smuac_numberofproducts'] ); $i++ ) {

				$unit_price_display = $values[ 'smuac_product_' . $i . '_price' ];
				// adjust for tax
				// get bundle product
				$bundle_id      = intval( get_option( 'smuac_bundle_product_id_setting', 0 ) );
				$bundle_product = wc_get_product( $bundle_id );
				if ( wc_prices_include_tax() && ( 'incl' !== get_option( 'woocommerce_tax_display_shop' ) || WC()->customer->is_vat_exempt() ) ) {
					// if prices are entered including tax, but display is without tax, remove tax
					// get tax rate for the bundle product
					$tax_rates          = WC_Tax::get_base_tax_rates( $bundle_product->get_tax_class( 'unfiltered' ) );
					$taxes              = WC_Tax::calc_tax( $unit_price_display, $tax_rates, true );
					$unit_price_display = WC_Tax::round( $unit_price_display - array_sum( $taxes ) );

				} elseif ( ! wc_prices_include_tax() && ( 'incl' === get_option( 'woocommerce_tax_display_shop' ) && ! WC()->customer->is_vat_exempt() ) ) {
					// if prices are entered excluding tax, but display is with tax, add tax
					$tax_rates          = WC_Tax::get_rates( $bundle_product->get_tax_class() );
					$taxes              = WC_Tax::calc_tax( $unit_price_display, $tax_rates, false );
					$unit_price_display = WC_Tax::round( $unit_price_display + array_sum( $taxes ) );
				}

				$details .= $values[ 'smuac_product_' . $i . '_name' ] . ' - ' . esc_html__( 'Qty', 'smuac' ) . ': ' . $values[ 'smuac_product_' . $i . '_quantity' ] . ' - ' . esc_html__( 'Unit Price', 'smuac' ) . ': ' . round( $unit_price_display, wc_get_price_decimals() ) . ' <br />';
			}

			$item->update_meta_data( esc_html__( 'Details', 'smuac' ), $details );
		}
	}

	public function smuac_display_metadata_cart( $product_name, $values, $cart_item_key ) {
		// If product is an bundle
		if ( ! empty( $values['smuac_numberofproducts'] ) ) {
			$details = '';
			for ( $i = 1; $i <= intval( $values['smuac_numberofproducts'] ); $i++ ) {
				// adjust unit price for tax
				$unit_price_display = $values[ 'smuac_product_' . $i . '_price' ];
				// get bundle product
				$bundle_id      = intval( get_option( 'smuac_bundle_product_id_setting', 0 ) );
				$bundle_product = wc_get_product( $bundle_id );
				if ( wc_prices_include_tax() && ( 'incl' !== get_option( 'woocommerce_tax_display_shop' ) || WC()->customer->is_vat_exempt() ) ) {
					// if prices are entered including tax, but display is without tax, remove tax
					// get tax rate for the bundle product
					$tax_rates          = WC_Tax::get_base_tax_rates( $bundle_product->get_tax_class( 'unfiltered' ) );
					$taxes              = WC_Tax::calc_tax( $unit_price_display, $tax_rates, true );
					$unit_price_display = WC_Tax::round( $unit_price_display - array_sum( $taxes ) );

				} elseif ( ! wc_prices_include_tax() && ( 'incl' === get_option( 'woocommerce_tax_display_shop' ) && ! WC()->customer->is_vat_exempt() ) ) {
					// if prices are entered excluding tax, but display is with tax, add tax
					$tax_rates          = WC_Tax::get_rates( $bundle_product->get_tax_class() );
					$taxes              = WC_Tax::calc_tax( $unit_price_display, $tax_rates, false );
					$unit_price_display = WC_Tax::round( $unit_price_display + array_sum( $taxes ) );
				}

				$details .= $values[ 'smuac_product_' . $i . '_name' ] . ' - ' . esc_html__( 'Qty', 'smuac' ) . ': ' . $values[ 'smuac_product_' . $i . '_quantity' ] . ' - ' . esc_html__( 'Unit Price', 'smuac' ) . ': ' . round( $unit_price_display, wc_get_price_decimals() ) . ' <br />';
			}
			return $product_name . '<br />' . $values['smuac_bundle_name'] . '<br /><strong>' . esc_html__( 'Details', 'smuac' ) . ':</strong><br />' . $details;
		} else {
			return $product_name;
		}

	}

	// Add custom items to My account WooCommerce user menu
	public function smuac_my_account_custom_items( $items ) {
		// Get current user
		$user_id      = get_current_user_id();
		$account_type = get_user_meta( $user_id, 'smuac_account_type', true );
		if ( 'multiaccount' === $account_type ) {
			// for all intents and purposes set current user as the multiaccount parent
			$parent_user_id = get_user_meta( $user_id, 'smuac_account_parent', true );
			$user_id        = $parent_user_id;
		}

		// Add shopping lists
		if ( 1 === intval( get_option( 'smuac_enable_shopping_lists_setting', 1 ) ) ) {
			$items = array_slice( $items, 0, 5, true ) +
				array( 'purchase-lists' => esc_html__( 'Shopping Lists', 'smuac' ) ) +
				array_slice( $items, 5, count( $items ) - 5, true );
		}

		// Add multiaccounts
		if ( 1 === intval( get_option( 'smuac_enable_multiaccounts_setting', 1 ) ) ) {
			// only show if current account is not itself a multiaccount
			if ( 'multiaccount' !== $account_type ) {
				$items = array_slice( $items, 0, 6, true ) +
					array( 'multiaccounts' => esc_html__( 'Multi Accounts', 'smuac' ) ) +
					array_slice( $items, 6, count( $items ) - 6, true );
			}
		}

		return $items;

	}

	// Add custom endpoints
	public function smuac_custom_endpoints() {

		// Multiaccounts
		if ( 1 === intval( get_option( 'smuac_enable_multiaccounts_setting', 1 ) ) ) {
			// only show if current account is not itself a multiaccount
			$account_type = get_user_meta( get_current_user_id(), 'smuac_account_type', true );
			if ( 'multiaccount' !== $account_type ) {
				add_rewrite_endpoint( 'multiaccounts', EP_ROOT | EP_PAGES | EP_PERMALINK );
				add_rewrite_endpoint( 'multiaccount', EP_ROOT | EP_PAGES | EP_PERMALINK );
			}
		}
		// Shopping Lists
		if ( 1 === intval( get_option( 'smuac_enable_shopping_lists_setting', 1 ) ) ) {
			add_rewrite_endpoint( 'purchase-lists', EP_ROOT | EP_PAGES | EP_PERMALINK );
			add_rewrite_endpoint( 'purchase-list', EP_ROOT | EP_PAGES | EP_PERMALINK );
		}

	}

	public function smuac_add_query_vars_filter( $vars ) {
		$vars[] = 'id';
		return $vars;
	}

	public function smuac_redirects_my_account_default() {

		if ( isset( $_SERVER['HTTPS'] ) &&
				( 'on' === $_SERVER['HTTPS'] || 1 === $_SERVER['HTTPS'] ) ||
				isset( $_SERVER['HTTP_X_FORWARDED_PROTO'] ) &&
				'https' === $_SERVER['HTTP_X_FORWARDED_PROTO'] ) {
				$protocol = 'https://';
		} else {
			$protocol = 'http://';
		}

		if ( isset( $_SERVER['HTTP_HOST'] ) && isset( $_SERVER['REQUEST_URI'] ) ) {
			$currenturl = $protocol . sanitize_text_field( $_SERVER['HTTP_HOST'] ) . sanitize_text_field( $_SERVER['REQUEST_URI'] );
		}
		$currenturl_relative = wp_make_link_relative( remove_query_arg( 'id', $currenturl ) );
		$idqueryvar          = get_query_var( 'id' );

		$bulkorderurl      = wp_make_link_relative( wc_get_endpoint_url( 'bulkorder' ) );
		$bulkorderurlbuilt = wp_make_link_relative( get_permalink( get_option( 'woocommerce_myaccount_page_id' ) ) ) . 'bulkorder/';

		$discussionsurl      = wp_make_link_relative( wc_get_endpoint_url( 'discussions' ) );
		$discussionsurlbuilt = wp_make_link_relative( get_permalink( get_option( 'woocommerce_myaccount_page_id' ) ) ) . 'discussions/';

		$discussionurl      = wp_make_link_relative( wc_get_endpoint_url( 'discussion' ) );
		$discussionurlbuilt = wp_make_link_relative( get_permalink( get_option( 'woocommerce_myaccount_page_id' ) ) ) . 'discussion/';

		$bundlesurl      = wp_make_link_relative( wc_get_endpoint_url( 'bundles' ) );
		$bundlesurlbuilt = wp_make_link_relative( get_permalink( get_option( 'woocommerce_myaccount_page_id' ) ) ) . 'bundles/';

		$multiaccountsurl      = wp_make_link_relative( wc_get_endpoint_url( 'multiaccounts' ) );
		$multiaccountsurlbuilt = wp_make_link_relative( get_permalink( get_option( 'woocommerce_myaccount_page_id' ) ) ) . 'multiaccounts/';

		$multiaccounturl      = wp_make_link_relative( wc_get_endpoint_url( 'multiaccount' ) );
		$multiaccounturlbuilt = wp_make_link_relative( get_permalink( get_option( 'woocommerce_myaccount_page_id' ) ) ) . 'multiaccount/';

		$purchaselistssurl      = wp_make_link_relative( wc_get_endpoint_url( 'purchase-lists' ) );
		$purchaselistssurlbuilt = wp_make_link_relative( get_permalink( get_option( 'woocommerce_myaccount_page_id' ) ) ) . 'purchase-lists/';

		$purchaselisturl      = wp_make_link_relative( wc_get_endpoint_url( 'purchase-list' ) );
		$purchaselisturlbuilt = wp_make_link_relative( get_permalink( get_option( 'woocommerce_myaccount_page_id' ) ) ) . 'purchase-list/';

		$setredirect = 'no';
		switch ( $currenturl_relative ) {

			case $bulkorderurl:
			case $bulkorderurlbuilt:
				$urlto       = get_permalink( get_option( 'woocommerce_myaccount_page_id' ) ) . '?bulkorder';
				$setredirect = 'yes';
				break;

			case $discussionsurl:
			case $discussionsurlbuilt:
				$urlto       = get_permalink( get_option( 'woocommerce_myaccount_page_id' ) ) . '?discussions';
				$setredirect = 'yes';
				break;

			case $purchaselistssurl:
			case $purchaselistssurlbuilt:
				$urlto       = get_permalink( get_option( 'woocommerce_myaccount_page_id' ) ) . '?purchase-lists';
				$setredirect = 'yes';
				break;

			case $bundlesurl:
			case $bundlesurlbuilt:
				$urlto       = get_permalink( get_option( 'woocommerce_myaccount_page_id' ) ) . '?bundles';
				$setredirect = 'yes';
				break;

			case $multiaccountsurl:
			case $multiaccountsurlbuilt:
				$urlto       = get_permalink( get_option( 'woocommerce_myaccount_page_id' ) ) . '?multiaccounts';
				$setredirect = 'yes';
				break;

			case $multiaccounturl:
			case $multiaccounturlbuilt:
				$urlto       = get_permalink( get_option( 'woocommerce_myaccount_page_id' ) ) . '?multiaccount&id=' . $idqueryvar;
				$setredirect = 'yes';
				break;

			case $purchaselisturl:
			case $purchaselisturlbuilt:
				$urlto       = get_permalink( get_option( 'woocommerce_myaccount_page_id' ) ) . '?purchase-list&id=' . $idqueryvar;
				$setredirect = 'yes';
				break;

			case $discussionurl:
			case $discussionurlbuilt:
				$urlto       = get_permalink( get_option( 'woocommerce_myaccount_page_id' ) ) . '?discussion&id=' . $idqueryvar;
				$setredirect = 'yes';
				break;

			default:
				return;
		}

		if ( 'yes' === $setredirect ) {
			wp_redirect( $urlto );
			exit();
		}

	}


	// Multiaccounts Endpoint Content
	public function smuac_multiaccounts_endpoint_content() {
		$account_type = get_user_meta( get_current_user_id(), 'smuac_account_type', true );
		?>
		<div class="smuac_multiaccounts_container">
			<div class="smuac_multiaccounts_container_top">
				<div class="smuac_multiaccounts_container_top_title">
					<?php esc_html_e( 'Multi-User Accounts', 'smuac' ); ?>
				</div>
				<?php
				// only available if current account is not itself a multiaccount
				if ( 'multiaccount' !== $account_type ) {
					?>
					<button class="smuac_multiaccounts_container_top_button" type="button">
						<svg class="smuac_multiaccounts_container_top_button_icon" xmlns="http://www.w3.org/2000/svg" width="20" height="20" fill="none" viewBox="0 0 20 20">
						  <path fill="#fff" fill-rule="evenodd" d="M10.833 10.833v5.834a.834.834 0 01-1.666 0v-5.834H3.333a.834.834 0 010-1.666h5.834V3.333a.833.833 0 111.666 0v5.834h5.834a.833.833 0 010 1.666h-5.834z" clip-rule="evenodd"/>
						</svg>
						<?php esc_html_e( 'New multiaccount', 'smuac' ); ?>
					</button>
					<?php
				}
				?>
			</div>

			<!-- Hidden New Multiaccount Container -->
			<?php
			// only available if current account is not itself a multiaccount
			if ( 'multiaccount' !== $account_type ) {
				?>
				<div class="smuac_multiaccounts_new_account_container">
					<div class="smuac_multiaccounts_new_account_container_top">
						<div class="smuac_multiaccounts_new_account_container_top_title">
							<?php esc_html_e( 'New Multiaccount', 'smuac' ); ?>
						</div>
						<div class="smuac_multiaccounts_new_account_container_top_close">
							<?php esc_html_e( 'Close X', 'smuac' ); ?>
						</div>
					</div>
					<div class="smuac_multiaccounts_new_account_container_content">
						<div class="smuac_multiaccounts_new_account_container_content_large_title">
							<svg class="smuac_multiaccounts_new_account_container_content_large_title_icon" xmlns="http://www.w3.org/2000/svg" width="35" height="25" fill="none" viewBox="0 0 35 25">
							  <path fill="#4E4E4E" d="M22.75 10.5H35V14H22.75v-3.5zm1.75 7H35V21H24.5v-3.5zM21 3.5h14V7H21V3.5zm-17.5 21H21v-1.75c0-4.825-3.925-8.75-8.75-8.75h-3.5C3.925 14 0 17.925 0 22.75v1.75h3.5zm7-12.25c3.491 0 6.125-2.634 6.125-6.125S13.991 0 10.5 0 4.375 2.634 4.375 6.125 7.009 12.25 10.5 12.25z"/>
							</svg>
							<span class="smuac_span_title_text_multiaccount"><?php esc_html_e( 'Login Details', 'smuac' ); ?></span>
						</div>
						<?php

						if ( apply_filters( 'smuac_disable_username_subaccounts', 1 ) === 0 ) {
							?>
							<div class="smuac_multiaccounts_new_account_container_content_element">
								<div class="smuac_multiaccounts_new_account_container_content_element_label">
									<?php esc_html_e( 'Username', 'smuac' ); ?>
								</div>
								<input type="text" class="smuac_multiaccounts_new_account_container_content_element_text" name="smuac_multiaccounts_new_account_username" placeholder="<?php esc_attr_e( 'Enter the multiaccount username here...', 'smuac' ); ?>" >
							</div>
							<?php
						}
						?>
						<div class="smuac_multiaccounts_new_account_container_content_element">
							<div class="smuac_multiaccounts_new_account_container_content_element_label">
								<?php esc_html_e( 'Email Address', 'smuac' ); ?>
							</div>
							<input type="text" class="smuac_multiaccounts_new_account_container_content_element_text" name="smuac_multiaccounts_new_account_email_address" placeholder="<?php esc_attr_e( 'Enter the multiaccount email here...', 'smuac' ); ?>">
						</div>
						<div class="smuac_multiaccounts_new_account_container_content_element smuac_multiaccount_horizontal_line">
							<div class="smuac_multiaccounts_new_account_container_content_element_label">
								<?php esc_html_e( 'Password', 'smuac' ); ?>
							</div>
							<input type="text" class="smuac_multiaccounts_new_account_container_content_element_text" name="smuac_multiaccounts_new_account_password" placeholder="<?php esc_attr_e( 'Enter the multiaccount password here...', 'smuac' ); ?>" >
						</div>
						<div class="smuac_multiaccounts_new_account_container_content_large_title smuac_multiaccount_top_margin">
							<svg class="smuac_multiaccounts_new_account_container_content_large_title_icon" xmlns="http://www.w3.org/2000/svg" width="35" height="29" fill="none" viewBox="0 0 35 29">
							  <path fill="#4E4E4E" d="M12.25 14.063c3.867 0 7-3.148 7-7.031 0-3.884-3.133-7.032-7-7.032-3.866 0-7 3.148-7 7.032 0 3.883 3.134 7.031 7 7.031zm4.9 1.758h-.913a9.494 9.494 0 01-3.986.879 9.512 9.512 0 01-3.987-.88H7.35C3.292 15.82 0 19.129 0 23.205v2.285a2.632 2.632 0 002.625 2.637H17.66a2.648 2.648 0 01-.142-1.17l.372-3.346.066-.61.432-.433 4.227-4.247c-1.34-1.521-3.281-2.5-5.463-2.5zm2.478 7.982l-.372 3.35a.873.873 0 00.963.968l3.33-.374 7.542-7.575-3.921-3.94-7.542 7.57zm14.99-9.031l-2.072-2.082a1.306 1.306 0 00-1.849 0l-2.067 2.076-.224.225 3.927 3.94 2.285-2.297a1.327 1.327 0 000-1.862z"/>
							</svg>
							<span class="smuac_span_title_text_multiaccount"><?php esc_html_e( 'Personal Details', 'smuac' ); ?></span>
						</div>
						<div class="smuac_multiaccounts_new_account_container_content_element">
							<div class="smuac_multiaccounts_new_account_container_content_element_label">
								<?php esc_html_e( 'Name', 'smuac' ); ?>
							</div>
							<input type="text" class="smuac_multiaccounts_new_account_container_content_element_text" name="smuac_multiaccounts_new_account_name" placeholder="<?php esc_attr_e( 'Enter the account holder\'s name here...', 'smuac' ); ?>">
						</div>
						<div class="smuac_multiaccounts_new_account_container_content_element">
							<div class="smuac_multiaccounts_new_account_container_content_element_label">
								<?php esc_html_e( 'Job Title', 'smuac' ); ?>
							</div>
							<input type="text" class="smuac_multiaccounts_new_account_container_content_element_text" name="smuac_multiaccounts_new_account_job_title" placeholder="<?php esc_attr_e( 'Enter the account holder\'s title here...', 'smuac' ); ?>">
						</div>
						<div class="smuac_multiaccounts_new_account_container_content_element smuac_multiaccount_horizontal_line">
							<div class="smuac_multiaccounts_new_account_container_content_element_label">
								<?php esc_html_e( 'Phone Number', 'smuac' ); ?>
							</div>
							<input type="text" class="smuac_multiaccounts_new_account_container_content_element_text" name="smuac_multiaccounts_new_account_phone_number" placeholder="<?php esc_attr_e( 'Enter the account holder\'s phone here...', 'smuac' ); ?>">
						</div>
						<div class="smuac_multiaccounts_new_account_container_content_large_title smuac_multiaccount_top_margin">
							<svg class="smuac_multiaccounts_new_account_container_content_large_title_icon" xmlns="http://www.w3.org/2000/svg" width="35" height="24" fill="none" viewBox="0 0 35 24">
							  <path fill="#575757" d="M16.042 8.75v2.917h-1.459v2.916h-2.916v-2.916H8.502a4.36 4.36 0 01-4.127 2.916 4.375 4.375 0 110-8.75A4.36 4.36 0 018.502 8.75h7.54zm-11.667 0a1.458 1.458 0 100 2.917 1.458 1.458 0 000-2.917zm18.958 5.833c3.894 0 11.667 1.955 11.667 5.834v2.916H11.667v-2.916c0-3.88 7.773-5.834 11.666-5.834zm0-2.916a5.833 5.833 0 110-11.667 5.833 5.833 0 010 11.667z"/>
							</svg>
							<span class="smuac_span_title_text_multiaccount"><?php esc_html_e( 'Permissions', 'smuac' ); ?></span>
						</div>
						<div class="smuac_multiaccounts_new_account_container_content_element_checkbox">
							<div class="smuac_multiaccounts_new_account_container_content_element_checkbox_name">
								<?php esc_html_e( 'Place order', 'smuac' ); ?>
							</div>
							<input type="checkbox" value="1" class="smuac_multiaccounts_new_account_container_content_element_checkbox_input" name="smuac_multiaccounts_new_account_container_content_element_checkbox_buy">
						</div>
						<div class="smuac_multiaccounts_new_account_container_content_element_checkbox">
							<div class="smuac_multiaccounts_new_account_container_content_element_checkbox_name">
								<?php esc_html_e( 'View account orders', 'smuac' ); ?>
							</div>
							<input type="checkbox" value="1" class="smuac_multiaccounts_new_account_container_content_element_checkbox_input" name="smuac_multiaccounts_new_account_container_content_element_checkbox_view_orders">
						</div>
						<?php if ( 1 === intval( get_option( 'smuac_enable_shopping_lists_setting', 1 ) ) ) { ?>
							<div class="smuac_multiaccounts_new_account_container_content_element_checkbox">
								<div class="smuac_multiaccounts_new_account_container_content_element_checkbox_name">
									<?php esc_html_e( 'View account shopping lists', 'smuac' ); ?>
								</div>
								<input type="checkbox" value="1" class="smuac_multiaccounts_new_account_container_content_element_checkbox_input" name="smuac_multiaccounts_new_account_container_content_element_checkbox_view_lists">
							</div>
							<?php
						}

						do_action( 'smuac_after_my_account_permissions' );

						?>
						<div class="smuac_multiaccounts_new_account_container_content_bottom">
							<div class="smuac_multiaccounts_new_account_container_content_bottom_validation_errors">
							</div>
							<button class="smuac_multiaccounts_new_account_container_content_bottom_button" type="button">
								<svg class="smuac_multiaccounts_new_account_container_content_bottom_button_icon" xmlns="http://www.w3.org/2000/svg" width="30" height="20" fill="none" viewBox="0 0 30 20">
								  <path fill="#fff" d="M4.375 5.115c0 2.827 2.132 4.959 4.958 4.959 2.827 0 4.959-2.132 4.959-4.959 0-2.826-2.132-4.958-4.959-4.958-2.826 0-4.958 2.132-4.958 4.958zm20.542-.782h-2.834v4.25h-4.25v2.834h4.25v4.25h2.834v-4.25h4.25V8.583h-4.25v-4.25zM3.667 19.917h14.166V18.5a7.091 7.091 0 00-7.083-7.083H7.917A7.091 7.091 0 00.833 18.5v1.417h2.834z"/>
								</svg>
								<?php esc_html_e( 'Create Multiaccount', 'smuac' ); ?>
							</button>
						</div>
					</div>
				</div>
				<?php
			}

			// Get all multiaccounts and display them;
			$user_id                 = get_current_user_id();
			$user_multiaccounts_list = get_user_meta( $user_id, 'smuac_multiaccounts_list', true );
			$multiaccounts_array     = explode( ',', $user_multiaccounts_list );
			$multiaccounts_array     = array_filter( $multiaccounts_array ); // removing blank, null, false, 0 (zero) values
			$multiaccounts_array     = array_reverse( $multiaccounts_array ); // show newest first

			if ( empty( $multiaccounts_array ) ) {
				wc_print_notice( esc_html__( 'No multiaccounts exist.', 'smuac' ), 'notice' );
			}
			foreach ( $multiaccounts_array as $multiaccount ) {
				// display multiaccount
				$user      = get_user_by( 'ID', $multiaccount );
				$username  = $user->user_login;
				$name      = get_user_meta( $multiaccount, 'smuac_account_name', true );
				$job_title = get_user_meta( $multiaccount, 'smuac_account_job_title', true );
				$phone     = get_user_meta( $multiaccount, 'smuac_account_phone', true );
				$email     = $user->user_email;
				// Get Multiaccount Endpoint URL
				$endpointurl = wc_get_endpoint_url( 'multiaccount' );
				?>
				<div class="smuac_multiaccounts_account_container">
					<div class="smuac_multiaccounts_account_top">
						<svg class="smuac_multiaccounts_account_top_icon" xmlns="http://www.w3.org/2000/svg" width="24" height="24" fill="none" viewBox="0 0 24 24">
						  <path fill="#fff" d="M12 2C6.579 2 2 6.579 2 12s4.579 10 10 10 10-4.579 10-10S17.421 2 12 2zm0 5c1.727 0 3 1.272 3 3s-1.273 3-3 3c-1.726 0-3-1.272-3-3s1.274-3 3-3zm-5.106 9.772c.897-1.32 2.393-2.2 4.106-2.2h2c1.714 0 3.209.88 4.106 2.2A6.969 6.969 0 0112 19a6.969 6.969 0 01-5.106-2.228z"/>
						</svg>
						<?php echo esc_html( $username ); ?>
					</div>
					<div class="smuac_multiaccounts_account_line">
						<div class="smuac_multiaccounts_account_name_title">
							<div class="smuac_multiaccounts_account_name">
								<?php echo esc_html( $name ); ?>
							</div>
							<div class="smuac_multiaccounts_account_title">
								<?php echo esc_html( $job_title ); ?>
							</div>
						</div>
						<div class="smuac_multiaccounts_account_phone_email">
							<div class="smuac_multiaccounts_account_phone_email_text">
								<?php echo esc_html( $phone ); ?>
							</div>
							<div class="smuac_multiaccounts_account_phone_email_text">
								<?php echo esc_html( $email ); ?>
							</div>
						</div>
					</div>
					<div class="smuac_multiaccounts_account_line">
						
						<a href="<?php echo esc_url( add_query_arg( 'id', $multiaccount, $endpointurl ) ); ?>">
							<button class="smuac_multiaccounts_account_button" type="button">
								<svg class="smuac_multiaccounts_account_button_icon" xmlns="http://www.w3.org/2000/svg" width="17" height="17" fill="none" viewBox="0 0 17 17">
								  <path fill="#fff" d="M14.61 13.879H2.39a.53.53 0 00-.53.531v.598c0 .073.06.133.132.133h13.016c.073 0 .133-.06.133-.133v-.598a.53.53 0 00-.532-.531zM4.277 12.484a.67.67 0 00.1-.008l2.792-.49a.162.162 0 00.088-.046l7.038-7.038a.165.165 0 000-.234l-2.76-2.76a.164.164 0 00-.117-.049.164.164 0 00-.118.049L4.263 8.945a.169.169 0 00-.046.088l-.49 2.792a.556.556 0 00.156.495c.11.106.247.164.395.164z"/>
								</svg>
								<?php esc_html_e( 'Edit account', 'smuac' ); ?>
							</button>
						</a>
					</div>
				</div>
				<?php
			}
			?>
			</div>	
		<?php
	}

	// Individual multiaccount endpoint content
	public function smuac_multiaccount_endpoint_content() {
		// get multiaccount
		if ( isset( $_GET['id'] ) ) {
			$multiaccount_id = sanitize_text_field( $_GET['id'] );
		}
		// check if current user has permission to access this multiaccount
		$current_user               = get_current_user_id();
		$current_user_multiaccounts = get_user_meta( $current_user, 'smuac_multiaccounts_list', true );
		$current_user_multiaccounts = array_filter( explode( ',', $current_user_multiaccounts ) );
		if ( in_array( $multiaccount_id, $current_user_multiaccounts ) ) {
			// has permission
			// get multiaccount meta
			$name                        = get_user_meta( $multiaccount_id, 'smuac_account_name', true );
			$job_title                   = get_user_meta( $multiaccount_id, 'smuac_account_job_title', true );
			$phone                       = get_user_meta( $multiaccount_id, 'smuac_account_phone', true );
			$permission_buy              = filter_var( get_user_meta( $multiaccount_id, 'smuac_account_permission_buy', true ), FILTER_VALIDATE_BOOLEAN );
			$permission_view_orders      = filter_var( get_user_meta( $multiaccount_id, 'smuac_account_permission_view_orders', true ), FILTER_VALIDATE_BOOLEAN );
			$permission_view_bundles     = filter_var( get_user_meta( $multiaccount_id, 'smuac_account_permission_view_bundles', true ), FILTER_VALIDATE_BOOLEAN );
			$permission_view_discussions = filter_var( get_user_meta( $multiaccount_id, 'smuac_account_permission_view_discussions', true ), FILTER_VALIDATE_BOOLEAN );
			$permission_view_lists       = filter_var( get_user_meta( $multiaccount_id, 'smuac_account_permission_view_lists', true ), FILTER_VALIDATE_BOOLEAN );
			?>

			<div class="smuac_multiaccounts_edit_account_container">
				<div class="smuac_multiaccounts_new_account_container_top">
					<div class="smuac_multiaccounts_new_account_container_top_title">
						<?php esc_html_e( 'Edit Multiaccount', 'smuac' ); ?>
					</div>
					<div class="smuac_multiaccounts_edit_account_container_top_close">
						<?php esc_html_e( 'Close X', 'smuac' ); ?>
					</div>
				</div>
				<div class="smuac_multiaccounts_new_account_container_content">
					<div class="smuac_multiaccounts_new_account_container_content_large_title smuac_multiaccount_top_margin">
						<svg class="smuac_multiaccounts_new_account_container_content_large_title_icon" xmlns="http://www.w3.org/2000/svg" width="35" height="29" fill="none" viewBox="0 0 35 29">
						  <path fill="#4E4E4E" d="M12.25 14.063c3.867 0 7-3.148 7-7.031 0-3.884-3.133-7.032-7-7.032-3.866 0-7 3.148-7 7.032 0 3.883 3.134 7.031 7 7.031zm4.9 1.758h-.913a9.494 9.494 0 01-3.986.879 9.512 9.512 0 01-3.987-.88H7.35C3.292 15.82 0 19.129 0 23.205v2.285a2.632 2.632 0 002.625 2.637H17.66a2.648 2.648 0 01-.142-1.17l.372-3.346.066-.61.432-.433 4.227-4.247c-1.34-1.521-3.281-2.5-5.463-2.5zm2.478 7.982l-.372 3.35a.873.873 0 00.963.968l3.33-.374 7.542-7.575-3.921-3.94-7.542 7.57zm14.99-9.031l-2.072-2.082a1.306 1.306 0 00-1.849 0l-2.067 2.076-.224.225 3.927 3.94 2.285-2.297a1.327 1.327 0 000-1.862z"/>
						</svg>
						<?php esc_html_e( 'Personal Details', 'smuac' ); ?>
					</div>
					<div class="smuac_multiaccounts_new_account_container_content_element">
						<div class="smuac_multiaccounts_new_account_container_content_element_label">
							<?php esc_html_e( 'Name', 'smuac' ); ?>
						</div>
						<input type="text" class="smuac_multiaccounts_new_account_container_content_element_text" name="smuac_multiaccounts_new_account_name" placeholder="<?php esc_attr_e( 'Enter the account holder\'s name here...', 'smuac' ); ?>" value="<?php echo esc_attr( $name ); ?>">
					</div>
					<div class="smuac_multiaccounts_new_account_container_content_element">
						<div class="smuac_multiaccounts_new_account_container_content_element_label">
							<?php esc_html_e( 'Job Title', 'smuac' ); ?>
						</div>
						<input type="text" class="smuac_multiaccounts_new_account_container_content_element_text" name="smuac_multiaccounts_new_account_job_title" placeholder="<?php esc_attr_e( 'Enter the account holder\'s title here...', 'smuac' ); ?>" value="<?php echo esc_attr( $job_title ); ?>">
					</div>
					<div class="smuac_multiaccounts_new_account_container_content_element smuac_multiaccount_horizontal_line">
						<div class="smuac_multiaccounts_new_account_container_content_element_label">
							<?php esc_html_e( 'Phone Number', 'smuac' ); ?>
						</div>
						<input type="text" class="smuac_multiaccounts_new_account_container_content_element_text" name="smuac_multiaccounts_new_account_phone_number" placeholder="<?php esc_attr_e( 'Enter the account holder\'s phone here...', 'smuac' ); ?>" value="<?php echo esc_attr( $phone ); ?>">
					</div>
					<div class="smuac_multiaccounts_new_account_container_content_large_title smuac_multiaccount_top_margin">
						<svg class="smuac_multiaccounts_new_account_container_content_large_title_icon" xmlns="http://www.w3.org/2000/svg" width="35" height="24" fill="none" viewBox="0 0 35 24">
						  <path fill="#575757" d="M16.042 8.75v2.917h-1.459v2.916h-2.916v-2.916H8.502a4.36 4.36 0 01-4.127 2.916 4.375 4.375 0 110-8.75A4.36 4.36 0 018.502 8.75h7.54zm-11.667 0a1.458 1.458 0 100 2.917 1.458 1.458 0 000-2.917zm18.958 5.833c3.894 0 11.667 1.955 11.667 5.834v2.916H11.667v-2.916c0-3.88 7.773-5.834 11.666-5.834zm0-2.916a5.833 5.833 0 110-11.667 5.833 5.833 0 010 11.667z"/>
						</svg>
						<?php esc_html_e( 'Permissions', 'smuac' ); ?>
					</div>
					<div class="smuac_multiaccounts_new_account_container_content_element_checkbox">
						<div class="smuac_multiaccounts_new_account_container_content_element_checkbox_name">
							<?php esc_html_e( 'Checkout (place order)', 'smuac' ); ?>
						</div>
						<input type="checkbox" value="1" class="smuac_multiaccounts_new_account_container_content_element_checkbox_input" name="smuac_multiaccounts_new_account_container_content_element_checkbox_buy" <?php checked( true, $permission_buy, true ); ?>>
					</div>
					<div class="smuac_multiaccounts_new_account_container_content_element_checkbox">
						<div class="smuac_multiaccounts_new_account_container_content_element_checkbox_name">
							<?php esc_html_e( 'View account orders', 'smuac' ); ?>
						</div>
						<input type="checkbox" value="1" class="smuac_multiaccounts_new_account_container_content_element_checkbox_input" name="smuac_multiaccounts_new_account_container_content_element_checkbox_view_orders" <?php checked( true, $permission_view_orders, true ); ?>>
					</div>
					<?php if ( 1 === intval( get_option( 'smuac_enable_shopping_lists_setting', 1 ) ) ) { ?>
						<div class="smuac_multiaccounts_new_account_container_content_element_checkbox">
							<div class="smuac_multiaccounts_new_account_container_content_element_checkbox_name">
								<?php esc_html_e( 'View account shopping lists', 'smuac' ); ?>
							</div>
							<input type="checkbox" value="1" class="smuac_multiaccounts_new_account_container_content_element_checkbox_input" name="smuac_multiaccounts_new_account_container_content_element_checkbox_view_lists" <?php checked( true, $permission_view_lists, true ); ?>>
						</div>
						<?php
					}

					do_action( 'smuac_after_my_account_permissions' );

					?>
					<div class="smuac_multiaccounts_new_account_container_content_bottom">
						<button class="smuac_multiaccounts_edit_account_container_content_bottom_button_delete" type="button" value="<?php echo esc_attr( $multiaccount_id ); ?>">
							<svg class="smuac_multiaccounts_new_account_container_content_bottom_button_icon" xmlns="http://www.w3.org/2000/svg" width="32" height="33" fill="none" viewBox="0 0 32 33">
							  <path fill="#fff" d="M11 16.572c2.743 0 4.813-2.07 4.813-4.813S13.742 6.946 11 6.946s-4.813 2.07-4.813 4.813 2.07 4.813 4.813 4.813zm1.375 1.303h-2.75A6.883 6.883 0 002.75 24.75v1.375h16.5V24.75a6.883 6.883 0 00-6.875-6.875zm15.528-6.472l-3.153 3.153-3.153-3.153-1.944 1.944 3.151 3.152-3.152 3.152 1.944 1.945 3.153-3.153 3.154 3.154 1.944-1.944-3.153-3.153 3.153-3.153-1.944-1.944z"/>
							</svg>
							<?php esc_html_e( 'Delete multiaccount', 'smuac' ); ?>
						</button>
						<button class="smuac_multiaccounts_edit_account_container_content_bottom_button" type="button" value="<?php echo esc_attr( $multiaccount_id ); ?>">
							<svg class="smuac_multiaccounts_new_account_container_content_bottom_button_icon" xmlns="http://www.w3.org/2000/svg" width="29" height="21" fill="none" viewBox="0 0 29 21">
							  <path fill="#fff" d="M8.626 10.063c2.868 0 5.032-2.163 5.032-5.031S11.494 0 8.626 0 3.594 2.164 3.594 5.032s2.164 5.031 5.032 5.031zm1.437 1.363H7.188C3.225 11.426 0 14.651 0 18.614v1.438h17.252v-1.438c0-3.963-3.225-7.188-7.189-7.188zM26.3 4.658l-6.182 6.17-1.857-1.857-2.033 2.033 3.89 3.887 8.212-8.197-2.03-2.036z"/>
							</svg>
							<?php esc_html_e( 'Update multiaccount', 'smuac' ); ?>
						</button>
					</div>
				</div>
			</div>


			<?php

		} else {
			// no permission
			esc_html_e( 'Multiaccount does not exist!', 'smuac' );
		}

	}

	public function smuac_shopping_lists_endpoint_content() {
		$bulk_order_endpoint_url = wc_get_endpoint_url( 'bulkorder' );
		?>
		<div class="smuac_shopping_list_top_container">
			<div class="smuac_shopping_lists_top_title">
				<?php esc_html_e( 'Shopping Lists', 'smuac' ); ?>
			</div>

		</div>
		<table id="smuac_shopping_lists_table">
				<thead>
					<tr>
						<th><?php esc_html_e( 'List name', 'smuac' ); ?></th>
						<th><?php esc_html_e( 'Number of items', 'smuac' ); ?></th>
						<th><?php esc_html_e( 'User', 'smuac' ); ?></th>
						<th><?php esc_html_e( 'Actions', 'smuac' ); ?></th>

					</tr>
				</thead>
				<tbody>
					<?php
					// get all lists of the user and his multiaccounts
					$current_user       = get_current_user_id();
					$multiaccounts_list = explode( ',', get_user_meta( $current_user, 'smuac_multiaccounts_list', true ) );
					$multiaccounts_list = array_filter( $multiaccounts_list ); // filter blank, null, etc.
					// add current user to multiaccounts to form a complete accounts list
					array_push( $multiaccounts_list, $current_user );

					// if current account is multiaccount AND has permission to view all account shopping lists, add parent account+all multiaccounts lists
					$account_type = get_user_meta( $current_user, 'smuac_account_type', true );
					if ( 'multiaccount' === $account_type ) {
						$permission_view_all_lists = filter_var( get_user_meta( $current_user, 'smuac_account_permission_view_lists', true ), FILTER_VALIDATE_BOOLEAN );
						if ( true === $permission_view_all_lists ) {
							// has permission
							$parent_account            = get_user_meta( $current_user, 'smuac_account_parent', true );
							$parent_multiaccounts_list = explode( ',', get_user_meta( $parent_account, 'smuac_multiaccounts_list', true ) );
							$parent_multiaccounts_list = array_filter( $parent_multiaccounts_list ); // filter blank, null, etc.
							array_push( $parent_multiaccounts_list, $parent_account ); // add parent itself to form complete parent accounts list

							$multiaccounts_list = array_merge( $multiaccounts_list, $parent_multiaccounts_list );
						}
					}

					$shopping_lists = get_posts(
						array(
							'post_type'   => 'smuac_list',
							'post_status' => 'publish',
							'numberposts' => -1,
							'author__in'  => $multiaccounts_list,
						)
					);

					$endpointurl = wc_get_endpoint_url( 'purchase-list' );

					foreach ( $shopping_lists as $list ) {
						$list_details         = get_post_meta( $list->ID, 'smuac_shopping_list_details', true );
						$list_items_array     = explode( '|', $list_details );
						$list_items_array     = array_filter( $list_items_array );
						$items_number         = count( $list_items_array );
						$list_author_id       = get_post_field( 'post_author', $list->ID );
						$list_author_username = get_user_by( 'id', $list_author_id )->user_login;
						?>
						<tr>
							<td><?php echo esc_html( $list->post_title ); ?></td>
							<td>
								<?php
								echo esc_html( $items_number );
								if ( 1 === $items_number ) {
									esc_html_e( ' item', 'smuac' );
								} else {
									esc_html_e( ' items', 'smuac' );
								}
								?>
								
							</td>
							<td><?php echo esc_html( $list_author_username ); ?></td>
							<td>
								<a class="smuac_shopping_list_button_href" href="<?php echo esc_url( add_query_arg( 'id', $list->ID, $endpointurl ) ); ?>">
									<button type="button" class="smuac_shopping_lists_view_list"><?php esc_html_e( 'Open List', 'smuac' ); ?></button>
								</a>
							</td>
						</tr>

						<?php
					}

					?>
									   
				</tbody>
				<tfoot>
					<tr>
						<th><?php esc_html_e( 'List name', 'smuac' ); ?></th>
						<th><?php esc_html_e( 'Number of items', 'smuac' ); ?></th>
						<th><?php esc_html_e( 'User', 'smuac' ); ?></th>
						<th><?php esc_html_e( 'Actions', 'smuac' ); ?></th>
					</tr>
				</tfoot>
			</table>
			<?php
	}

	// Content of individual shopping list in my account (based on bulk order form content)
	public function smuac_shopping_list_endpoint_content() {
		// get list name
		if ( isset( $_GET['id'] ) ) {
			$shopping_list_id = sanitize_text_field( $_GET['id'] );
		} else {
			$shopping_list_id = 0;
		}
		$list_author_id = get_post_field( 'post_author', $shopping_list_id );

		// check permissions
		$current_user       = get_current_user_id();
		$multiaccounts_list = explode( ',', get_user_meta( $current_user, 'smuac_multiaccounts_list', true ) );
		$multiaccounts_list = array_filter( $multiaccounts_list ); // filter blank, null, etc.
		array_push( $multiaccounts_list, $current_user );

		// if current account is multiaccount AND has permission to view all account shopping lists, add parent account + all multiaccounts
		$account_type = get_user_meta( $current_user, 'smuac_account_type', true );
		if ( 'multiaccount' === $account_type ) {
			$permission_view_all_lists = filter_var( get_user_meta( $current_user, 'smuac_account_permission_view_lists', true ), FILTER_VALIDATE_BOOLEAN );
			if ( true === $permission_view_all_lists ) {

				// has permission, add all account orders (parent+parent multiaccount list orders)
				$parent_account            = get_user_meta( $current_user, 'smuac_account_parent', true );
				$parent_multiaccounts_list = explode( ',', get_user_meta( $parent_account, 'smuac_multiaccounts_list', true ) );
				$parent_multiaccounts_list = array_filter( $parent_multiaccounts_list ); // filter blank, null, etc.
				array_push( $parent_multiaccounts_list, $parent_account ); // add parent itself to form complete parent accounts list

				$multiaccounts_list = array_merge( $multiaccounts_list, $parent_multiaccounts_list );
			}
		}

		if ( in_array( $list_author_id, $multiaccounts_list ) ) {
			// has permission to view shopping list
			$list_title   = get_the_title( $shopping_list_id );
			$list_details = get_post_meta( $shopping_list_id, 'smuac_shopping_list_details', true );
			$list_items   = explode( '|', $list_details );
			$list_items   = array_filter( $list_items );
			?>
			<div class="smuac_bulkorder_form_container">
				<div class="smuac_bulkorder_form_container_top">
					<?php echo esc_html( $list_title ); ?>
				</div>
				<div class="smuac_bulkorder_form_container_content">
					<div class="smuac_bulkorder_form_container_content_header">
						<div class="smuac_bulkorder_form_container_content_header_product">
							<?php esc_html_e( 'Product', 'smuac' ); ?>
						</div>
						<div class="smuac_bulkorder_form_container_content_header_qty">
							<?php esc_html_e( 'Qty', 'smuac' ); ?>
						</div>
						<div class="smuac_bulkorder_form_container_content_header_subtotal">
							<?php esc_html_e( 'Subtotal', 'smuac' ); ?>
						</div>
					</div>

					<?php
						$total = 0;

						require_once SMUAC_DIR . 'public/class-smuac-helper.php';
						$helper = new Smuac_Helper();

					foreach ( $list_items as $list_item ) {
						$item        = explode( ':', $list_item );
						$product_id  = $item[0];
						$product_qty = $item[1];
						$productobj  = wc_get_product( $product_id );

						$product_title = $productobj->get_name();
						if ( $productobj->is_on_sale() ) {
							$product_price = $productobj->get_sale_price();
						} else {
							$product_price = $productobj->get_price();
						}

						$product_price = round( floatval( $helper->smuac_wc_get_price_to_display( $productobj, array( 'price' => $product_price ) ) ), 2 );

						$subtotal = $product_qty * $product_price;
						$total   += $subtotal;
						?>
							<input type="hidden" id="smuac_shopping_list_page" value="1">
							<div class="smuac_bulkorder_form_container_content_line"><input type="text" class="smuac_bulkorder_form_container_content_line_product smuac_selected_product_id_<?php echo esc_attr( $product_id ); ?>" placeholder="<?php esc_attr_e( 'Search for an item...', 'smuac' ); ?>" value="<?php echo esc_attr( $product_title ); ?>" disabled><button class="smuac_bulkorder_clear"><?php esc_html_e( 'Clear X', 'smuac' ); ?></button><input type="number" min="0" class="smuac_bulkorder_form_container_content_line_qty" value="<?php echo esc_attr( $product_qty ); ?>"><div class="smuac_bulkorder_form_container_content_line_subtotal"><?php echo esc_html( get_woocommerce_currency_symbol() ) . esc_html( $subtotal ); ?></div><div class="smuac_bulkorder_form_container_content_line_livesearch"></div></div>
							<?php
					}

					?>

					<!-- new line button -->
					<div class="smuac_bulkorder_form_container_newline_container">
						<button class="smuac_bulkorder_form_container_newline_button">
							<svg xmlns="http://www.w3.org/2000/svg" width="16" height="16" fill="none" viewBox="0 0 16 16">
							  <path fill="#eaeaea" fill-rule="evenodd" d="M8.667 8.667v4.666a.667.667 0 11-1.334 0V8.667H2.667a.667.667 0 010-1.334h4.666V2.667a.667.667 0 011.334 0v4.666h4.666a.667.667 0 110 1.334H8.667z" clip-rule="evenodd"/>
							</svg>
							<?php esc_html_e( 'add item', 'smuac' ); ?>
						</button>
					</div>

					<!-- add to cart button -->
					<div class="smuac_bulkorder_form_container_bottom">
						<div class="smuac_bulkorder_form_container_bottom_add">
							<button class="smuac_bulkorder_form_container_bottom_add_button" type="button">
								<svg class="smuac_bulkorder_form_container_bottom_add_button_icon" xmlns="http://www.w3.org/2000/svg" width="23" height="23" fill="none" viewBox="0 0 23 23">
								   <path fill="#fff" d="M9.217 20.633a2.2 2.2 0 11-4.4 0 2.2 2.2 0 014.4 0zm8.8-2.2a2.2 2.2 0 100 4.4 2.2 2.2 0 000-4.4zm-10.78-3.707a.268.268 0 000 .132.275.275 0 00.275.275h12.705v2.2h-13.2a2.2 2.2 0 01-2.2-2.2c0-.385.077-.759.264-1.1l1.496-2.65-3.96-8.35h-2.2v-2.2h3.597l1.034 2.2h16.269a1.1 1.1 0 011.1 1.1c0 .187-.055.374-.132.55l-3.938 7.15a2.226 2.226 0 01-1.925 1.1H8.227l-.99 1.793zm1.98-5.643h4.4v2.2l3.3-3.3-3.3-3.3v2.2h-4.4v2.2z"/>
								</svg>
							<?php esc_html_e( 'Add to Cart', 'smuac' ); ?>
							</button>
							<button class="smuac_bulkorder_form_container_bottom_update_button" type="button" value="<?php echo esc_attr( $shopping_list_id ); ?>">
								<svg class="smuac_bulkorder_form_container_bottom_save_button_icon" xmlns="http://www.w3.org/2000/svg" width="24" height="18" fill="none" viewBox="0 0 24 18">
								 <path fill="#fff" d="M.5.625h15.125v2.75H.5V.625zm0 5.5h15.125v2.75H.5v-2.75zm0 5.5h9.625v2.75H.5v-2.75zm21.036-3.723l-5.912 5.9-1.777-1.775-1.944 1.945 3.72 3.718 7.855-7.842-1.942-1.947z"></path>
								</svg>
							<?php esc_html_e( 'Update list', 'smuac' ); ?>
							</button>
							<button class="smuac_bulkorder_form_container_bottom_delete_button" type="button" value="<?php echo esc_attr( $shopping_list_id ); ?>">
								<svg class="smuac_bulkorder_form_container_bottom_delete_button_icon" xmlns="http://www.w3.org/2000/svg" width="29" height="29" fill="none" viewBox="0 0 29 29">
								  <path fill="#fff" d="M17.4 5.8h4.35c.87 0 1.45.58 1.45 1.45V8.7H4.35V7.25c0-.87.725-1.45 1.45-1.45h4.35c.29-1.595 1.885-2.9 3.625-2.9S17.11 4.205 17.4 5.8zm-5.8 0h4.35c-.29-.87-1.305-1.45-2.175-1.45-.87 0-1.885.58-2.175 1.45zm-5.8 4.35h15.95l-1.305 14.645c0 .725-.725 1.305-1.45 1.305H8.555c-.725 0-1.305-.58-1.45-1.305L5.8 10.15z"/>
								</svg>
							<?php esc_html_e( 'Trash', 'smuac' ); ?>
							</button>
						</div>
						<div class="smuac_bulkorder_form_container_bottom_total">
							<?php esc_html_e( 'Total: ', 'smuac' ); ?><strong><?php echo wp_kses( wc_price( $total ), array() ); ?></strong>
						</div>
					</div>


				</div>
			</div>
			<?php
		} else {
			esc_html_e( 'Shopping list does not exist!', 'smuac' );
		}

	}

	// Add "Save as Shopping List" button to cart
	public function smuac_shopping_list_cart_button() {
		// should never appear to a guest user + check setting
		if ( is_user_logged_in() && ( 1 === intval( get_option( 'smuac_enable_shopping_lists_setting', 1 ) ) ) ) {
			// should not appear if user has a dynamic rule replace prices with quote
			$user_id      = get_current_user_id();
			$account_type = get_user_meta( $user_id, 'smuac_account_type', true );
			if ( 'multiaccount' === $account_type ) {
				// for all intents and purposes set current user as the multiaccount parent
				$parent_user_id = get_user_meta( $user_id, 'smuac_account_parent', true );
				$user_id        = $parent_user_id;
			}
			if ( 'yes' !== get_transient( 'smuac_replace_prices_quote_user_' . $user_id ) ) {
				?>
				<button type="button" class="smuac_add_cart_to_shopping_list_button button">
					<?php esc_html_e( 'Save as shopping list', 'smuac' ); ?>
				</button>
				<?php
			}
		}
	}

	// Add "Placed by" column to orders
	public function smuac_orders_placed_by_column( $columns ) {

		$new_columns = array();
		foreach ( $columns as $key => $name ) {
			$new_columns[ $key ] = $name;
			// add ship-to after order status column
			if ( 'order-number' === $key ) {
				$new_columns['order-placed-by'] = esc_html__( 'Placed by', 'smuac' );
			}
		}
		return $new_columns;
	}

	// Add content to the "Placed by" column
	public function smuac_orders_placed_by_column_content( $order ) {
		$customer_id = $order->get_customer_id();
		$username    = get_user_by( 'id', $customer_id )->user_login;
		echo esc_html( $username );
	}

	// Show user multiaccount orders as well
	public function smuac_add_multiaccounts_orders_to_main_query( $q ) {
		// Set customer orders to Current User + Multiaccounts
		$current_user       = get_current_user_id();
		$multiaccounts_list = explode( ',', get_user_meta( $current_user, 'smuac_multiaccounts_list', true ) );
		$multiaccounts_list = array_filter( $multiaccounts_list ); // filter blank, null, etc.
		// add current user to multiaccounts to form a complete accounts list
		array_push( $multiaccounts_list, $current_user );

		// if current account is multiaccount AND has permission to view all account orders, add parent account+all multiaccounts orders
		$account_type = get_user_meta( $current_user, 'smuac_account_type', true );
		if ( 'multiaccount' === $account_type ) {
			$permission_view_all_orders = filter_var( get_user_meta( $current_user, 'smuac_account_permission_view_orders', true ), FILTER_VALIDATE_BOOLEAN );
			if ( true === $permission_view_all_orders ) {

				// has permission, add all account orders (parent+parent multiaccount list orders)
				$parent_account            = get_user_meta( $current_user, 'smuac_account_parent', true );
				$parent_multiaccounts_list = explode( ',', get_user_meta( $parent_account, 'smuac_multiaccounts_list', true ) );
				$parent_multiaccounts_list = array_filter( $parent_multiaccounts_list ); // filter blank, null, etc.
				array_push( $parent_multiaccounts_list, $parent_account ); // add parent itself to form complete parent accounts list

				$multiaccounts_list = array_merge( $multiaccounts_list, $parent_multiaccounts_list );
			}
		}

		$q['customer'] = $multiaccounts_list;
		return $q;
	}

	// Give user permission to access multiaccount orders
	public function smuac_give_main_account_view_multiaccount_orders_permission( $allcaps, $cap, $args ) {
		if ( isset( $cap[0] ) ) {
			if ( 'view_order' === $cap[0] ) {
				// build list of current user and multiaccounts
				$current_user       = get_current_user_id();
				$multiaccounts_list = explode( ',', get_user_meta( $current_user, 'smuac_multiaccounts_list', true ) );
				$multiaccounts_list = array_filter( $multiaccounts_list ); // filter blank, null, etc.
				array_push( $multiaccounts_list, $current_user );

				// if current account is multiaccount AND has permission to view all account orders, add parent account + all multiaccounts orders
				$account_type = get_user_meta( $current_user, 'smuac_account_type', true );
				if ( 'multiaccount' === $account_type ) {
					$permission_view_all_orders = filter_var( get_user_meta( $current_user, 'smuac_account_permission_view_orders', true ), FILTER_VALIDATE_BOOLEAN );
					if ( true === $permission_view_all_orders ) {

						// has permission, add all account orders (parent+parent multiaccount list orders)
						$parent_account            = get_user_meta( $current_user, 'smuac_account_parent', true );
						$parent_multiaccounts_list = explode( ',', get_user_meta( $parent_account, 'smuac_multiaccounts_list', true ) );
						$parent_multiaccounts_list = array_filter( $parent_multiaccounts_list ); // filter blank, null, etc.
						array_push( $parent_multiaccounts_list, $parent_account ); // add parent itself to form complete parent accounts list

						$multiaccounts_list = array_merge( $multiaccounts_list, $parent_multiaccounts_list );
					}
				}

				// check if the current order is part of the list
				$order_placed_by = wc_get_order( $args[2] )->get_customer_id();
				if ( in_array( $order_placed_by, $multiaccounts_list ) ) {
					// give permission
					$allcaps[ $cap[0] ] = true;
				}
			}
		}
		return ( $allcaps );
	}

	// Give permissions to order again
	public function smuac_multiaccounts_orderagain_cap( $allcaps, $cap, $args ) {
		if ( isset( $cap[0] ) ) {
			if ( 'order_again' === $cap[0] ) {
				// build list of current user and multiaccounts
				$current_user       = get_current_user_id();
				$multiaccounts_list = explode( ',', get_user_meta( $current_user, 'smuac_multiaccounts_list', true ) );
				$multiaccounts_list = array_filter( $multiaccounts_list ); // filter blank, null, etc.
				array_push( $multiaccounts_list, $current_user );

				// if current account is multiaccount AND has permission to view all account orders, add parent account + all multiaccounts orders
				$account_type = get_user_meta( $current_user, 'smuac_account_type', true );
				if ( 'multiaccount' === $account_type ) {
					$permission_view_all_orders = filter_var( get_user_meta( $current_user, 'smuac_account_permission_view_orders', true ), FILTER_VALIDATE_BOOLEAN );
					if ( true === $permission_view_all_orders ) {

						// has permission, add all account orders (parent+parent multiaccount list orders)
						$parent_account            = get_user_meta( $current_user, 'smuac_account_parent', true );
						$parent_multiaccounts_list = explode( ',', get_user_meta( $parent_account, 'smuac_multiaccounts_list', true ) );
						$parent_multiaccounts_list = array_filter( $parent_multiaccounts_list ); // filter blank, null, etc.
						array_push( $parent_multiaccounts_list, $parent_account ); // add parent itself to form complete parent accounts list

						$multiaccounts_list = array_merge( $multiaccounts_list, $parent_multiaccounts_list );
					}
				}

				// check if the current order is part of the list
				$order_placed_by = wc_get_order( $args[2] )->get_customer_id();
				if ( in_array( $order_placed_by, $multiaccounts_list ) ) {
					// give permission
					$allcaps[ $cap[0] ] = true;
				}
			}
		}
		return ( $allcaps );
	}

	public function smuac_multiaccount_order_note( $order_id ) {
		$order        = wc_get_order( $order_id );
		$customer_id  = $order->get_customer_id();
		$account_type = get_user_meta( $customer_id, 'smuac_account_type', true );
		if ( 'multiaccount' === $account_type ) {
			$parent_id    = intval( get_user_meta( $customer_id, 'smuac_account_parent', true ) );
			$parent_user  = new WP_User( $parent_id );
			$parent_login = $parent_user->user_login;

			$note = esc_html__( 'This is an order placed by a multiaccount of the user ', 'smuac' ) . '<a href="' . esc_attr( get_edit_user_link( $parent_id ) ) . '">' . esc_html( $parent_login ) . '</a>';
			$order->add_order_note( $note );
		}
	}


	public function force_permalinks_rewrite() {
		// Trigger post types and endpoints functions
		require_once SMUAC_DIR . 'admin/class-smuac-admin.php';
		$adminobj = new Smuac_Admin();

		$this->smuac_custom_endpoints();

		// Flush rewrite rules
		flush_rewrite_rules();
	}

	public function smuac_custom_color() {
		$color      = get_option( 'smuac_color_setting', '#94678b' );
		$colorhover = get_option( 'smuac_colorhover_setting', '#904d82' );
		?>

		<style type="text/css">
			.smuac_myaccount_individual_bundle_bottom_line_add button.smuac_myaccount_individual_bundle_bottom_line_button, #smuac_myaccount_discussions_container_top button, button.smuac_myaccount_start_discussion_button, .smuac_myaccount_discussion_endpoint_button, button.smuac_bulkorder_form_container_bottom_add_button, button.smuac_multiaccounts_container_top_button, button.smuac_multiaccounts_new_account_container_content_bottom_button, button.smuac_multiaccounts_edit_account_container_content_bottom_button, button#smuac_shopping_list_new_button, button.smuac_shopping_lists_view_list{
				background: <?php echo esc_html( $color ); ?>;
			}

			.smuac_myaccount_individual_bundle_bottom_line_add button:hover.smuac_myaccount_individual_bundle_bottom_line_button, #smuac_myaccount_discussions_container_top button:hover, button:hover.smuac_myaccount_start_discussion_button, .smuac_myaccount_discussion_endpoint_button, button:hover.smuac_bulkorder_form_container_bottom_add_button, button:hover.smuac_multiaccounts_container_top_button, button:hover.smuac_multiaccounts_new_account_container_content_bottom_button, button:hover.smuac_multiaccounts_edit_account_container_content_bottom_button, button:hover#smuac_shopping_list_new_button, button:hover.smuac_shopping_lists_view_list{
				background: <?php echo esc_html( $colorhover ); ?>;
			}
		</style>
		<?php
	}

	public function enqueue_public_resources() {

		// scripts and styles already registered by default
		wp_enqueue_script( 'jquery' );
		// the following 3 scripts enable WooCommerce Country and State selectors
		wp_enqueue_script( 'selectWoo' );
		wp_enqueue_style( 'select2' );
		wp_enqueue_script( 'wc-country-select' );

		wp_enqueue_script( 'smuac_public_script', plugins_url( 'assets/js/public.js', __FILE__ ), $deps = array(), $ver = false, $in_footer = true );
		wp_enqueue_style( 'smuac_main_style', plugins_url( '../includes/assets/css/style.css', __FILE__ ), $deps = array(), $ver = false );

		wp_enqueue_script( 'dataTables', plugins_url( '../includes/assets/lib/dataTables/jquery.dataTables.min.js', __FILE__ ), $deps = array(), $ver = false, $in_footer = true );
		wp_enqueue_style( 'dataTables', plugins_url( '../includes/assets/lib/dataTables/jquery.dataTables.min.css', __FILE__ ), $deps = array(), $ver = false );

		// Get number of allowed countries and pass it to registration public.js
		$countries           = new WC_Countries();
		$countries_allowed   = $countries->get_allowed_countries();
		$number_of_countries = count( $countries_allowed );

		// Send display settings to JS
		$data_to_be_passed = array(
			'security'                                 => wp_create_nonce( 'smuac_security_nonce' ),
			'ajaxurl'                                  => admin_url( 'admin-ajax.php' ),
			'carturl'                                  => wc_get_cart_url(),
			'currency_symbol'                          => get_woocommerce_currency_symbol(),
			'discussionurl'                            => wc_get_account_endpoint_url( 'discussion' ), // discussion endpoint URL, for start discussion redirect
			'multiaccountsurl'                         => wc_get_account_endpoint_url( 'multiaccounts' ),
			'purchaselistsurl'                         => wc_get_account_endpoint_url( 'purchase-lists' ),
			'newMultiaccountUsernameError'             => esc_html__( 'Username must be between 8 and 30 characters. ', 'smuac' ),
			'newMultiaccountEmailError'                => esc_html__( 'Email is invalid. ', 'smuac' ),
			'newMultiaccountPasswordError'             => esc_html__( 'Password must have minimum eight characters, at least one letter and one number. ', 'smuac' ),
			'newMultiaccountAccountError'              => esc_html__( 'Account creation error. Username or Email are already taken. ', 'smuac' ),
			'newMultiaccountMaximumMultiaccountsError' => esc_html__( 'You have reached the maximum number of multiaccounts. ', 'smuac' ),
			'are_you_sure_delete'                      => esc_html__( 'Are you sure you want to delete this multiaccount?', 'smuac' ),
			'are_you_sure_delete_list'                 => esc_html__( 'Are you sure you want to delete this shopping list?', 'smuac' ),
			'no_products_found'                        => esc_html__( 'No products found...', 'smuac' ),
			'save_list_name'                           => esc_html__( 'Name for the new shopping list:', 'smuac' ),
			'list_saved'                               => esc_html__( 'List has been saved', 'smuac' ),
			'quote_request_success'                    => esc_html__( 'Your quote request has been received. We will get back to you as soon as possible.', 'smuac' ),
			'custom_quote_request'                     => esc_html__( 'Custom Quote Request', 'smuac' ),
			'send_quote_request'                       => esc_html__( 'Send custom quote request', 'smuac' ),
			'clearx'                                   => esc_html__( 'Clear X', 'smuac' ),
			'number_of_countries'                      => $number_of_countries,
			'datatables_folder'                        => plugins_url( '../includes/assets/lib/dataTables/i18n/', __FILE__ ),
			'loaderurl'                                => plugins_url( '../includes/assets/images/loader.svg', __FILE__ ),
			'shopping_lists_language_option'           => get_option( 'smuac_shopping_lists_language_setting', 'english' ),
			'accountingsubtotals'                      => get_option( 'smuac_show_accounting_subtotals_setting', 0 ),
			'validating'                               => esc_html__( 'Validating...', 'smuac' ),
			'vatinvalid'                               => esc_html__( 'Invalid VAT. Click to try again', 'smuac' ),
			'vatvalid'                                 => esc_html__( 'VAT Validated Successfully', 'smuac' ),
			'validatevat'                              => esc_html__( 'Validate VAT', 'smuac' ),
			'differentdeliverycountrysetting'          => intval( get_option( 'smuac_vat_exemption_different_country_setting', 0 ) ),
			'colorsetting'                             => get_option( 'smuac_color_setting', '#94678b' ),
			'myaccountloggedin'                        => ( is_account_page() && is_user_logged_in() ),
			'ischeckout'                               => is_checkout(),
			'quote_request_empty_fields'               => esc_html__( 'Please fill all fields to submit the quote request', 'smuac' ),
			'quote_request_invalid_email'              => esc_html__( 'The email address you entered is invalid', 'smuac' ),
			'is_required'                              => esc_html__( 'is required', 'smuac' ),
			'must_select_country'                      => esc_html__( 'You must select a country', 'smuac' ),
			'disable_username_subaccounts'             => apply_filters( 'smuac_disable_username_subaccounts', 1 ),

		);

		wp_localize_script( 'smuac_public_script', 'smuac_display_settings', $data_to_be_passed );
	}

}

