<?php

class Smuac_Admin {


	public function __construct() {

		// Require WooCommerce notification
		add_action( 'admin_notices', array( $this, 'smuac_plugin_dependencies' ) );
		// Onboarding notification
		add_action( 'admin_notices', array( $this, 'smuac_onboarding_notification' ) );
		// Load admin notice resources (enables notification dismissal)
		add_action( 'admin_enqueue_scripts', array( $this, 'load_global_admin_notice_resource' ) );

		add_action(
			'plugins_loaded',
			function () {
				if ( class_exists( 'woocommerce' ) ) {

					/*
					 Custom User Meta */
					// Show the new user meta in New User, User Profile and Edit
					add_action( 'user_new_form', array( $this, 'smuac_show_user_meta_profile' ) );
					add_action( 'show_user_profile', array( $this, 'smuac_show_user_meta_profile' ) );
					add_action( 'edit_user_profile', array( $this, 'smuac_show_user_meta_profile' ) );
					// Save the new user meta (Update or Create)
					add_action( 'personal_options_update', array( $this, 'smuac_save_user_meta_customer_group' ) );
					add_action( 'edit_user_profile_update', array( $this, 'smuac_save_user_meta_customer_group' ) );
					add_action( 'user_register', array( $this, 'smuac_save_user_meta_customer_group' ) );
					// Add columns to Users Table
					add_filter( 'manage_users_columns', array( $this, 'smuac_add_columns_user_table' ) );
					// Retrieve group column content (user meta) in the Users Table
					add_filter( 'manage_users_custom_column', array( $this, 'smuac_retrieve_group_column_contents_users_table' ), 10, 3 );

					/*
					 Order Meta Custom Fields */
					// Add custom registration fields to billing order
					add_filter( 'woocommerce_order_get_formatted_billing_address', array( $this, 'smuac_admin_order_meta_billing' ), 10, 3 );

					/*
					 Load resources */
					// Load global admin styles
					add_action( 'admin_enqueue_scripts', array( $this, 'load_global_admin_resources' ) );
					// Only load scripts and styles in this specific admin page
					add_action( 'admin_enqueue_scripts', array( $this, 'load_admin_resources' ) );

					/*
					 Settings */
					// Registers settings
					add_action( 'admin_init', array( $this, 'smuac_settings_init' ) );
					// Renders settings
					add_action( 'admin_menu', array( $this, 'smuac_settings_page' ) );
				}
			}
		);
	}


	/*
	* Functions dealing with custom user meta data START
	*/
	public function smuac_show_user_meta_profile( $user ) {
		if ( isset( $user->ID ) ) {
			$user_id = $user->ID;
		} else {
			$user_id = 0;
		}
		?>
			<input type="hidden" id="smuac_admin_user_id" value="<?php echo esc_attr( $user_id ); ?>">
			
			<?php
			// Only show B2B Enabled and User customer group if user account is not in approval process
			// Also don't show for multiaccounts

			// check this is not "new panel"
			if ( isset( $user->ID ) ) {
				$account_type = get_user_meta( $user->ID, 'smuac_account_type', true );
				if ( 'multiaccount' === $account_type ) {
					esc_html_e( 'This account is a multiaccount. Its parent account is: ', 'smuac' );
					$parent_account = get_user_meta( $user->ID, 'smuac_account_parent', true );
					$parent_user    = get_user_by( 'id', $parent_account );
					echo esc_html( $parent_user->user_login );
				}

				$user_approved = get_user_meta( $user->ID, 'smuac_account_approved', true );
			}
	}

	public function smuac_save_user_meta_customer_group( $user_id ) {
		if ( isset( $_POST['_inline_edit'] ) ) {
			if ( wp_verify_nonce( sanitize_key( $_POST['_inline_edit'] ), 'inlineeditnonce' ) ) {
				return;
			}
		}

		if ( ! current_user_can( 'edit_user', $user_id ) ) {
			return false;
		}

		// delete all smuac transients
		// Must clear transients and rules cache when user group is changed because now new rules may apply.
		update_option( 'smuac_dynamic_rules_have_changed', 'yes' );

		global $wpdb;
		$plugin_options = $wpdb->get_results( "SELECT option_name FROM $wpdb->options WHERE option_name LIKE '%transient_smuac%'" );
		foreach ( $plugin_options as $option ) {
			delete_option( $option->option_name );
		}

		if ( isset( $_POST['smuac_customergroup'] ) ) {
			$customer_group = sanitize_text_field( $_POST['smuac_customergroup'] );
			if ( 'b2cuser' === $customer_group ) {
				update_user_meta( $user_id, 'smuac_b2buser', 'no' );
				update_user_meta( $user_id, 'smuac_customergroup', 'no' );
			} else {
				update_user_meta( $user_id, 'smuac_customergroup', $customer_group );
				update_user_meta( $user_id, 'smuac_b2buser', 'yes' );
			}
		}

		// remove existing roles of smuac, and add new role
		$groups = get_posts(
			array(
				'post_type'   => 'smuac_group',
				'post_status' => 'publish',
				'numberposts' => -1,
				'fields'      => 'ids',
			)
		);

		$user_obj = new WP_User( $user_id );
		$user_obj->remove_role( 'smuac_role_b2cuser' );
		foreach ( $groups as $group ) {
			$user_obj->remove_role( 'smuac_role_' . $group );
		}

		if ( isset( $_POST['smuac_customergroup'] ) ) {
			$user_obj->add_role( 'smuac_role_' . $customer_group );
		}

		// Save Payment methods and Shipping Methods
		$user_override = filter_input( INPUT_POST, 'smuac_user_shipping_payment_methods_override' );
		if ( null !== $user_override ) {
			update_user_meta( $user_id, 'smuac_user_shipping_payment_methods_override', $user_override );
		}

		$shipping_methods = WC()->shipping->get_shipping_methods();

		foreach ( $shipping_methods as $shipping_method ) {
			$method = sanitize_text_field( filter_input( INPUT_POST, 'smuac_user_shipping_method_' . $shipping_method->id ) );
			if ( null !== $method ) {
				update_user_meta( $user_id, 'smuac_user_shipping_method_' . $shipping_method->id, $method );
			}
		}

		$payment_methods = WC()->payment_gateways->payment_gateways();

		foreach ( $payment_methods as $payment_method ) {
			$method = sanitize_text_field( filter_input( INPUT_POST, 'smuac_user_payment_method_' . $payment_method->id ) );
			if ( null !== $method ) {
				update_user_meta( $user_id, 'smuac_user_payment_method_' . $payment_method->id, $method );
			}
		}
	}
	public function smuac_add_columns_user_table( $columns ) {
		$columns['smuac_customergroup'] = esc_html__( 'Customer Group', 'smuac' );
		$columns['smuac_approval']      = esc_html__( 'Pending Approval', 'smuac' );
		return $columns;
	}
	public function smuac_retrieve_group_column_contents_users_table( $val, $column_name, $user_id ) {
		switch ( $column_name ) {
			case 'smuac_customergroup':
				// first check if multiaccount. If multiaccount, user is equivalent with parent
				$account_type = get_user_meta( $user_id, 'smuac_account_type', true );
				if ( 'multiaccount' === $account_type ) {
					// get parent
					$is_multiaccount   = 'yes';
					$parent_account_id = get_user_meta( $user_id, 'smuac_account_parent', true );
					$user_id           = $parent_account_id;
				} else {
					$is_multiaccount = 'no';
				}

				$user_is_b2b = get_the_author_meta( 'smuac_b2buser', $user_id );
				if ( 'yes' !== $user_is_b2b ) {
					$user_is_b2b = 'no';
				}

				if ( 'yes' === $user_is_b2b ) {
					if ( 'yes' === $is_multiaccount ) {
						return esc_html__( 'Multiaccount of ', 'smuac' ) . esc_html( get_the_title( get_the_author_meta( 'smuac_customergroup', $user_id ) ) );
					} else {
						return esc_html( get_the_title( get_the_author_meta( 'smuac_customergroup', $user_id ) ) );
					}
				} else {
					return esc_html__( 'B2C Users', 'smuac' );
				}
			case 'smuac_approval':
				$account_approved = get_user_meta( $user_id, 'smuac_account_approved', true );
				if ( 'no' === $account_approved ) {
					return '<span class="smuac_users_column_waiting_approval">' . esc_html__( 'Waiting approval', 'smuac' ) . '</span>';
				} else {
					return esc_html__( 'No', 'smuac' );
				}
			default:
		}
		return $val;
	}
	/*
	* Functions dealing with custom user meta data END
	*/

	public function smuac_settings_page() {
		// Admin Menu Settings
		$page_title = 'Subaccounts';
		$menu_title = 'Subaccounts';
		$capability = 'manage_woocommerce';
		$slug       = 'smuac';
		$callback   = array( $this, 'smuac_settings_page_content' );

		$position = 58;
		add_menu_page( $page_title, $menu_title, $capability, $slug, $callback, 'dashicons-id', $position );

		// Add "Settings" submenu page
		add_submenu_page(
			'smuac',
			esc_html__( 'Settings', 'smuac' ), // page title
			esc_html__( 'Settings', 'smuac' ), // menu title
			'manage_woocommerce', // capability,
			'smuac', // menu slug
			'', // callback function
			0
		);

		// Build plugin file path relative to plugins folder
		$absolutefilepath = dirname( plugins_url( '', __FILE__ ) );
		$pluginsurllength = strlen( plugins_url() ) + 1;
		$relativepath     = substr( $absolutefilepath, $pluginsurllength );

		// Add the action links
		add_filter( 'plugin_action_links_' . $relativepath . '/smuac.php', array( $this, 'smuac_action_links' ) );
	}

	public function smuac_action_links( $links ) {
		// Build and escape the URL.
		$url = esc_url( add_query_arg( 'page', 'smuac', get_admin_url() . 'admin.php' ) );

		// Create the link.
		$settings_link = '<a href=' . esc_attr( $url ) . '>' . esc_html__( 'Settings', 'smuac' ) . '</a>';

		// Adds the link to the end of the array.
		array_unshift( $links, $settings_link );
		return $links;
	}


	public function smuac_settings_init() {
		if ( isset( $_POST['_inline_edit'] ) ) {
			if ( wp_verify_nonce( sanitize_key( $_POST['_inline_edit'] ), 'inlineeditnonce' ) ) {
				return;
			}
		}

		require_once SMUAC_DIR . 'admin/class-smuac-settings.php';
		$settings = new Smuac_Settings();
		$settings->register_all_settings();

		// if a POST variable exists indicating the user saved settings, flush permalinks
		if ( isset( $_POST['smuac_plugin_status_setting'] ) ) {
			require_once SMUAC_DIR . 'public/class-smuac-public.php';
			$publicobj = new Smuac_Public();
			$this->smuac_register_post_type_customer_groups();
			if ( 1 === intval( get_option( 'smuac_enable_discussions_setting', 1 ) ) ) {
				$this->smuac_register_post_type_discussion();
			}
			if ( 1 === intval( get_option( 'smuac_enable_bundles_setting', 1 ) ) ) {
				$this->smuac_register_post_type_bundle();
			}
			$this->smuac_register_post_type_dynamic_rules();
			$this->smuac_register_post_type_custom_role();
			$this->smuac_register_post_type_custom_field();
			$publicobj->smuac_custom_endpoints();

			flush_rewrite_rules();
		}
	}

	public function smuac_settings_page_content() {
		require_once SMUAC_DIR . 'admin/class-smuac-settings.php';
		$settings = new Smuac_Settings();
		$settings->render_settings_page_content();
	}

	public function smuac_admin_order_meta_billing( $address, $raw_address, $order ) {
		$custom_fields = get_posts(
			array(
				'post_type'   => 'smuac_custom_field',
				'post_status' => 'publish',
				'numberposts' => -1,
				'meta_key'    => 'smuac_custom_field_sort_number',
				'orderby'     => 'meta_value_num',
				'order'       => 'ASC',
				'meta_query'  => array(
					'relation' => 'AND',
					array(
						'key'   => 'smuac_custom_field_status',
						'value' => 1,
					),
					array(
						'key'   => 'smuac_custom_field_add_to_billing',
						'value' => 1,
					),
				),
			)
		);

		foreach ( $custom_fields as $custom_field ) {
			$label      = get_post_meta( $custom_field->ID, 'smuac_custom_field_field_label', true );
			$fieldvalue = get_post_meta( $order->get_id(), 'smuac_custom_field_' . $custom_field->ID, true );
			if ( ! empty( $fieldvalue ) ) {
				$address .= '<br>' . esc_html( $label ) . ': ' . esc_html( $fieldvalue );
			}
		}

		return $address;
	}

	public function load_global_admin_notice_resource() {
		wp_enqueue_script( 'smuac_global_admin_notice_script', plugins_url( 'assets/js/adminnotice.js', __FILE__ ), $deps = array(), $ver = false, $in_footer = true );

		// Send data to JS
		$data_js = array(
			'security' => wp_create_nonce( 'smuac_notice_security_nonce' ),
		);
		wp_localize_script( 'smuac_global_admin_notice_script', 'smuac_notice', $data_js );
	}

	public function load_global_admin_resources( $hook ) {
		wp_enqueue_style( 'select2', plugins_url( '../includes/assets/lib/select2/select2.min.css', __FILE__ ), array(), 1 );
		wp_enqueue_script( 'select2', plugins_url( '../includes/assets/lib/select2/select2.min.js', __FILE__ ), array( 'jquery' ), 1 );

		wp_enqueue_style( 'smuac_global_admin_style', plugins_url( 'assets/css/adminglobal.css', __FILE__ ), array(), 1 );

		// Enqueue color picker
		wp_enqueue_style( 'wp-color-picker' );
		wp_enqueue_script( 'smuac_global_admin_script', plugins_url( 'assets/js/adminglobal.js', __FILE__ ), $deps = array( 'wp-color-picker' ), $ver = false, $in_footer = true );

		// Send data to JS
		$translation_array = array(
			'admin_url'                          => get_admin_url(),
			'security'                           => wp_create_nonce( 'smuac_security_nonce' ),
			'currency_symbol'                    => get_woocommerce_currency_symbol(),
			'are_you_sure_approve'               => esc_html__( 'Are you sure you want to approve this user? This is irreversible.', 'smuac' ),
			'are_you_sure_reject'                => esc_html__( 'Are you sure you want to REJECT and DELETE this user? This is irreversible.', 'smuac' ),
			'are_you_sure_set_users'             => esc_html__( 'Are you sure you want to move ALL users to this group? This is irreversible.', 'smuac' ),
			'are_you_sure_set_categories'        => esc_html__( 'Are you sure you want to set ALL categories? This is irreversible.', 'smuac' ),
			'are_you_sure_update_user'           => esc_html__( 'Are you sure you want to update this user\'s data?', 'smuac' ),
			'user_has_been_updated'              => esc_html__( 'User data has been updated', 'smuac' ),
			'user_has_been_updated_vat_failed'   => esc_html__( 'VAT VIES validation failed. Please check the VAT number you entered, or disable VIES validation in Subaccounts > Custom Fields > VAT. Other fields have been successfully updated.', 'smuac' ),
			'categories_have_been_set'           => esc_html__( 'All categories have been set', 'smuac' ),
			'feedback_sent'                      => esc_html__( 'Thank you. The feedback was sent successfully.', 'smuac' ),
			'username_already_list'              => esc_html__( 'Username already in the list!', 'smuac' ),
			'add_user'                           => esc_html__( 'Add user', 'smuac' ),
			'cart_total_quantity'                => esc_html__( 'Cart Total Quantity', 'smuac' ),
			'cart_total_value'                   => esc_html__( 'Cart Total Value', 'smuac' ),
			'category_product_quantity'          => esc_html__( 'Category Product Quantity', 'smuac' ),
			'category_product_value'             => esc_html__( 'Category Product Value', 'smuac' ),
			'product_quantity'                   => esc_html__( 'Product Quantity', 'smuac' ),
			'product_value'                      => esc_html__( 'Product Value', 'smuac' ),
			'greater'                            => esc_html__( 'greater (>)', 'smuac' ),
			'equal'                              => esc_html__( 'equal (=)', 'smuac' ),
			'smaller'                            => esc_html__( 'smaller (<)', 'smuac' ),
			'delete'                             => esc_html__( 'Delete', 'smuac' ),
			'enter_quantity_value'               => esc_html__( 'Enter the quantity/value', 'smuac' ),
			'add_condition'                      => esc_html__( 'Add Condition', 'smuac' ),
			'conditions_apply_cumulatively'      => esc_html__( 'Conditions must apply cumulatively.', 'smuac' ),
			'conditions_multiselect'             => esc_html__( 'Each category must meet all category conditions + cart total conditions. Each product must meet all product conditions + cart total conditions.', 'smuac' ),
			'shopping_lists_language_option'     => get_option( 'smuac_shopping_lists_language_setting', 'english' ),
			'replace_product_selector'           => intval( get_option( 'smuac_replace_product_selector_setting', 0 ) ),
			'smuac_customers_panel_ajax_setting' => intval( get_option( 'smuac_customers_panel_ajax_setting', 0 ) ),
			'smuac_plugin_status_setting'        => get_option( 'smuac_plugin_status_setting', 'b2b' ),
			'min_quantity_text'                  => esc_html__( 'Min. Quantity', 'smuac' ),
			'final_price_text'                   => esc_html__( 'Final Price', 'smuac' ),
			'label_text'                         => esc_html__( 'Label', 'smuac' ),
			'text_text'                          => esc_html__( 'Text', 'smuac' ),
			'are_you_sure_set_multiaccounts'         => esc_html__( 'Are you sure you want to set these users as subaccounts of the parent?', 'smuac' ),
			'multiaccounts_have_been_set'            => esc_html__( 'All subaccounts have been set', 'smuac' ),
			'are_you_sure_set_multiaccounts_regular' => esc_html__( 'Are you sure you want to set these users as regular accounts and no longer subaccounts?', 'smuac' ),
			'multiaccounts_have_been_set'            => esc_html__( 'All subaccounts have been set', 'smuac' ),
		);
		if ( isset( $_GET['post'] ) ) {
			$translation_array['current_post_type'] = get_post_type( sanitize_text_field( $_GET['post'] ) );
		}
		if ( isset( $_GET['action'] ) ) {
			$translation_array['current_action'] = sanitize_text_field( $_GET['action'] );
		}

		wp_localize_script( 'smuac_global_admin_script', 'smuac', $translation_array );
	}

	public function load_admin_resources( $hook ) {
		// Load only on this specific plugin admin
		if ( 'toplevel_page_smuac' != $hook ) {
			return;
		}

		wp_enqueue_script( 'jquery' );
		wp_enqueue_style( 'smuac_style', plugins_url( '../includes/assets/css/style.css', __FILE__ ), array(), 1 );
	}


	public function smuac_onboarding_notification() {
		if ( class_exists( 'woocommerce' ) ) {
			// if notice has not already been dismissed once by the current user
			if ( 1 !== intval( get_user_meta( get_current_user_id(), 'smuac_dismiss_onboarding_notice', true ) ) ) {
				?>
				<div class="smuac_dismiss_onboarding_notice notice notice-info is-dismissible">
					<p><?php echo esc_html__( 'Subaccounts & Multi-User Accounts is ready to go! ', 'smuac' ) . esc_html__( 'Go to ', 'smuac' ) . '<a id="smuac_dismiss_onboarding_link" href="' . esc_attr( admin_url( 'admin.php?page=smuac' ) ) . '">' . esc_html__( 'Settings', 'smuac' ) . '</a>' . esc_html__( ' to configure the plugin.' ); ?></p>
				</div>
				<?php
			}
		}
	}


	public function smuac_plugin_dependencies() {
		if ( ! class_exists( 'woocommerce' ) ) {
			// if notice has not already been dismissed once by the current user
			if ( 1 !== intval( get_user_meta( get_current_user_id(), 'smuac_dismiss_activate_woocommerce_notice', true ) ) ) {
				?>
				<div class="smuac_activate_woocommerce_notice notice notice-warning is-dismissible">
					<p><?php esc_html_e( 'Warning: The plugin "B2BWhs" requires WooCommerce to be installed and activated.', 'smuac' ); ?></p>
				</div>
				<?php
			}
		}
	}
}
