<?php
/**
 * Group Memberships
 *
 * @package   rcp-group-accounts
 * @copyright Copyright (c) 2019, Restrict Content Pro
 * @license   GPL2+
 * @since     2.0
 */

global $rcp_options;

$dashboard_page = ! empty( $rcp_options['rcpga_dashboard_page'] ) ? get_permalink( $rcp_options['rcpga_dashboard_page'] ) : false;

$memberships = rcpga_get_group_members( array(
	'user_id'  => get_current_user_id(),
	'role__in' => array( 'admin', 'member', 'invited' )
	// Exclude "owners" because their membership is already displayed in the normal table.
) );

// Bail if this user doesn't have any group memberships.
if ( empty( $memberships ) ) {
	return;
}

$can_remove = ! empty( $rcp_options['groups_allow_member_removal'] );
?>
	<h2 id="rcpga-group-memberships-header">
		<?php echo _n( 'Group Membership', 'Group Memberships', count( $memberships ), 'rcp-group-accounts' ); ?>
	</h2>

	<table id="rcpga-group-memberships-table" class="rcp-table">
		<thead>
		<tr>
			<th><?php _e( 'Group', 'rcp-group-accounts' ); ?></th>
			<th><?php _e( 'Status', 'rcp-group-accounts' ); ?></th>
			<th><?php _e( 'Role', 'rcp-group-accounts' ); ?></th>
			<th><?php _e( 'Date Added', 'rcp-group-accounts' ); ?></th>
			<th><?php _e( 'Actions', 'rcp-group-accounts' ); ?></th>
		</tr>
		</thead>
		<tbody>
		<?php foreach ( $memberships as $group_membership ) :
			/**
			 * @var RCPGA_Group_Member $group_membership
			 */
			?>
			<tr>
				<td data-th="<?php esc_attr_e( 'Group', 'rcp-group-accounts' ); ?>">
					<?php echo esc_html( $group_membership->get_group()->get_name() ); ?>
				</td>

				<td data-th="<?php esc_attr_e( 'Status', 'rcp-group-accounts' ); ?>">
					<?php
					$membership = $group_membership->get_group()->get_membership();
					$status     = ! empty( $membership ) ? rcp_get_status_label( $membership->get_status() ) : '';
					echo $status;
					?>
				</td>

				<td data-th="<?php esc_attr_e( 'Role', 'rcp-group-accounts' ); ?>">
					<?php echo rcpga_get_member_role_label( $group_membership->get_role() ); ?>
				</td>

				<td data-th="<?php esc_attr_e( 'Date Added', 'rcp-group-accounts' ); ?>">
					<?php echo $group_membership->get_date_added( true ); ?>
				</td>

				<td data-th="<?php esc_attr_e( 'Group Actions', 'rcp-group-accounts' ); ?>">
					<?php
					$links = array();

					if ( 'invited' == $group_membership->get_role() ) {
						$links[] = '<a href="' . esc_url( rcpga_generate_group_invite_link_for_user( get_current_user_id(), $group_membership->get_group_id() ) ) . '">' . __( 'Accept Invite', 'rcp-group-accounts' ) . '</a>';
					} elseif ( $group_membership->can( 'manage_members' ) && ! empty( $dashboard_page ) ) {
						$links[] = '<a href="' . esc_url( add_query_arg( 'rcpga-group', urlencode( $group_membership->get_group_id() ), $dashboard_page ) ) . '">' . __( 'Manage Group', 'rcp-group-accounts' ) . '</a>';
					}

					if ( $can_remove ) {
						$leave_url = wp_nonce_url( add_query_arg( array(
							'rcpga-action'    => 'leave-group',
							'group-member-id' => urlencode( $group_membership->get_id() )
						) ), 'rcpga_leave_group' );

						$links[] = '<a href="' . esc_url( $leave_url ) . '" class="rcpga-leave-group">' . __( 'Leave Group', 'rcp-group-accounts' ) . '</a>';
					}

					if ( $links ) {
						echo implode( '<br/>', $links );
					}
					?>
				</td>
			</tr>
		<?php endforeach; ?>
		</tbody>
	</table>

<?php if ( $can_remove ) : ?>
	<script>
		var remove_links = document.getElementsByClassName( "rcpga-leave-group" );

		if ( remove_links ) {

			for ( let i = 0; i < remove_links.length; i++ ) {
				remove_links[ i ].addEventListener( "click", function ( event ) {
					event.preventDefault();

					let message = '<?php esc_attr_e( 'Are you sure you wish to leave this group?', 'rcp-group-accounts' ); ?>';
					let confirmed = confirm( message );

					if ( true === confirmed ) {
						location.assign( event.target.href );
					} else {
						return false;
					}
				} );
			}

		}
	</script>
<?php endif;
