<?php
/**
 * Shortcodes
 *
 * @package   rcp-group-accounts
 * @copyright Copyright (c) 2019, Restrict Content Pro team
 * @license   GPL2+
 * @since     2.0
 */

namespace RCPGA\Shortcodes;

/**
 * Get the group dashboard object
 *
 * @since 2.0
 * @return Dashboard
 */
function get_dashboard() {

	/**
	 * @var Dashboard $rcpga_group_dashboard
	 */
	global $rcpga_group_dashboard;

	if ( is_object( $rcpga_group_dashboard ) ) {
		return $rcpga_group_dashboard;
	}

	$rcpga_group_dashboard = new Dashboard();

	return $rcpga_group_dashboard;

}

/**
 * Display the group dashboard
 *
 * @param array  $atts    Shortcode attributes.
 * @param string $content Shortcode content.
 *
 * @since 2.0
 * @return string
 */
function group_dashboard( $atts, $content = '' ) {

	global $rcp_options, $rcp_load_css;

	$atts = shortcode_atts( array(
		'show_name'                   => true,
		'show_seat_count'             => true,
		'show_members_list'           => true,
		'show_member_add'             => true,
		'show_group_member_join_info' => ! empty( $rcp_options['groups_allow_member_registration'] ),
		'show_csv_import'             => true,
		'show_group_edit'             => true
	), $atts, 'rcp_group_dashboard' );

	/*
	 * Maybe show group creation form.
	 */
	if ( ! empty( $_GET['rcpga-create-group'] ) ) {
		$membership = rcp_get_membership( absint( $_GET['rcpga-create-group'] ) );

		/*
		 * Requirements:
		 *
		 * 		- Current user is the owner of the supplied membership.
		 * 		- Membership is active.
		 * 		- There's no associated group.
		 * 		- The membership level ID is Group Accounts-enabled.
		 */
		if ( get_current_user_id() == $membership->get_customer()->get_user_id() && $membership->is_active() && ! rcpga_get_group_by( 'membership_id', $membership->get_id() ) && rcpga_is_level_group_accounts_enabled( $membership->get_object_id() ) ) {
			$rcp_load_css = true;

			ob_start();
			rcp_get_template_part( 'group', 'create' );
			return ob_get_clean();
		}
	}

	/*
	 * Otherwise show normal group dashboard.
	 */
	$dashboard = get_dashboard();
	$dashboard->set_atts( $atts );

	return $dashboard->get_dashboard();

}

add_shortcode( 'rcp_group_dashboard', __NAMESPACE__ . '\group_dashboard' );

/**
 * Display the group members list
 *
 * @param array  $atts    Shortcode attributes.
 * @param string $content Shortcode content.
 *
 * @since 2.0
 * @return string
 */
function group_members_list( $atts, $content = '' ) {

	$atts = shortcode_atts( array(
		'show_name'         => true,
		'show_seat_count'   => true,
		'show_members_list' => true
	), $atts, 'rcp_group_members_list' );

	$dashboard = get_dashboard();
	$dashboard->set_atts( $atts );

	return $dashboard->get_dashboard();

}

add_shortcode( 'rcp_group_members_list', __NAMESPACE__ . '\group_members_list' );

/**
 * Display the "Add Group Member" form.
 *
 * @param array  $atts    Shortcode attributes.
 * @param string $content Shortcode content.
 *
 * @since 2.0
 * @return string
 */
function group_member_add( $atts, $content = '' ) {

	$atts = shortcode_atts( array(
		'show_name'       => true,
		'show_seat_count' => true,
		'show_member_add' => true,
		'show_csv_import' => false
	), $atts, 'rcp_group_member_add' );

	$dashboard = get_dashboard();
	$dashboard->set_atts( $atts );

	return $dashboard->get_dashboard();

}

add_shortcode( 'rcp_group_member_add', __NAMESPACE__ . '\group_member_add' );

/**
 * Display the "Import Group Member" form.
 *
 * @param array  $atts    Shortcode attributes.
 * @param string $content Shortcode content.
 *
 * @since 2.0
 * @return string
 */
function group_member_import( $atts, $content = '' ) {

	$atts = shortcode_atts( array(
		'show_name'       => true,
		'show_seat_count' => true,
		'show_csv_import' => true
	), $atts, 'rcp_group_member_import' );

	$dashboard = get_dashboard();
	$dashboard->set_atts( $atts );

	return $dashboard->get_dashboard();

}

add_shortcode( 'rcp_group_member_import', __NAMESPACE__ . '\group_member_import' );

/**
 * Display the "Edit Group" form
 *
 * @param array  $atts    Shortcode attributes.
 * @param string $content Shortcode content.
 *
 * @since 2.0
 * @return string
 */
function group_edit( $atts, $content = '' ) {

	$atts = shortcode_atts( array(
		'show_name'       => true,
		'show_seat_count' => false,
		'show_group_edit' => true
	), $atts, 'rcp_group_edit_group' );

	$dashboard = get_dashboard();
	$dashboard->set_atts( $atts );

	return $dashboard->get_dashboard();

}

add_shortcode( 'rcp_group_edit_group', __NAMESPACE__ . '\group_edit' );

/**
 * Displays the "Group Member Join Info". This gives group owners information on how to invite people
 * to join their group.
 *
 * @param array  $atts    Shortcode attributes.
 * @param string $content Shortcode content.
 *
 * @since 2.1
 * @return string
 */
function group_join_info( $atts, $content = '' ) {

	global $rcp_options;

	// Bail if member registration is not enabled.
	if ( empty( $rcp_options['groups_allow_member_registration'] ) ) {
		return '';
	}

	$atts = shortcode_atts( array(
		'show_name'                   => true,
		'show_group_member_join_info' => true
	), $atts, 'rcp_group_member_join_info' );

	$dashboard = get_dashboard();
	$dashboard->set_atts( $atts );

	return $dashboard->get_dashboard();

}

add_shortcode( 'rcp_group_member_join_info', __NAMESPACE__ . '\group_join_info' );

/**
 * Shows content only to group owners
 *
 * @param array  $atts    Shortcode attributes.
 * @param string $content Shortcode content.
 *
 * @since 2.0
 * @return string
 */
function is_group_owner( $atts, $content = '' ) {

	if ( rcpga_user_is_group_member( get_current_user_id(), 'owner' ) ) {
		return do_shortcode( $content );
	}

	return '';

}

add_shortcode( 'rcp_group_is_owner', __NAMESPACE__ . '\is_group_owner' );

/**
 * Shows content only to group admins
 *
 * @param array  $atts    Shortcode attributes.
 * @param string $content Shortcode content.
 *
 * @since 2.0
 * @return string
 */
function is_group_admin( $atts, $content = '' ) {

	if ( rcpga_user_is_group_member( get_current_user_id(), 'admin' ) ) {
		return do_shortcode( $content );
	}

	return '';

}

add_shortcode( 'rcp_group_is_admin', __NAMESPACE__ . '\is_group_admin' );

/**
 * Shows content only to group members, and optionally owners/admins as well
 *
 * @param array  $atts    Shortcode attributes.
 * @param string $content Shortcode content.
 *
 * @since 2.0
 * @return string
 */
function is_group_member( $atts, $content = '' ) {

	$atts = shortcode_atts( array(
		'include_owner' => false,
		'include_admin' => false
	), $atts, 'rcp_group_is_member' );

	$roles = array( 'member' );

	if ( $atts['include_owner'] ) {
		$roles[] = 'owner';
	}

	if ( $atts['include_admin'] ) {
		$roles[] = 'admin';
	}

	if ( rcpga_user_is_group_member( get_current_user_id(), $roles ) ) {
		return do_shortcode( $content );
	}

	return '';

}

add_shortcode( 'rcp_group_is_member', __NAMESPACE__ . '\is_group_member' );

/**
 * Displays the group self-registration form
 *
 * @param array  $atts    Shortcode attributes.
 * @param string $content Shortcode content.
 *
 * @since 2.1
 * @return string
 */
function group_join_form( $atts, $content = '' ) {

	// Bail if self-registration isn't enabled.
	global $rcp_options;
	if ( empty( $rcp_options['groups_allow_member_registration'] ) ) {
		if ( current_user_can( 'manage_options' ) ) {
			return __( 'Group member registration is disabled. Please enable it in Restrict > Settings > Misc.', 'rcp-group-accounts' );
		} else {
			return '';
		}
	}

	global $rcp_load_css;

	$rcp_load_css = true;

	ob_start();
	rcp_get_template_part( 'group', 'join-form' );

	return ob_get_clean();

}

add_shortcode( 'rcp_group_join_form', __NAMESPACE__ . '\group_join_form' );