<?php
/**
 * Group Dashboard Shortcode
 *
 * @package   rcp-group-accounts
 * @copyright Copyright (c) 2019, Restrict Content Pro team
 * @license   GPL2+
 */

namespace RCPGA\Shortcodes;

use RCPGA_Group;
use RCPGA_Group_Member;

/**
 * Class Dashboard
 * @package RCPGA\Shortcodes
 *
 * @since   2.0
 */
class Dashboard {

	/**
	 * Shortcode attributes
	 *
	 * @var array
	 */
	private $atts = array();

	/**
	 * ID of the current user
	 *
	 * @var int
	 */
	private $user_id = 0;

	/**
	 * Groups the current user is able to manage (user must have `owner` or `admin` role).
	 *
	 * @var array Array of RCPGA_Group objects.
	 */
	private $groups = array();

	/**
	 * Object for the group being displayed, or false if none. Will be false if viewing the overview
	 * of groups to select. Will be group object if managing an individual group.
	 *
	 * @var RCPGA_Group|false
	 */
	private $current_group = false;

	/**
	 * Group member object for the current user/group pair.
	 *
	 * @var RCPGA_Group_Member|false
	 */
	private $current_group_member = false;

	/**
	 * Whether or not the overview has been rendered yet.
	 * This prevents it from rendering multiple times on one page.
	 *
	 * @var bool
	 */
	private $overview_rendered = false;

	/**
	 * Error codes that have been displayed. This prevents the same error
	 * from displaying multiple times on the same page.
	 *
	 * @var array
	 */
	private $errors = array();

	/**
	 * Dashboard constructor.
	 *
	 * @param array $atts
	 *
	 * @since 2.0
	 * @return void
	 */
	public function __construct() {

		$this->user_id = get_current_user_id();

		$group_memberships = rcpga_get_group_members( array(
			'number'   => 9999,
			'user_id'  => $this->user_id,
			'role__in' => array( 'owner', 'admin' )
		) );

		if ( ! empty( $group_memberships ) ) {
			foreach ( $group_memberships as $group_membership ) {
				/**
				 * @var RCPGA_Group_Member $group_membership
				 */
				$this->groups[] = rcpga_get_group( $group_membership->get_group_id() );
			}
		}

		$this->set_current_group();

	}

	/**
	 * Set shortcode attributes
	 *
	 * @param array $atts Shortcode attributes.
	 *
	 * @since 2.0
	 * @return void
	 */
	public function set_atts( $atts = array() ) {
		$this->atts = $atts;
	}

	/**
	 * Load RCP core CSS if it's not disabled.
	 *
	 * @since 2.0
	 * @return void
	 */
	protected function load_styles() {

		global $rcp_load_css;

		$rcp_load_css = true;

	}

	/**
	 * Determines whether or not a certain component should be shown.
	 *
	 * @param string $component Name of the component to check.
	 *
	 * @since 2.0
	 * @return bool
	 */
	public function show( $component ) {
		$component = str_replace( 'show_', '', $component );

		return ! empty( $this->atts[ 'show_' . $component ] ) && 'false' !== $this->atts[ 'show_' . $component ];
	}

	/**
	 * Get the currently queried group and the current user's group member record for that group.
	 *
	 * @param RCPGA_Group|false $group Group object to set as the current group. If omitted, this is determined from
	 *                                 the `rcpga-group` query string.
	 *
	 * @access protected
	 * @since  2.0
	 * @return void
	 */
	protected function set_current_group( $group = false ) {

		if ( ! empty( $group ) ) {
			$this->current_group = $group;
		} else {
			$group_id = ! empty( $_GET['rcpga-group'] ) ? absint( $_GET['rcpga-group'] ) : 0;

			if ( empty( $group_id ) ) {
				return;
			}

			$this->current_group = rcpga_get_group( $group_id );
		}

		if ( empty( $this->current_group ) ) {
			return;
		}

		$this->current_group_member = rcpga_get_group_member( $this->user_id, $this->current_group->get_group_id() );

	}

	/**
	 * Get the current group.
	 *
	 * @access public
	 * @since  2.0
	 * @return RCPGA_Group|false
	 */
	public function get_group() {
		return $this->current_group;
	}

	/**
	 * Get the current group member.
	 *
	 * @access public
	 * @since  2.0
	 * @return RCPGA_Group_Member|false
	 */
	public function get_group_member() {
		return $this->current_group_member;
	}

	/**
	 * Get the current group's member list
	 *
	 * This includes search arguments and pagination for table display.
	 *
	 * @since 2.0
	 * @return array
	 */
	public function get_members_list() {

		$current_page = get_query_var( 'paged' ) ? get_query_var( 'paged' ) : 1;
		$per_page     = 20;
		$offset       = $per_page * ( absint( $current_page ) - 1 );

		$args = array(
			'group_id' => $this->get_group()->get_group_id(),
			'number'   => $per_page,
			'offset'   => $offset,
			'orderby'  => 'role',
			'order'    => 'DSEC'
		);

		$search = ! empty( $_GET['rcpga-search'] ) ? rawurldecode( $_GET['rcpga-search'] ) : '';
		$role   = ! empty( $_GET['rcpga-role'] ) ? urldecode( $_GET['rcpga-role'] ) : '';

		if ( ! empty( $search ) ) {
			// First we have to search for user accounts.
			$user_ids = get_users( array(
				'number' => - 1,
				'search' => '*' . $search . '*',
				'fields' => 'ids'
			) );

			// Maybe add results to query args.
			if ( ! empty( $user_ids ) ) {
				$args['user_id__in'] = array_map( 'absint', $user_ids );
			}
		}

		if ( ! empty( $role ) ) {
			$args['role'] = $role;
		}

		return rcpga_get_group_members( $args );

	}

	/**
	 * Get the total number of members in the list
	 *
	 * This includes search arguments.
	 *
	 * @since 2.0
	 * @return int
	 */
	public function get_members_list_count() {

		$args   = array(
			'group_id' => $this->get_group()->get_group_id()
		);
		$search = ! empty( $_GET['rcpga-search'] ) ? rawurldecode( $_GET['rcpga-search'] ) : '';
		$role   = ! empty( $_GET['rcpga-role'] ) ? urldecode( $_GET['rcpga-role'] ) : '';

		if ( ! empty( $search ) ) {
			// First we have to search for user accounts.
			$user_ids = get_users( array(
				'number' => - 1,
				'search' => '*' . $search . '*',
				'fields' => 'ids'
			) );

			// Maybe add results to query args.
			if ( ! empty( $user_ids ) ) {
				$args['user_id__in'] = array_map( 'absint', $user_ids );
			}
		}

		if ( ! empty( $role ) ) {
			$args['role'] = $role;
		}

		return rcpga_count_group_members( $args );

	}

	/**
	 * Get rendered shortcode content.
	 *
	 * @access public
	 * @since  2.0
	 * @return string
	 */
	public function get_dashboard() {

		// Error if user is not logged in.
		if ( ! is_user_logged_in() ) {
			return $this->maybe_get_error_message( 'log_in' );
		}

		$this->load_styles();

		if ( $this->get_group() ) {
			ob_start();
			$this->render_group_dashboard();

			return ob_get_clean();
		}

		if ( empty( $this->groups ) ) {
			return $this->maybe_get_error_message( 'missing_group_membership' );
		}

		// If there's only one group, let's show it immediately.
		if ( 1 === count( $this->groups ) ) {
			$this->set_current_group( $this->groups[0] );

			ob_start();
			$this->render_group_dashboard();

			return ob_get_clean();
		}

		ob_start();
		$this->render_overview();

		return ob_get_clean();

	}

	/**
	 * Render group overview
	 *
	 * This displays a table of all groups the user is able to manage.
	 *
	 * @since 2.0
	 * @return void
	 */
	protected function render_overview() {

		if ( $this->overview_rendered ) {
			return;
		}
		?>
		<table id="rcpga-group-dashboard-selection" class="rcp-table">
			<thead>
			<tr>
				<th><?php _e( 'Group Name', 'rcp-group-accounts' ); ?></th>
				<th><?php _e( 'Members', 'rcp-group-accounts' ); ?></th>
				<th><?php _e( 'Status', 'rcp-group-accounts' ); ?></th>
				<th><?php _e( 'Actions', 'rcp-group-accounts' ); ?></th>
			</tr>
			</thead>
			<tbody>
			<?php foreach ( $this->groups as $group ) :
				/**
				 * @var \RCPGA_Group $group
				 */
				?>
				<tr>
					<td><?php echo esc_html( $group->get_name() ); ?></td>
					<td><?php printf( '%d / %d', $group->get_member_count(), $group->get_seats() ); ?></td>
					<td><?php rcp_print_membership_status( $group->get_membership_id() ); ?></td>
					<td>
						<a href="<?php echo esc_url( add_query_arg( 'rcpga-group', urlencode( $group->get_group_id() ) ) ); ?>"><?php _e( 'Manage Group', 'rcp-group-accounts' ); ?></a>
					</td>
				</tr>
			<?php endforeach; ?>
			</tbody>
		</table>
		<?php
		$this->overview_rendered = true;
	}

	/**
	 * Render the dashboard page for a single group.
	 *
	 * @param RCPGA_Group|false $group Group to display. False to use currently selected group.
	 *
	 * @access public
	 * @since  2.0
	 * @return void
	 */
	protected function render_group_dashboard() {

		if ( ! $this->get_group() ) {
			echo $this->maybe_get_error_message( 'invalid_group' );

			return;
		}

		if ( ! $this->get_group_member() ) {
			echo $this->maybe_get_error_message( 'no_permission' );

			return;
		}

		if ( $this->get_group_member()->get_group_id() != $this->get_group()->get_group_id() || ! $this->get_group_member()->can( 'manage_members' ) ) {
			echo $this->maybe_get_error_message( 'no_permission' );

			return;
		}

		rcp_get_template_part( 'group', 'dashboard' );

	}

	/**
	 * Retrieve the error message for an error code. The message is only returned if it hasn't already
	 * been displayed this page load.
	 *
	 * @param string $code Error code to retrieve the message for.
	 *
	 * @since 2.0
	 * @return string
	 */
	protected function maybe_get_error_message( $code = '' ) {

		// Prevent the same error from being displayed more than once per page load.
		if ( in_array( $code, $this->errors ) ) {
			return '';
		}

		$message = '';

		switch ( $code ) {

			case 'invalid_group';
				$message = __( 'Invalid group.', 'rcp-group-accounts' );
				break;

			case 'log_in' :
				$message = '<p>' . __( 'Please sign in to manage your group.', 'rcp-group-accounts' ) . '</p>' . do_shortcode( '[login_form]' );
				break;

			case 'missing_group_membership' :
				$message = __( 'You must have a membership that allows group accounts to manage a group.', 'rcp-group-accounts' );
				break;

			case 'no_permission' :
				$message = __( 'You do not have permission to manage this group.', 'rcp-group-accounts' );
				break;

		}

		$this->errors[] = $code;

		return $message;

	}

}