<?php
/**
 * Registration Actions
 *
 * @package   rcp-group-accounts
 * @copyright Copyright (c) 2019, Restrict Content Pro team
 * @license   GPL2+
 * @since     2.0
 */

namespace RCPGA;

use RCP_Customer;
use RCP_Membership;

/**
 * Add group creation fields to registration form
 *
 * @since 2.0
 * @return void
 */
function add_registration_fields() {
	rcp_get_template_part( 'group', 'register-seats' );
	rcp_get_template_part( 'group', 'register' );
	wp_enqueue_script( 'rcp-group-accounts' );
}

add_action( 'rcp_after_register_form_fields', __NAMESPACE__ . '\add_registration_fields', 5 );

/**
 * Customers cannot change levels if the level in question has fewer seats available than the total
 * number in their group already.
 *
 * @param array $post Posted form data.
 *
 * @since 2.0
 * @return void
 */
function maybe_prevent_level_change( $post ) {

	$show_error = false;
	$level_id   = absint( $post['rcp_level'] );

	if ( method_exists( rcp_get_registration(), 'get_registration_type' ) ) {

		/**
		 * RCP 3.1+
		 */

		// Bail if this isn't an upgrade/downgrade.
		if ( ! in_array( rcp_get_registration()->get_registration_type(), array( 'upgrade', 'downgrade' ) ) ) {
			return;
		}

		$membership = rcp_get_registration()->get_membership();

		if ( empty( $membership ) ) {
			return;
		}

		// Bail if the membership in question doesn't have a group.
		$group = rcpga_get_group_by( 'membership_id', $membership->get_id() );

		if ( empty( $group ) ) {
			return;
		}

		$member_count = $group->get_member_count();

		// If member count is 1, set it to 0. This allows us to exclude the group owner if they're the only one in the group.
		if ( 1 === $member_count ) {
			$member_count = 0;
		}

		$this_level_max_seats = 0;

		if ( rcpga_is_level_group_accounts_enabled( $level_id ) ) {
			if ( 'per_seat' === rcpga_get_level_seat_pricing_type( $level_id ) ) {
				$this_level_max_seats = ! empty( $_POST['rcpga-group-number-seats'] ) ? absint( $_POST['rcpga-group-number-seats'] ) : 1;
			} else {
				$this_level_max_seats = rcpga_get_level_group_seats_allowed( $level_id );
			}
		}

		if ( $this_level_max_seats < $member_count ) {
			$show_error = true;
		}
	} else {

		/**
		 * RCP 3.0
		 */

		if ( empty( $post['rcp_level'] ) ) {
			return;
		}

		// Bail if this user isn't a group owner.
		if ( ! rcpga_user_is_group_member( get_current_user_id(), 'owner' ) ) {
			return;
		}

		$group = rcpga_get_group_by( 'owner_id', get_current_user_id() );

		if ( empty( $group ) ) {
			return;
		}

		$member_count = $group->get_member_count();

		// If member count is 1, set it to 0. This allows us to exclude the group owner if they're the only one in the group.
		if ( 1 === $member_count ) {
			$member_count = 0;
		}

		if ( rcpga_get_level_group_seats_allowed( absint( $post['rcp_level'] ) ) < $member_count ) {
			$show_error = true;
		}

	}

	if ( $show_error ) {
		rcp_errors()->add( 'remove_children', __( 'You have too many members in your group to change to this level.', 'rcp-group-accounts' ), 'register' );
	}

}

add_action( 'rcp_form_errors', __NAMESPACE__ . '\maybe_prevent_level_change' );

/**
 * Require group name for Group Accounts registrations.
 *
 * @param array $post Posted form data.
 *
 * @since 2.0
 * @return void
 */
function require_group_name( $post ) {

	$show_error = false;

	if ( method_exists( rcp_get_registration(), 'get_registration_type' ) ) {

		/**
		 * RCP 3.1+
		 */

		// Make sure we have a membership level.
		if ( ! $membership_level_id = rcp_get_registration()->get_membership_level_id() ) {
			return;
		}

		// Bail if this level doesn't support group accounts.
		if ( ! rcpga_is_level_group_accounts_enabled( $membership_level_id ) ) {
			return;
		}

		// Bail if this user is already a group owner and multiple memberships is not enabled.
		if ( rcpga_user_is_group_member( get_current_user_id(), 'owner' ) && ! rcp_multiple_memberships_enabled() ) {
			return;
		}

		// Not required on renewals.
		if ( 'renewal' == rcp_get_registration()->get_registration_type() ) {
			return;
		}

		// Not required if upgrading a membership that already has a group.
		$membership = rcp_get_registration()->get_membership();
		$is_change  = in_array( rcp_get_registration()->get_registration_type(), array( 'upgrade', 'downgrade' ) );
		if ( ! empty( $membership ) && $is_change && rcpga_get_group_by( 'membership_id', $membership->get_id() ) ) {
			return;
		}

		$name       = isset( $post['rcpga-group-name'] ) ? trim( $post['rcpga-group-name'] ) : '';
		$show_error = empty( $name );

	} else {

		/**
		 * RCP 3.0
		 */
		if ( ! $membership_id = rcp_get_registration()->get_membership_level_id() ) {
			return;
		}

		// Bail if this user is already a group owner.
		if ( rcpga_user_is_group_member( get_current_user_id(), 'owner' ) ) {
			return;
		}

		// Bail if this level doesn't support group accounts.
		if ( ! rcpga_is_level_group_accounts_enabled( $membership_id ) ) {
			return;
		}

		$name       = isset( $post['rcpga-group-name'] ) ? trim( $post['rcpga-group-name'] ) : '';
		$show_error = empty( $name );

	}

	if ( $show_error ) {
		rcp_errors()->add( 'group_name_required', __( 'Please enter a group name.', 'rcp-group-accounts' ), 'register' );
	}

}

add_action( 'rcp_form_errors', __NAMESPACE__ . '\require_group_name' );

/**
 * Ensure the entered seat number is within the min and max limits.
 *
 * @param array $post Posted form data.
 *
 * @since 2.2
 */
function validate_number_seats( $post ) {

	// Make sure we have a membership level.
	if ( ! $membership_level_id = rcp_get_registration()->get_membership_level_id() ) {
		return;
	}

	// Bail if this level doesn't support group accounts.
	if ( ! rcpga_is_level_group_accounts_enabled( $membership_level_id ) ) {
		return;
	}

	// Bail if this isn't per-seat pricing.
	if ( 'per_seat' === rcpga_get_level_seat_pricing_type( $membership_level_id ) ) {
		return;
	}

	$chosen_seats = ! empty( $post['rcpga-group-number-seats'] ) ? intval( $post['rcpga-group-number-seats'] ) : 0;
	$min_seats    = rcpga_get_level_min_seats( $membership_level_id );
	$max_seats    = rcpga_get_level_max_seats( $membership_level_id );

	if ( $chosen_seats < $min_seats ) {
		rcp_errors()->add( 'group_seat_number_below_min', sprintf( __( 'Your group seat number must be higher than %d.', 'rcp-group-accounts' ), $min_seats ), 'register' );
	}

	if ( $max_seats > 0 && $chosen_seats > $max_seats ) {
		rcp_errors()->add( 'group_seat_number_above_max', sprintf( __( 'Your group seat number must be less than %d.', 'rcp-group-accounts' ), $max_seats ), 'register' );
	}

}

add_action( 'rcp_form_errors', __NAMESPACE__ . '\validate_number_seats' );

/**
 * Create pending group during registration.
 *
 * Group is saved in membership meta until payment is confirmed.
 *
 * @see   activate_membership_group()
 *
 * @param array                $post                Posted data.
 * @param int                  $user_id             ID of the user who's registering.
 * @param float                $price               Price of the subscription.
 * @param int                  $payment_id          ID of the pending payment associated with this registration.
 * @param RCP_Customer|false   $customer            Customer object.
 * @param int                  $membership_id       ID of the new pending membership.
 * @param RCP_Membership|false $previous_membership Previous membership object, or false if none.
 * @param string               $registration_type   Type of registration: 'new', 'renewal', or 'upgrade'.
 *
 * @since 2.0
 * @return void
 */
function create_pending_group( $post, $user_id, $price, $payment_id = 0, $customer = false, $membership_id = 0, $previous_membership = false, $registration_type = 'new' ) {

	if ( ! $membership_level_id = rcp_get_registration()->get_membership_level_id() ) {
		return;
	}

	// If multiple memberships is disabled, then the member must not already be a group owner.
	if ( ! rcp_multiple_memberships_enabled() && rcpga_user_is_group_member( $user_id, 'owner' ) ) {
		return;
	}

	// Make sure this level supports group accounts.
	if ( ! rcpga_is_level_group_accounts_enabled( $membership_level_id ) ) {
		return;
	}

	// Finally, make sure we have a group name.
	if ( empty( $post['rcpga-group-name'] ) ) {
		return;
	}

	if ( 'per_seat' === rcpga_get_level_seat_pricing_type( $membership_level_id ) ) {
		$seats = ! empty( $_POST['rcpga-group-number-seats'] ) ? intval( $_POST['rcpga-group-number-seats'] ) : 1;

		// Check against min/max.
		$min_seats = rcpga_get_level_min_seats( $membership_level_id );
		$max_seats = rcpga_get_level_max_seats( $membership_level_id );

		if ( $seats < $min_seats ) {
			$seats = $min_seats;
		}

		if ( $max_seats > 0 && $seats > $max_seats ) {
			$seats = $max_seats;
		}
	} else {
		$seats = rcpga_get_level_group_seats_allowed( $membership_level_id );

		if ( empty( $eats ) ) {
			$seats = 0;
		}
	}

	$args = array(
		'owner_id'      => absint( $user_id ),
		'membership_id' => absint( $membership_id ),
		'seats'         => absint( $seats ),
		'name'          => wp_unslash( sanitize_text_field( $post['rcpga-group-name'] ) ),
		'description'   => ! empty( $post['rcpga-group-description'] ) ? wp_unslash( wp_filter_post_kses( $post['rcpga-group-description'] ) ) : '',
	);

	rcp_add_membership_meta( $membership_id, 'rcpga_pending_group', $args );

}

add_action( 'rcp_form_processing', __NAMESPACE__ . '\create_pending_group', 10, 8 );

/**
 * Saves pending group seat count changes during renewals and upgrades.
 *
 * Changes are saved in membership meta until payment is confirmed.
 *
 * @see   activate_membership_group()
 *
 * @param array                $post                Posted data.
 * @param int                  $user_id             ID of the user who's registering.
 * @param float                $price               Price of the subscription.
 * @param int                  $payment_id          ID of the pending payment associated with this registration.
 * @param RCP_Customer|false   $customer            Customer object.
 * @param int                  $membership_id       ID of the new pending membership.
 * @param RCP_Membership|false $previous_membership Previous membership object, or false if none.
 * @param string               $registration_type   Type of registration: 'new', 'renewal', or 'upgrade'.
 *
 * @since 2.2
 * @return void
 */
function save_pending_seat_count_change( $post, $user_id, $price, $payment_id = 0, $customer = false, $membership_id = 0, $previous_membership = false, $registration_type = 'new' ) {

	// Prevent values from getting stale.
	rcp_delete_membership_meta( $membership_id, 'rcpga_pending_seat_change' );

	// Bail if this isn't a renewal or change.
	if ( ! in_array( $registration_type, array( 'renewal', 'upgrade', 'downgrade' ) ) ) {
		return;
	}

	if ( ! $membership_level_id = rcp_get_registration()->get_membership_level_id() ) {
		return;
	}

	// Make sure this level supports group accounts and per-seat pricing.
	if ( ! rcpga_is_level_group_accounts_enabled( $membership_level_id ) || 'per_seat' !== rcpga_get_level_seat_pricing_type( $membership_level_id ) ) {
		return;
	}

	// Bail if we don't have a group with this membership.
	$group = false;
	if ( 'renewal' === $registration_type ) {
		$group = rcpga_get_group_by( 'membership_id', $membership_id );
	} elseif ( $previous_membership instanceof RCP_Membership ) {
		$group = rcpga_get_group_by( 'membership_id', $previous_membership->get_id() );
	}
	if ( ! $group instanceof \RCPGA_Group ) {
		return;
	}

	// Bail if the current user isn't the group owner.
	if ( $user_id != $group->get_owner_id() ) {
		return;
	}

	$seats = ! empty( $_POST['rcpga-group-number-seats'] ) ? intval( $_POST['rcpga-group-number-seats'] ) : 0;

	if ( empty( $seats ) ) {
		return;
	}

	rcp_log( sprintf( 'Group Accounts: Setting seat change flag for membership #%d.', $membership_id ) );

	// Check against min/max.
	$min_seats = rcpga_get_level_min_seats( $membership_level_id );
	$max_seats = rcpga_get_level_max_seats( $membership_level_id );

	if ( $seats < $min_seats ) {
		$seats = $min_seats;
	}

	if ( $max_seats > 0 && $seats > $max_seats ) {
		$seats = $max_seats;
	}

	// Bail if the seat count isn't changing.
	if ( $seats == $group->get_seats() ) {
		return;
	}

	rcp_log( sprintf( 'Group Accounts: Pending seat count: %d', $seats ) );

	rcp_update_membership_meta( $membership_id, 'rcpga_pending_seat_change', intval( $seats ) );

}
add_action( 'rcp_form_processing', __NAMESPACE__ . '\save_pending_seat_count_change', 10, 8 );

/**
 * Add group information to Stripe payment intent meta data
 *
 * @param array                       $intent_args Payment intent arguments.
 * @param \RCP_Payment_Gateway_Stripe $stripe_gateway
 *
 * @since 2.0.1
 * @return array
 */
function add_stripe_payment_intent_args( $intent_args, $stripe_gateway ) {

	$group = rcpga_get_group_by( 'membership_id', $stripe_gateway->membership->get_id() );

	if ( $group ) {
		$group_name = $group->get_name();
	} else {
		// Check membership meta.
		$pending_group_args = rcp_get_membership_meta( $stripe_gateway->membership->get_id(), 'rcpga_pending_group', true );
		$group_name         = ! empty( $pending_group_args['name'] ) ? $pending_group_args['name'] : '';
	}

	if ( ! empty( $group_name ) ) {
		$intent_args['metadata']['rcp_group_name'] = esc_html( $group_name );
	}

	return $intent_args;

}

add_filter( 'rcp_stripe_create_payment_intent_args', __NAMESPACE__ . '\add_stripe_payment_intent_args', 10, 2 );

/**
 * Add Membership level price filter takes the registration and uses the supplied filter to manipulate the price
 *
 * @since 2.2
 * @param \RCP_Registration $registration
 */
function add_membership_level_price_filter(\RCP_Registration $registration) {
	add_filter( 'rcp_membership_level_price', __NAMESPACE__ . '\maybe_filter_membership_level_price', 10, 2 );
}

add_action( 'rcp_registration_init', __NAMESPACE__ . '\add_membership_level_price_filter' );

/**
 * Filter to take in the price and registration details and return the evaluated price.
 * @param $price
 * @param $membership_level
 *
 * @since 2.2
 * @return float
 */
function maybe_filter_membership_level_price( $price, $membership_level ) {

	$membership_level_id = $membership_level->get_id();

	if ( ! rcpga_is_level_group_accounts_enabled( $membership_level_id ) ||
	     'per_seat' !== rcpga_get_level_seat_pricing_type( $membership_level_id ) ||
	     ! isset( $_POST['rcpga-group-number-seats'] )
	) {
		return $price;
	}

	$chosen_seats = intval( $_POST['rcpga-group-number-seats'] );
	$min_seats    = rcpga_get_level_min_seats( $membership_level_id );
	$max_seats    = rcpga_get_level_max_seats( $membership_level_id );

	if ( $chosen_seats < $min_seats) {
		$chosen_seats = $min_seats;
	}

	if ( $max_seats > 0 && $chosen_seats > $max_seats ) {
		$chosen_seats = $max_seats;
	}

	$total = $chosen_seats * $price;

	if ( 0 > $total ) {
		$total = 0;
	}

	$total = round( $total, rcp_currency_decimal_filter() );

	return floatval( $total );
}
