<?php
/**
 * Post Access
 *
 * Handles post access permissions for group members.
 *
 * @package   rcp-group-accounts
 * @copyright Copyright (c) 2019, Restrict Content Pro team
 * @license   GPL2+
 * @since     2.0
 */

/**
 * Grants a group member access to view the post if they are in a group where the owner has
 * permission to view that post.
 *
 * @param bool       $can_access Whether or not the user can access the post.
 * @param int        $user_id    The ID of the user being checked.
 * @param int        $post_id    The ID of the post being checked.
 * @param RCP_Member $member     Member object.
 *
 * @since 2.0
 * @return bool
 */
function rcpga_grant_group_members_post_access( $can_access, $user_id, $post_id, $member ) {

	// Don't mess with a good thing.
	if ( $can_access ) {
		return $can_access;
	}

	$group_memberships = rcpga_get_group_members( array(
		'number'   => 9999,
		'user_id'  => $user_id,
		'role__in' => array( 'admin', 'member' )
	) );

	// Bail if this user is not in a group.
	if ( empty( $group_memberships ) ) {
		return $can_access;
	}

	foreach ( $group_memberships as $group_membership ) {
		/**
		 * @var RCPGA_Group_Member $group_membership
		 */

		$group = rcpga_get_group( $group_membership->get_group_id() );

		// Skip "owner" and "invited" roles.
		if ( in_array( $group_membership->get_role(), array( 'owner', 'invited' ) ) ) {
			continue;
		}

		// Check the membership's permission.
		$membership = $group->get_membership();

		if ( ! empty( $membership ) && $membership->can_access( $post_id ) ) {
			$can_access = true;
			break;
		}
	}

	return $can_access;

}

add_filter( 'rcp_member_can_access', 'rcpga_grant_group_members_post_access', 10, 4 );

/**
 * Grants a group member access to view `[restrict]` shortcode content if they are in a group where the owner has
 * permission to view that content.
 *
 * @param bool  $can_access Whether or not the user can view the shortcode content.
 * @param int   $user_id    ID of the user being checked.
 * @param array $atts       Shortcode attributes.
 *
 * @since 2.0
 * @return bool
 */
function rcpga_grant_group_members_shortcode_access( $can_access, $user_id, $atts ) {

	// Don't mess with a good thing.
	if ( $can_access ) {
		return $can_access;
	}

	$group_memberships = rcpga_get_group_members( array(
		'number'   => 9999,
		'user_id'  => $user_id,
		'role__in' => array( 'admin', 'member' )
	) );

	// Bail if this user is not in a group.
	if ( empty( $group_memberships ) ) {
		return $can_access;
	}

	$membership_level_ids = array_map( 'trim', explode( ',', $atts['subscription'] ) );

	foreach ( $group_memberships as $group_membership ) {
		$this_group_access = false;

		/**
		 * @var RCPGA_Group_Member $group_membership
		 */

		$group = rcpga_get_group( $group_membership->get_group_id() );

		// Skip "owner" and "invited" roles.
		if ( in_array( $group_membership->get_role(), array( 'owner', 'invited' ) ) ) {
			continue;
		}

		// Check the membership's permission.
		$membership = $group->get_membership();

		if ( empty( $membership ) ) {
			continue;
		}

		if ( $atts['paid'] ) {
			if ( $membership->is_paid() && $membership->is_active() && $membership->has_access_level( $atts['level'] ) ) {
				$this_group_access = true;
			}
		} elseif ( $membership->has_access_level( $atts['level'] ) ) {
			$this_group_access = true;
		}

		if ( ! empty( $membership_level_ids ) && ! empty( $membership_level_ids[0] ) ) {
			if ( $membership->is_active() && in_array( $membership->get_object_id(), $membership_level_ids ) ) {
				$this_group_access = true;
			} else {
				$this_group_access = false;
			}
		}

		if ( $atts['userlevel'] === 'none' && ! is_user_logged_in() ) {
			$this_group_access = false;
		}
		if ( 'none' != $atts['userlevel'] ) {
			$roles = array_map( 'trim', explode( ',', $atts['userlevel'] ) );

			foreach ( $roles as $role ) {
				if ( current_user_can( strtolower( $role ) ) ) {
					$this_group_access = true;
					break;
				} else {
					$this_group_access = false;
				}
			}
		}

		if ( $this_group_access ) {
			$can_access = true;
			break;
		}
	}

	return $can_access;

}

add_filter( 'rcp_restrict_shortcode_has_access', 'rcpga_grant_group_members_shortcode_access', 10, 3 );

/**
 * Filters whether or not a group member has an active membership.
 *
 * @param bool               $has_active_membership Whether or not the user has an active membership.
 * @param int                $user_id               ID of the user being checked.
 * @param RCP_Customer|false $customer              User's customer record, if any.
 *
 * @since 2.0
 * @return bool
 */
function rcpga_grant_group_members_active_membership( $has_active_membership, $user_id, $customer ) {

	if ( $has_active_membership ) {
		return $has_active_membership;
	}

	if ( empty( $user_id ) ) {
		$user_id = get_current_user_id();
	}

	$group_memberships = rcpga_get_group_members( array(
		'number'   => 9999,
		'user_id'  => $user_id,
		'role__in' => array( 'admin', 'member' )
	) );

	// Bail if this user is not in a group.
	if ( empty( $group_memberships ) ) {
		return $has_active_membership;
	}

	foreach ( $group_memberships as $group_membership ) {
		/**
		 * @var RCPGA_Group_Member $group_membership
		 */

		$group = rcpga_get_group( $group_membership->get_group_id() );

		// Skip "owner" and "invited" roles.
		if ( in_array( $group_membership->get_role(), array( 'owner', 'invited' ) ) ) {
			continue;
		}

		$membership = $group->get_membership();

		if ( ! empty( $membership ) && $membership->is_active() ) {
			$has_active_membership = true;
			break;
		}

	}

	return $has_active_membership;

}

add_filter( 'rcp_user_has_active_membership', 'rcpga_grant_group_members_active_membership', 10, 3 );

/**
 * Filters whether or not a group member has a paid membership.
 *
 * @param bool               $is_paid  Whether or not the user has a paid membership.
 * @param int                $user_id  ID of the user being checked.
 * @param RCP_Customer|false $customer User's customer record, if any.
 *
 * @since 2.0
 * @return bool
 */
function rcpga_grant_group_members_paid_membership( $is_paid, $user_id, $customer ) {

	if ( $is_paid ) {
		return $is_paid;
	}

	if ( empty( $user_id ) ) {
		$user_id = get_current_user_id();
	}

	$group_memberships = rcpga_get_group_members( array(
		'number'   => 9999,
		'user_id'  => $user_id,
		'role__in' => array( 'admin', 'member' )
	) );

	// Bail if this user is not in a group.
	if ( empty( $group_memberships ) ) {
		return $is_paid;
	}

	foreach ( $group_memberships as $group_membership ) {
		/**
		 * @var RCPGA_Group_Member $group_membership
		 */

		$group = rcpga_get_group( $group_membership->get_group_id() );

		// Skip "owner" and "invited" roles.
		if ( in_array( $group_membership->get_role(), array( 'owner', 'invited' ) ) ) {
			continue;
		}

		$membership = $group->get_membership();

		if ( ! empty( $membership ) && $membership->is_paid() && $membership->is_active() ) {
			$is_paid = true;
			break;
		}

	}

	return $is_paid;

}

add_filter( 'rcp_user_has_paid_membership', 'rcpga_grant_group_members_paid_membership', 10, 3 );

/**
 * Filters whether or not a group member has a free membership.
 *
 * @param bool               $is_paid  Whether or not the user has a free membership.
 * @param int                $user_id  ID of the user being checked.
 * @param RCP_Customer|false $customer User's customer record, if any.
 *
 * @since 2.0
 * @return bool
 */
function rcpga_grant_group_members_free_membership( $has_free_membership, $user_id, $customer ) {

	if ( $has_free_membership ) {
		return $has_free_membership;
	}

	if ( empty( $user_id ) ) {
		$user_id = get_current_user_id();
	}

	$group_memberships = rcpga_get_group_members( array(
		'number'   => 9999,
		'user_id'  => $user_id,
		'role__in' => array( 'admin', 'member' )
	) );

	// Bail if this user is not in a group.
	if ( empty( $group_memberships ) ) {
		return $has_free_membership;
	}

	foreach ( $group_memberships as $group_membership ) {
		/**
		 * @var RCPGA_Group_Member $group_membership
		 */

		$group = rcpga_get_group( $group_membership->get_group_id() );

		// Skip "owner" and "invited" roles.
		if ( in_array( $group_membership->get_role(), array( 'owner', 'invited' ) ) ) {
			continue;
		}

		$membership = $group->get_membership();

		if ( ! empty( $membership ) && $membership->is_active() && ! $membership->is_paid() ) {
			$has_free_membership = true;
			break;
		}

	}

	return $has_free_membership;

}

add_filter( 'rcp_user_has_free_membership', 'rcpga_grant_group_members_free_membership', 10, 3 );

/**
 * Filters whether or not a group member has an expired membership.
 *
 * @param bool               $is_paid  Whether or not the user has an expired membership.
 * @param int                $user_id  ID of the user being checked.
 * @param RCP_Customer|false $customer User's customer record, if any.
 *
 * @since 2.0
 * @return bool
 */
function rcpga_grant_group_members_expired_membership( $has_expired_membership, $user_id, $customer ) {

	if ( $has_expired_membership ) {
		return $has_expired_membership;
	}

	if ( empty( $user_id ) ) {
		$user_id = get_current_user_id();
	}

	$group_memberships = rcpga_get_group_members( array(
		'number'   => 9999,
		'user_id'  => $user_id,
		'role__in' => array( 'admin', 'member' )
	) );

	// Bail if this user is not in a group.
	if ( empty( $group_memberships ) ) {
		return $has_expired_membership;
	}

	foreach ( $group_memberships as $group_membership ) {
		/**
		 * @var RCPGA_Group_Member $group_membership
		 */

		$group = rcpga_get_group( $group_membership->get_group_id() );

		// Skip "owner" and "invited" roles.
		if ( in_array( $group_membership->get_role(), array( 'owner', 'invited' ) ) ) {
			continue;
		}

		$membership = $group->get_membership();

		if ( ! empty( $membership ) && $membership->is_expired() ) {
			$has_expired_membership = true;
			break;
		}

	}

	return $has_expired_membership;

}

add_filter( 'rcp_user_has_expired_membership', 'rcpga_grant_group_members_expired_membership', 10, 3 );

/**
 * Allow group members to purchase WooCommerce products if the group owner can.
 *
 * @param bool $can_purchase Whether or not the user is allowed to purchase the product.
 * @param int  $user_id      ID of the user being checked.
 * @param int  $product_id   ID of the product being checked.
 *
 * @since 2.0
 * @return bool
 */
function rcpga_grant_group_members_woocommerce_purchase_access( $can_purchase, $user_id, $product_id ) {

	if ( $can_purchase ) {
		return $can_purchase;
	}

	$group_memberships = rcpga_get_group_members( array(
		'number'   => 9999,
		'user_id'  => $user_id,
		'role__in' => array( 'admin', 'member' )
	) );

	// Bail if this user is not in a group.
	if ( empty( $group_memberships ) ) {
		return $can_purchase;
	}

	$active_only  = get_post_meta( $product_id, '_rcp_woo_active_to_purchase', true );
	$levels       = (array) get_post_meta( $product_id, '_rcp_woo_subscription_levels_to_purchase', true );
	$access_level = get_post_meta( $product_id, '_rcp_woo_access_level_to_purchase', true );

	foreach ( $group_memberships as $group_membership ) {
		/**
		 * @var RCPGA_Group_Member $group_membership
		 */

		$group = rcpga_get_group( $group_membership->get_group_id() );

		// Skip "owner" and "invited" roles.
		if ( in_array( $group_membership->get_role(), array( 'owner', 'invited' ) ) ) {
			continue;
		}

		$membership = $group->get_membership();

		if ( empty( $membership ) ) {
			continue;
		}

		// At this point assume access.
		$has_access = true;

		// Check active membership.
		if ( $active_only && ! $membership->is_active() ) {
			$has_access = false;
		}

		// Check specific level requirements.
		if ( is_array( $levels ) && ! empty( $levels[0] ) && ! in_array( $membership->get_object_id(), $levels ) ) {
			$has_access = false;
		}

		// Check access level.
		if ( $access_level && ! $membership->has_access_level( $access_level ) ) {
			$has_access = false;
		}

		// If it's still true, let's grant access!
		if ( $has_access ) {
			$can_purchase = true;
			break;
		}

	}

	return $can_purchase;

}

add_filter( 'rcp_user_can_purchase_woocommerce_product', 'rcpga_grant_group_members_woocommerce_purchase_access', 10, 3 );

/**
 * Allow group members to view WooCommerce products if the group owner can.
 *
 * @param bool $can_view   Whether or not the user is allowed to view the product.
 * @param int  $user_id    ID of the user being checked.
 * @param int  $product_id ID of the product being checked.
 *
 * @since 2.0
 * @return bool
 */
function rcpga_grant_group_members_woocommerce_viewing_access( $can_view, $user_id, $product_id ) {

	if ( $can_view ) {
		return $can_view;
	}

	$group_memberships = rcpga_get_group_members( array(
		'number'   => 9999,
		'user_id'  => $user_id,
		'role__in' => array( 'admin', 'member' )
	) );

	// Bail if this user is not in a group.
	if ( empty( $group_memberships ) ) {
		return $can_view;
	}

	$active_only  = get_post_meta( $product_id, '_rcp_woo_active_to_view', true );
	$levels       = (array) get_post_meta( $product_id, '_rcp_woo_subscription_levels_to_view', true );
	$access_level = get_post_meta( $product_id, '_rcp_woo_access_level_to_view', true );

	foreach ( $group_memberships as $group_membership ) {
		/**
		 * @var RCPGA_Group_Member $group_membership
		 */

		$group = rcpga_get_group( $group_membership->get_group_id() );

		// Skip "owner" and "invited" roles.
		if ( in_array( $group_membership->get_role(), array( 'owner', 'invited' ) ) ) {
			continue;
		}

		$membership = $group->get_membership();

		if ( empty( $membership ) ) {
			continue;
		}

		// At this point assume access.
		$has_access = true;

		// Check active membership.
		if ( $active_only && ! $membership->is_active() ) {
			$has_access = false;
		}

		// Check specific level requirements.
		if ( is_array( $levels ) && ! empty( $levels[0] ) && ! in_array( $membership->get_object_id(), $levels ) ) {
			$has_access = false;
		}

		// Check access level.
		if ( $access_level && ! $membership->has_access_level( $access_level ) ) {
			$has_access = false;
		}

		/**
		 * @todo We really need to check taxonomy restrictions as well, but that's trickier because we'll need adjustments in RCP core to handle it.
		 * @see  rcp_is_post_taxonomy_restricted()
		 * @see  rcp_is_post_taxonomy_restricted()
		 */

		// If it's still true, let's grant access!
		if ( $has_access ) {
			$can_view = true;
			break;
		}

	}

	return $can_view;

}

add_filter( 'rcp_user_can_view_woocommerce_product', 'rcpga_grant_group_members_woocommerce_viewing_access', 10, 3 );