<?php
/**
 * Membership Level Functions
 *
 * @package   rcp-group-accounts
 * @copyright Copyright (c) 2019, Restrict Content Pro team
 * @license   GPL2+
 * @since     2.0
 */

/**
 * Removes the group seats metadata for the specified membership level.
 *
 * @param int $level_id The membership level ID.
 *
 * @return void
 */
function rcpga_remove_level_seat_count( $level_id ) {

	if ( function_exists( 'rcp_delete_membership_level_meta' ) ) {
		rcp_delete_membership_level_meta( $level_id, 'group_seats_allowed' );
	} else {
		$levels = new RCP_Levels;
		$levels->delete_meta( $level_id, 'group_seats_allowed' );
	}

	do_action( 'rcpga_remove_level_seat_count', $level_id );

}

add_action( 'rcp_remove_level', 'rcpga_remove_level_seat_count' );

/**
 * Gets the seat pricing type - either `fixed` or `per_seat`.
 *
 * `fixed` means the customer pays a fixed price for a fixed number of seats set by the admin.
 * `per_seat` means the customer can choose exactly how many seats they want to purchase and are charged per-seat.
 *
 * @param int $level_id
 *
 * @since 2.2
 * @return string|false
 */
function rcpga_get_level_seat_pricing_type( $level_id ) {

	$type = rcp_get_membership_level_meta( $level_id, 'group_seat_pricing_type', true );

	/*
	 * Seat type was introduced in version 2.2. If there's no type saved, then let's see if a seat count
	 * is saved. If so, we'll assume `fixed`.
	 */
	if ( empty( $type ) && rcpga_is_level_group_accounts_enabled( $level_id ) && rcpga_get_level_group_seats_allowed( $level_id ) > 0 ) {
		$type = 'fixed';

		rcp_update_membership_level_meta( $level_id, 'group_seat_pricing_type', $type );
	}

	return $type;
}

/**
 * Gets the number of seats allowed for the specified membership level.
 *
 * @param int $level_id The membership level ID.
 *
 * @return integer The number of seats allowed for the specified membership level.
 */
function rcpga_get_level_group_seats_allowed( $level_id ) {

	if ( function_exists( 'rcp_get_membership_level_meta' ) ) {
		$count = rcp_get_membership_level_meta( $level_id, 'group_seats_allowed', true );
	} else {
		$levels = new RCP_Levels;
		$count  = $levels->get_meta( $level_id, 'group_seats_allowed', true );
	}

	return apply_filters( 'rcpga_get_level_group_seats_allowed', absint( $count ), $level_id );

}

/**
 * Gets the minimum number of seats allowed to be purchased for the specified membership level.
 *
 * @param int $level_id
 *
 * @since 2.2
 * @return int Minimum number of seats allowed to be purchased.
 */
function rcpga_get_level_min_seats( $level_id ) {

	if ( function_exists( 'rcp_get_membership_level_meta' ) ) {
		$minimum = rcp_get_membership_level_meta( $level_id, 'group_seats_min', true );
	} else {
		$levels  = new RCP_Levels;
		$minimum = $levels->get_meta( $level_id, 'group_seats_min', true );
	}

	$minimum = (int) $minimum;

	if ( $minimum < 1 ) {
		$minimum = 1; // Can't go below 1.
	}

	return $minimum;

}

/**
 * Gets the maximum number of seats allowed to be purchased for the specified membership level.
 * A max of `0` indicates there is no maximum.
 *
 * @param int $level_id
 *
 * @since 2.2
 * @return int Maximum number of seats allowed to be purchased.
 */
function rcpga_get_level_max_seats( $level_id ) {

	if ( function_exists( 'rcp_get_membership_level_meta' ) ) {
		$max = rcp_get_membership_level_meta( $level_id, 'group_seats_max', true );
	} else {
		$levels = new RCP_Levels;
		$max    = $levels->get_meta( $level_id, 'group_seats_max', true );
	}

	$max = empty( $max ) ? 0 : (int) $max;

	if ( $max < 0 ) {
		$max = 0; // Can't go below 0.
	}

	return $max;

}

/**
 * Check if group accounts are disabled for this level.
 * Note, this does not check the number of seats for this level, which could be 0.
 *
 * @param int $level_id Membership level ID.
 *
 * @return bool True if the level is group enabled, false if not.
 */
function rcpga_is_level_group_accounts_enabled( $level_id ) {

	if ( function_exists( 'rcp_get_membership_level_meta' ) ) {
		$enabled = (bool) rcp_get_membership_level_meta( $level_id, 'group_seats_enabled', true );
	} else {
		$levels  = new RCP_Levels;
		$enabled = (bool) $levels->get_meta( $level_id, 'group_seats_enabled', true );
	}

	return apply_filters( 'rcpga_is_level_group_accounts_enabled', $enabled, $level_id );
}

/**
 * Sets the number of group seats allowed for the specified membership level.
 *
 * @param int $level_id      The membership level ID.
 * @param int $seats_allowed The number of seats allowed.
 *
 * @return void
 */
function rcpga_set_level_group_seats_allowed( $level_id, $seats_allowed ) {

	if ( function_exists( 'rcp_update_membership_level_meta' ) ) {
		rcp_update_membership_level_meta( $level_id, 'group_seats_allowed', absint( $seats_allowed ) );
	} else {
		$levels = new RCP_Levels;
		$levels->update_meta( $level_id, 'group_seats_allowed', absint( $seats_allowed ) );
	}

	do_action( 'rcpga_set_level_group_seats_allowed', $level_id, absint( $seats_allowed ) );

}

/**
 * Disable group accounts for this level
 *
 * @param int $level_id Membership level ID.
 *
 * @return void
 */
function rcpga_disable_level_group_accounts( $level_id ) {

	if ( function_exists( 'rcp_delete_membership_level_meta' ) ) {
		rcp_delete_membership_level_meta( $level_id, 'group_seats_enabled' );
	} else {
		$levels = new RCP_Levels;
		$levels->delete_meta( $level_id, 'group_seats_enabled' );
	}

	do_action( 'rcpga_disable_level_group_accounts', $level_id );

}

/**
 * Disable group accounts for this level
 *
 * @param int $level_id Membership level ID.
 *
 * @return void
 */
function rcpga_enable_level_group_accounts( $level_id ) {

	if ( function_exists( 'rcp_update_membership_level_meta' ) ) {
		rcp_update_membership_level_meta( $level_id, 'group_seats_enabled', true );
	} else {
		$levels = new RCP_Levels;
		$levels->update_meta( $level_id, 'group_seats_enabled', true );
	}

	do_action( 'rcpga_enable_level_group_accounts', $level_id );

}

/**
 * Return all levels that are enabled for group accounts
 *
 * @return array
 */
function rcpga_get_group_enabled_levels() {

	global $wpdb;

	$enabled_levels = $wpdb->get_results( $wpdb->prepare( "SELECT level_id FROM {$wpdb->levelmeta} WHERE meta_key = %s", 'group_seats_enabled' ), ARRAY_A );

	return apply_filters( 'rcpga_get_group_enabled_levels', wp_list_pluck( $enabled_levels, 'level_id' ) );
}

/**
 * Fetches the IDs of all membership levels that are enabled for Group Accounts and are set up to use
 * per-seat pricing.
 *
 * @since 2.2
 * @return array
 */
function rcpga_get_group_enabled_per_seat_levels() {

	global $wpdb;

	$enabled_levels        = rcpga_get_group_enabled_levels();
	$enabled_levels_string = implode( ', ', array_map( 'absint', $enabled_levels ) );

	$per_seat_levels = $wpdb->get_results( $wpdb->prepare(
		"SELECT level_id FROM {$wpdb->levelmeta} WHERE meta_key = %s AND meta_value = %s AND level_id IN({$enabled_levels_string})",
		'group_seat_pricing_type',
		'per_seat'
	), ARRAY_A );

	$level_data = array();
	$level_ids  = wp_list_pluck( $per_seat_levels, 'level_id' );

	if ( ! empty( $level_ids ) ) {
		foreach ( $level_ids as $level_id ) {
			$per_seat_price = rcp_get_subscription_price( absint( $level_id ) );
			$min_seats      = rcpga_get_level_min_seats( absint( $level_id ) );

			$level_data[ absint( $level_id ) ] = array(
				'perSeatPrice'          => $per_seat_price,
				'perSeatPriceFormatted' => rcp_currency_filter( $per_seat_price ),
				'minSeats'              => $min_seats,
				'maxSeats'              => rcpga_get_level_max_seats( absint( $level_id ) ),
				'minPrice'              => rcp_currency_filter( $per_seat_price * $min_seats )
			);
		}
	}

	return $level_data;

}