<?php
/**
 * Membership Actions
 *
 * @package   rcp-group-accounts
 * @copyright Copyright (c) 2019, Restrict Content Pro team
 * @license   GPL2+
 * @since     2.0
 */

namespace RCPGA;

use RCP_Customer;
use RCP_Membership;
use RCPGA_Group;
use RCPGA_Group_Member;

/**
 * Create a group for the membership when it's activated.
 *
 * @param int            $membership_id ID of the membership.
 * @param RCP_Membership $membership    Membership object.
 *
 * @since 2.0
 * @return void
 */
function activate_membership_group( $membership_id, $membership ) {

	$group_details = rcp_get_membership_meta( $membership_id, 'rcpga_pending_group', true );

	if ( empty( $group_details ) || ! is_array( $group_details ) ) {
		return;
	}

	// Make sure the pending meta has all the required arguments.
	$required = array( 'owner_id', 'membership_id', 'seats', 'name', 'description' );
	if ( count( array_intersect_key( array_flip( $required ), $group_details ) ) !== count( $required ) ) {
		return;
	}

	$group_id = rcpga_add_group( $group_details );

	if ( false === $group_id || is_wp_error( $group_id ) ) {
		return;
	}

	rcp_delete_membership_meta( $membership_id, 'rcpga_pending_group' );

	$membership->add_note( sprintf( __( 'Created associated group "%s" (ID #%d).', 'rcp-group-accounts' ), $group_details['name'], $group_id ) );

}

add_action( 'rcp_membership_post_activate', __NAMESPACE__ . '\activate_membership_group', 100, 2 );

/**
 * Change the seat count and membership ID associated with the group when the owner's membership is upgraded.
 *
 * @todo  Per-seat pricing.
 *
 * @param int            $membership_id ID of the membership.
 * @param RCP_Membership $membership    Membership object.
 *
 * @since 2.0
 * @return void
 */
function change_group_membership_id_on_upgrade( $membership_id, $membership ) {

	$user_id       = $membership->get_customer()->get_user_id();
	$upgraded_from = $membership->get_upgraded_from();

	// Bail if this wasn't an upgrade.
	if ( empty( $upgraded_from ) ) {
		return;
	}

	$group = rcpga_get_group_by( 'membership_id', $upgraded_from );

	// Bail if no group.
	if ( empty( $group ) ) {
		return;
	}

	// Bail if this user isn't the owner of the group for some reason... This should never happen.
	if ( $group->get_owner_id() != $user_id ) {
		return;
	}

	// Delete group if new membership isn't group-enabled.
	if ( ! rcpga_is_level_group_accounts_enabled( $membership->get_object_id() ) ) {
		rcp_log( sprintf( 'Group Accounts: Deleting group #%d as new membership #%d is not group-enabled.', $group->get_group_id(), $membership_id ) );
		$group->delete();
	}

	if ( 'fixed' === rcpga_get_level_seat_pricing_type( $membership->get_object_id() ) ) {
		$seats = rcpga_get_level_group_seats_allowed( $membership->get_object_id() );
	} else {
		$seats = rcp_get_membership_meta( $membership_id, 'rcpga_pending_seat_change', true );

		if ( ! empty( $seats ) && is_numeric( $seats ) ) {
			rcp_delete_membership_meta( $membership_id, 'rcpga_pending_seat_change' );
		} else {
			// No change.
			$seats = $group->get_seats();
		}
	}

	// Change number of seats and associated membership ID.
	rcp_log( sprintf( 'Group Accounts: Changing group #%d membership ID from #%d to #%d. Updating seat count to %d.', $group->get_group_id(), $group->get_membership_id(), $membership_id, $seats ) );

	$group->update( array(
		'membership_id' => absint( $membership_id ),
		'seats'         => absint( $seats )
	) );

}

add_action( 'rcp_membership_post_activate', __NAMESPACE__ . '\change_group_membership_id_on_upgrade', 10, 2 );

/**
 * If the seat count has changed during a renewal, update for the group.
 *
 * @param string          $new_expiration
 * @param int             $membership_id
 * @param \RCP_Membership $membership
 *
 * @since 2.2
 * @return void
 */
function maybe_change_group_seat_count( $new_expiration, $membership_id, $membership ) {
	
	// Bail if we don't have the meta.
	$new_seat_count = rcp_get_membership_meta( $membership_id, 'rcpga_pending_seat_change', true );

	if ( empty( $new_seat_count ) ) {
		return;
	}

	$group = rcpga_get_group_by( 'membership_id', $membership_id );

	// Bail if no group.
	if ( empty( $group ) ) {
		return;
	}

	// Bail if this user isn't the owner of the group for some reason... This should never happen.
	if ( $group->get_owner_id() != $membership->get_user_id() ) {
		return;
	}

	// Bail if there isn't per-seat pricing on this membership level.
	if ( 'per_seat' !== rcpga_get_level_seat_pricing_type( $membership->get_object_id() ) ) {
		return;
	}

	rcp_log( sprintf( 'Group Accounts: Changing seats on group #%d post membership renewal. Old count: %d; New count: %d', $group->get_group_id(), $group->get_seats(), $new_seat_count ) );

	// Change the seats!
	if ( $new_seat_count != $group->get_seats() ) {
		$group->update( array(
			'seats' => $new_seat_count
		) );
	}

	rcp_delete_membership_meta( $membership_id, 'rcpga_pending_seat_change' );

}

add_action( 'rcp_membership_post_renew', __NAMESPACE__ . '\maybe_change_group_seat_count', 10, 3 );

/**
 * When a group member signs up for their own membership, remove them from their group.
 *
 * Note: the priority on this needs to be less than 10 in order to ensure the customer
 * is removed from the group before the activation email gets sent out.
 * @link  https://github.com/restrictcontentpro/rcp-group-accounts/issues/297
 *
 * @param int            $membership_id ID of the membership.
 * @param RCP_Membership $membership    Membership object.
 *
 * @since 2.0
 * @return void
 */
function remove_group_member_on_membership_activate( $membership_id, $membership ) {

	// Does not apply if multiple memberships is enabled.
	if ( rcp_multiple_memberships_enabled() ) {
		return;
	}

	$user_id = $membership->get_customer()->get_user_id();

	$group_memberships = rcpga_get_group_members( array(
		'user_id' => $user_id
	) );

	// This user isn't a group member at all - bail.
	if ( empty( $group_memberships ) ) {
		return;
	}

	foreach ( $group_memberships as $group_membership ) {
		/**
		 * @var RCPGA_Group_Member $group_membership
		 */
		// Skip if this user is the owner of the group.
		if ( 'owner' === $group_membership->get_role() ) {
			continue;
		}

		// Otherwise remove them from the group.
		rcp_log( sprintf( 'Group Accounts: Removing user #%d from group #%d, as they now have their own membership (ID #%d ).', $user_id, $group_membership->get_group_id(), $membership_id ) );

		$group_membership->remove();
	}

}

add_action( 'rcp_membership_post_activate', __NAMESPACE__ . '\remove_group_member_on_membership_activate', 5, 2 );

/**
 * When a membership is deleted, delete the corresponding group.
 *
 * @param int            $membership_id
 * @param RCP_Membership $membership
 *
 * @since 2.1
 * @return void
 */
function delete_group_on_membership_delete( $membership_id, $membership ) {
	
	$delete = false;

	/*
	 * `rcp_membership_post_disable` runs in a variety of circumstances. We need to make sure we only
	 * delete the group if an admin explicitly clicked a `Delete` button.
	 */

	if ( ! empty( $_POST['rcp_delete_membership'] ) ) {
		$delete = true;
	}

	// Check bulk action.
	if (
		! $delete &&
		isset( $_REQUEST['_wpnonce'] ) &&
		wp_verify_nonce( $_REQUEST['_wpnonce'], 'bulk-memberships' ) &&
		isset( $_REQUEST['action'] ) &&
		'delete' === $_REQUEST['action']
	) {
		$delete = true;
	}

	// Bail if we're not deleting.
	if ( empty( $delete ) ) {
		return;
	}

	$group = rcpga_get_group_by( 'membership_id', $membership->get_id() );
	if ( ! $group instanceof RCPGA_Group ) {
		return;
	}

	rcp_log( sprintf( 'Group Accounts: Membership #%d has been deleted. Deleting corresponding group #%d.', $membership->get_id(), $group->get_group_id() ), true );

	$group->delete();

}

add_action( 'rcp_membership_post_disable', __NAMESPACE__ . '\delete_group_on_membership_delete', 10, 2 );