<?php
/**
 * Group Member Registration Actions
 *
 * @package   rcp-group-accounts
 * @copyright Copyright (c) 2020, Sandhills Development, LLC
 * @license   GPL2+
 * @since     2.1
 */

namespace RCPGA\Member_Registration;

/**
 * Process group member self-registration.
 *
 * @since 2.1
 */
function process_registration() {

	if ( ! isset( $_POST['rcpga_submit_member_registration'] ) ) {
		return;
	}

	if ( empty( $_POST['rcpga_member_register_nonce'] ) || ! wp_verify_nonce( $_POST['rcpga_member_register_nonce'], 'rcpga-member-register-nonce' ) ) {
		rcp_errors()->add( 'registration_disabled', __( 'Validation failed', 'rcp-group-accounts' ), 'rcpga_member_registration' );

		return;
	}

	$group_code = ! empty( $_POST['rcpga_group_id'] ) ? strtolower( $_POST['rcpga_group_id'] ) : '';

	rcp_log( sprintf( 'Group Accounts: Processing group member registration for group code %s.', $group_code ) );

	// Bail if self-registration isn't enabled.
	global $rcp_options;
	if ( empty( $rcp_options['groups_allow_member_registration'] ) ) {
		rcp_errors()->add( 'registration_disabled', __( 'Registration is disabled', 'rcp-group-accounts' ), 'rcpga_member_registration' );

		return;
	}

	// Group code must be filled out.
	if ( empty( $group_code ) ) {
		rcp_errors()->add( 'missing_group_code', __( 'Please enter a group code.', 'rcp-group-accounts' ), 'rcpga_member_registration' );

		return;
	}

	$group = rcpga_get_group_by( 'code', $group_code );

	if ( empty( $group ) ) {
		rcp_errors()->add( 'invalid_group_code', __( 'Invalid group code.', 'rcp-group-accounts' ), 'rcpga_member_registration' );
	}

	// Bail if we have errors.
	if ( rcp_errors()->has_errors() ) {
		rcp_log( sprintf( 'Group Accounts: Group member registration cancelled with the following errors: %s.', implode( ', ', rcp_errors()->get_error_messages() ) ), true );

		return;
	}

	if ( ! is_user_logged_in() ) {

		/**
		 * Validate new account information.
		 */

		rcp_log( 'Group Accounts: Group member registration is for a new account' );

		if ( username_exists( $_POST['rcpga_user_login'] ) ) {
			// Username already registered
			rcp_errors()->add( 'username_unavailable', __( 'Username already taken', 'rcp-group-accounts' ), 'rcpga_member_registration' );
		}
		if ( ! rcp_validate_username( $_POST['rcpga_user_login'] ) ) {
			// Invalid username
			rcp_errors()->add( 'username_invalid', __( 'Invalid username', 'rcp-group-accounts' ), 'rcpga_member_registration' );
		}
		if ( empty( $_POST['rcpga_user_login'] ) ) {
			// Empty username
			rcp_errors()->add( 'username_empty', __( 'Please enter a username', 'rcp-group-accounts' ), 'rcpga_member_registration' );
		}
		if ( ! is_email( $_POST['rcpga_user_email'] ) ) {
			// Invalid email
			rcp_errors()->add( 'email_invalid', __( 'Invalid email', 'rcp-group-accounts' ), 'rcpga_member_registration' );
		}
		if ( email_exists( $_POST['rcpga_user_email'] ) ) {
			// Email address already registered
			rcp_errors()->add( 'email_used', sprintf(
				__( 'This email address is already in use. If this is your email address, please <a href="%s">log in</a> and try again.', 'rcp-group-accounts' ),
				esc_url( rcp_get_login_url() )
			), 'rcpga_member_registration' );
		}
		if ( empty( $_POST['rcpga_user_pass'] ) ) {
			// Password is empty
			rcp_errors()->add( 'password_empty', __( 'Please enter a password', 'rcp-group-accounts' ), 'rcpga_member_registration' );
		}
		if ( $_POST['rcpga_user_pass'] !== $_POST['rcpga_user_pass_confirm'] ) {
			// Passwords do not match
			rcp_errors()->add( 'password_mismatch', __( 'Passwords do not match', 'rcp-group-accounts' ), 'rcpga_member_registration' );
		}

	} else {

		/**
		 * Ensure an existing user has permission to join a group.
		 */

		rcp_log( sprintf( 'Group Accounts: Group member registration is for an existing account - ID #%d.', get_current_user_id() ) );

		// Rules for multiple memberships.
		if ( ! rcp_multiple_memberships_enabled() ) {

			if ( rcpga_user_is_group_member() ) {
				// User is already in a group and cannot join another.
				rcp_errors()->add( 'has_group', __( 'You are already in a group.', 'rcp-group-account' ), 'rcpga_member_registration' );
			} elseif ( rcp_user_has_paid_membership( get_current_user_id() ) ) {
				// User already has their own membership and cannot join a group.
				rcp_errors()->add( 'active_user', __( 'Members with an active membership cannot be added to a group. Please cancel your membership and try again.', 'rcp-group-accounts' ), 'rcpga_member_registration' );
			}

		}

		// Error if user is already a member of this group.
		if ( rcpga_user_is_member_of_group( get_current_user_id(), $group->get_group_id() ) ) {
			rcp_errors()->add( 'already_in_group', sprintf( __( 'You are already a member of the group "%s".', 'rcp-group-accounts' ), esc_html( $group->get_name() ) ), 'rcpga_member_registration' );
		}

	}

	if ( ! $group->can_add_members() ) {
		rcp_errors()->add( 'group_seats_maxed', __( 'There are no more seats available in this group.', 'rcp-group-accounts' ), 'rcpga_member_registration' );
	}

	// Terms and conditions / privacy policy must be agreed if enabled.
	if ( ! empty( $rcp_options['enable_terms'] ) && empty( $_POST['rcpga_member_agree_to_terms'] ) ) {
		rcp_errors()->add( 'missing_terms_agreement', __( 'You must agree to the terms and conditions.', 'rcp-group-accounts' ) );
	}
	if ( ! empty( $rcp_options['enable_privacy_policy'] ) && empty( $_POST['rcpga_member_agree_to_privacy_policy'] ) ) {
		rcp_errors()->add( 'missing_privacy_policy_agreement', __( 'You must agree to the privacy policy.', 'rcp-group-accounts' ) );
	}

	// Bail if we have errors.
	if ( rcp_errors()->has_errors() ) {
		rcp_log( sprintf( 'Group Accounts: Group member registration cancelled with the following errors: %s.', implode( ', ', rcp_errors()->get_error_messages() ) ), true );

		return;
	}

	/**
	 * No errors means we can proceed with registration!
	 */

	$args = array(
		'role'        => 'member',
		'send_invite' => false,
	);

	if ( ! is_user_logged_in() ) {
		$args['user_email'] = sanitize_text_field( $_POST['rcpga_user_email'] );
		$args['user_login'] = sanitize_user( $_POST['rcpga_user_login'] );
		$args['first_name'] = ! empty( $_POST['rcpga_user_first'] ) ? sanitize_text_field( $_POST['rcpga_user_first'] ) : '';
		$args['last_name']  = ! empty( $_POST['rcpga_user_last'] ) ? sanitize_text_field( $_POST['rcpga_user_last'] ) : '';
		$args['user_pass']  = $_POST['rcpga_user_pass'];
	} else {
		$args['user_email'] = wp_get_current_user()->user_email;
	}

	$added = $group->add_member( $args );

	if ( is_wp_error( $added ) ) {
		rcp_errors()->add( $added->get_error_code(), $added->get_error_message(), 'rcpga_member_registration' );

		return;
	}

	// Log the user in.
	if ( ! is_user_logged_in() ) {
		wp_signon( array(
			'user_login'    => sanitize_user( $_POST['rcpga_user_login'] ),
			'user_password' => $_POST['rcpga_user_pass']
		) );
	}

	// Redirect to account page.
	$page_url = ! empty( $rcp_options['account_page'] ) ? get_permalink( $rcp_options['account_page'] ) : home_url( '/' );
	$redirect = add_query_arg( array( 'rcpga-message' => 'member-registered' ), $page_url );

	wp_safe_redirect( $redirect );
	exit;

}

add_action( 'init', __NAMESPACE__ . '\process_registration' );