<?php
/**
 * Invite Functions
 *
 * @package   rcp-group-accounts
 * @copyright Copyright (c) 2019, Restrict Content Pro team
 * @license   GPL2+
 * @since     2.0
 */

// Exit if accessed directly
defined( 'ABSPATH' ) || exit;

/**
 * Accept group invite after clicking link
 *
 * @since 2.0
 *
 * @since 2.1.1 Query arg was changed from `rcpga-invite-key` to `rcpga-activation-key`. Links containing the old
 *        query arg will be invalid and the customer will get sent a new email with a new link.
 *
 * @return void
 */
function rcpga_accept_group_invite() {

	global $rcp_options;

	$edit_profile_page = $rcp_options['edit_profile'];
	$edit_profile_link = get_permalink( $edit_profile_page );

	if ( empty( $edit_profile_link ) ) {
		return;
	}

	if ( empty( $_GET['rcpga-user'] ) || empty( $_GET['rcpga-group'] ) ) {
		return;
	}

	// Bail if both the old and new keys are not present.
	if ( empty( $_GET['rcpga-invite-key'] ) && empty( $_GET['rcpga-activation-key'] ) ) {
		return;
	}

	if ( ! $user = get_userdata( absint( $_GET['rcpga-user'] ) ) ) {
		$redirect = add_query_arg( array( 'rcpga-message' => 'invite-invalid' ), $edit_profile_link );

		wp_safe_redirect( $redirect );
		exit;
	}

	$group_member = rcpga_get_group_member( $user->ID, absint( $_GET['rcpga-group'] ) );

	if ( empty( $group_member ) ) {
		$redirect = add_query_arg( array( 'rcpga-message' => 'invite-invalid' ), $edit_profile_link );

		wp_safe_redirect( $redirect );
		exit;
	}

	// If they're using the old `rcpga-invite-key` force them to use a new link.
	if ( ! empty( $_GET['rcpga-invite-key'] ) && empty( $_GET['rcpga-activation-key'] ) ) {
		rcpga_send_group_invite( $group_member->get_user_id(), $group_member->get_group_id() );

		$redirect = add_query_arg( array( 'rcpga-message' => 'invite-resent' ), $edit_profile_link );

		wp_safe_redirect( $redirect );
		exit;
	}

	$is_valid = rcpga_validate_activation_key( rawurldecode( $_GET['rcpga-activation-key'] ), $group_member );

	// Link is not valid.
	if ( is_wp_error( $is_valid ) ) {
		if ( 'invited' == $group_member->get_role() ) {
			/*
			 * Invite link is invalid but user still needs to accept invite.
			 */
			$code     = 'expired_key' === $is_valid->get_error_code() ? 'invite-expired' : 'invite-invalid';
			$redirect = add_query_arg( array( 'rcpga-message' => urlencode( $code ) ), $edit_profile_link );

			wp_safe_redirect( $redirect );
			exit;
		} else {
			/*
			 * User's status is already "member" or higher. This means this is the second time they've clicked the link
			 * but they've already been activated anyway.
			 */
			$redirect = add_query_arg( array( 'rcpga-message' => 'invite-completed' ), $edit_profile_link );

			wp_safe_redirect( $redirect );
			exit;
		}
	}

	// Link is valid!

	// We only need to update the user's role if their status is still "invited".
	if ( 'invited' == $group_member->get_role() ) {
		$group_member->set_role( 'member' );

		/**
		 * Triggers after an invite has been accepted.
		 *
		 * @param WP_User            $user
		 * @param RCPGA_Group_Member $group_member
		 */
		do_action( 'rcpga_member_invite_accepted', $user, $group_member );
	}

	// Log the user in.
	wp_set_auth_cookie( $user->ID, false );
	wp_set_current_user( $user->ID, $user->user_login );
	do_action( 'wp_login', $user->user_login, $user );

	$redirect = add_query_arg( array( 'rcpga-message' => 'invite-accepted' ), $edit_profile_link );

	wp_safe_redirect( $redirect );
	exit;

}

add_action( 'template_redirect', 'rcpga_accept_group_invite' );

/**
 * Sends the group invitation to the user.
 *
 * @param int $user_id  ID of the user to send the invite to.
 * @param int $group_id ID of the group the user is being invited to.
 *
 * @since 1.0
 * @return bool
 */
function rcpga_send_group_invite( $user_id, $group_id = 0 ) {

	global $rcp_options;

	// Try to guess the group ID if not provided.
	if ( empty( $group_id ) ) {
		$group_ids = rcpga_get_group_members( array(
			'user_id' => $user_id,
			'role'    => 'invited',
			'fields'  => 'group_id'
		) );

		if ( empty( $group_ids ) ) {
			return false;
		}

		$group_id = reset( $group_ids );
	}

	if ( ! $group = rcpga_get_group( $group_id ) ) {
		return false;
	}

	if ( ! $user = get_user_by( 'id', $user_id ) ) {
		return false;
	}

	$email = array(
		'to'      => $user->user_email,
		'subject' => ( isset( $rcp_options['group_invite_subject'] ) ) ? $rcp_options['group_invite_subject'] : '',
		'message' => ( isset( $rcp_options['group_invite_email'] ) ) ? $rcp_options['group_invite_email'] : ''
	);

	/**
	 * Filters the email settings.
	 *
	 * @param array $email    Email settings, including address it's being sent to, subject, and message.
	 * @param int   $user_id  ID of the user the email is being sent to.
	 * @param int   $group_id ID of the group the user is being invited to.
	 */
	$email = apply_filters( 'rcpga_send_group_invite_args', $email, $user_id, $group_id );

	if ( empty( $email['subject'] ) || empty( $email['message'] ) || empty( $email['to'] ) ) {
		return false;
	}

	/**
	 * Triggers before the group invite is sent.
	 *
	 * @param int   $user_id  ID of the user the invite is being sent to.
	 * @param array $email    Array of email contents.
	 * @param int   $group_id ID of the group the user is in.
	 */
	do_action( 'rcpga_send_group_invite', $user_id, $email, $group_id );

	$email_class             = new RCP_Emails;
	$email_class->member_id  = $user_id;
	$email_class->membership = $group->get_membership();

	return $email_class->send( $email['to'], $email['subject'], $email['message'] );

}

/**
 * Generate a group invitation link for a specific user.
 *
 * @param int $user_id  ID of the user being invited.
 * @param int $group_id ID of the group the invite is for.
 *
 * @since 2.0
 * @return string|false Invite link on success, false on failure.
 */
function rcpga_generate_group_invite_link_for_user( $user_id, $group_id ) {

	// Failure if the user does not exist.
	if ( ! $user = get_user_by( 'id', $user_id ) ) {
		return false;
	}

	$member = rcpga_get_group_member( $user_id, $group_id );
	if ( ! $member instanceof RCPGA_Group_Member ) {
		return false;
	}

	// Failure if the user is not an "invited" group member.
	if ( 'invited' !== $member->get_role() ) {
		return false;
	}

	$key = rcpga_generate_member_activation_key( $member );

	// Failed to generate key.
	if ( is_wp_error( $key ) ) {
		return false;
	}

	$invite_link = add_query_arg( array(
		'rcpga-activation-key' => rawurlencode( $key ),
		'rcpga-user'           => urlencode( $user->ID ),
		'rcpga-group'          => urlencode( absint( $group_id ) )
	), trailingslashit( home_url() ) );

	/**
	 * Filters the group invite link.
	 *
	 * @param string $invite_link Full invite link.
	 * @param int    $user_id     ID of the user being invited.
	 * @param int    $group_id    ID of the group the user is being invited to.
	 */
	return apply_filters( 'rcpga_group_invite_link', $invite_link, $user_id, $group_id );

}

/**
 * Generates an activation key for a group member, then stores the hash in their record.
 *
 * @param RCPGA_Group_Member $group_member
 *
 * @since 2.1.1
 * @return string|WP_Error Activation key on success (not hashed). WP_Error on failure.
 */
function rcpga_generate_member_activation_key( $group_member ) {
	global $wp_hasher;

	if ( ! $group_member instanceof RCPGA_Group_Member ) {
		return new WP_Error( 'invalid_member', __( 'Invalid group member.', 'rcp-group-accounts' ) );
	}

	$key = wp_generate_password( 20, false );

	if ( empty( $wp_hasher ) ) {
		require_once ABSPATH . WPINC . '/class-phpass.php';
		$wp_hasher = new PasswordHash( 8, true );
	}

	$hashed = time() . ':' . $wp_hasher->HashPassword( $key );

	$updated = $group_member->update( array(
		'activation_key' => $hashed
	) );

	if ( ! $updated ) {
		return new WP_Error( 'update_failed', __( 'Failed to update group member.', 'rcp-group-accounts' ) );
	}

	return $key;
}

/**
 * Validates an activation key for a group member.
 *
 * @param string             $key          Activation key.
 * @param RCPGA_Group_Member $group_member Group member.
 * @param int|null           $time         Timestamp to compare the expiration time with. If `null`, current timestamp
 *                                         is used. This parameter really just exists to allow easy unit testing.
 *
 * @since 2.1.1
 * @return true|WP_Error True if the key is valid, WP_Error otherwise.
 */
function rcpga_validate_activation_key( $key, $group_member, $time = null ) {
	global $wp_hasher;

	if ( empty( $time ) ) {
		$time = time();
	}

	if ( ! $group_member instanceof RCPGA_Group_Member ) {
		return new WP_Error( 'invalid_member', __( 'Invalid group member.', 'rcp-group-accounts' ) );
	}

	if ( empty( $wp_hasher ) ) {
		require_once ABSPATH . WPINC . '/class-phpass.php';
		$wp_hasher = new PasswordHash( 8, true );
	}

	$activation_key = $group_member->get_activation_key();

	if ( false !== strpos( $activation_key, ':' ) ) {
		list( $key_generation_time, $hashed_key ) = explode( ':', $activation_key, 2 );
		$expiration_time = $key_generation_time + rcpga_get_invite_link_validity_length();
	} else {
		return new WP_Error( 'invalid_key', __( 'Invalid key.', 'rcp-group-accounts' ) );
	}

	if ( ! $hashed_key ) {
		return new WP_Error( 'invalid_key', __( 'Invalid key.', 'rcp-group-accounts' ) );
	}

	$is_valid = $wp_hasher->CheckPassword( $key, $hashed_key );

	if ( ! $is_valid ) {
		return new WP_Error( 'invalid_key', __( 'Invalid key.', 'rcp-group-accounts' ) );
	}

	if ( $expiration_time && $time < $expiration_time ) {
		return true;
	}

	return new WP_Error( 'expired_key', __( 'Expired key.', 'rcp-group-accounts' ) );
}

/**
 * Retrieves the amount of time in seconds invite links are valid for.
 *
 * @since 2.1.1
 * @return int Length of time, in seconds.
 */
function rcpga_get_invite_link_validity_length() {
	$validity_duration = WEEK_IN_SECONDS;

	/**
	 * Filters how long invite links remain valid for.
	 *
	 * @param int $validity_duration Validity length, in seconds. Default is 1 week.
	 *
	 * @since 2.1.1
	 */
	$validity_duration = apply_filters( 'rcpga_invite_link_validity_length', $validity_duration );

	// Minimum of 2 hours.
	if ( $validity_duration < ( 2 * HOUR_IN_SECONDS ) ) {
		$validity_duration = 2 * HOUR_IN_SECONDS;
	}

	return $validity_duration;
}

/**
 * Register group email template tags
 *
 * @param array $email_tags
 *
 * @since 1.1
 * @return array
 */
function rcpga_register_email_template_tags( $email_tags ) {

	$email_tags[] = array(
		'tag'         => 'groupname',
		'description' => __( 'The name of the group to which person receiving the email belongs or is being invited', 'rcp-group-accounts' ),
		'function'    => 'rcpga_email_tag_group_name'
	);

	$email_tags[] = array(
		'tag'         => 'groupdesc',
		'description' => __( 'The description of the group to which person receiving the email belongs or is being invited', 'rcp-group-accounts' ),
		'function'    => 'rcpga_email_tag_group_desc'
	);

	$email_tags[] = array(
		'tag'         => 'group_owner_name',
		'description' => __( 'The name of the group owner', 'rcp-group-accounts' ),
		'function'    => 'rcpga_email_tag_group_owner_name'
	);

	$email_tags[] = array(
		'tag'         => 'group_owner_email',
		'description' => __( 'The email address of the group owner', 'rcp-group-accounts' ),
		'function'    => 'rcpga_email_tag_group_owner_email'
	);

	$email_tags[] = array(
		'tag'         => 'invitelink',
		'description' => __( 'The invite link the user will click to join the group, only for the Group Invite Email', 'rcp-group-accounts' ),
		'function'    => 'rcpga_email_tag_invite_link'
	);

	$email_tags[] = array(
		'tag'         => 'invitername',
		'description' => __( 'The display name of the user who invited the person to the group (for example, the group owner or admin), only for the Group Invite Email', 'rcp-group-accounts' ),
		'function'    => 'rcpga_email_tag_inviter_name'
	);

	return $email_tags;

}

add_filter( 'rcp_email_template_tags', 'rcpga_register_email_template_tags' );

/**
 * Email template tag: groupname
 * Name of the group the user has been invited to.
 *
 * @param int            $user_id    ID of the member receiving the email.
 * @param int            $payment_id The ID of the latest payment made by the user.
 * @param string         $tag        Name of the tag being processed.
 * @param RCP_Membership $membership Membership object.
 *
 * @since 1.1
 * @return string
 */
function rcpga_email_tag_group_name( $user_id, $payment_id, $tag, $membership ) {

	if ( empty( $membership ) ) {
		return '';
	}

	$group = rcpga_get_group_by( 'membership_id', $membership->get_id() );

	return ! empty( $group ) ? $group->get_name() : '';

}

/**
 * Email template tag: groupdesc
 * Description of the group the user has been invited to.
 *
 * @param int            $user_id    ID of the member receiving the email.
 * @param int            $payment_id The ID of the latest payment made by the user.
 * @param string         $tag        Name of the tag being processed.
 * @param RCP_Membership $membership Membership object.
 *
 * @since 1.1
 * @return string
 */
function rcpga_email_tag_group_desc( $user_id, $payment_id, $tag, $membership ) {

	if ( empty( $membership ) ) {
		return '';
	}

	$group = rcpga_get_group_by( 'membership_id', $membership->get_id() );

	return ! empty( $group ) ? $group->get_description() : '';

}

/**
 * Email template tag: group_owner_name
 * Name of the group owner.
 *
 * @param int            $user_id    ID of the member receiving the email.
 * @param int            $payment_id The ID of the latest payment made by the user.
 * @param string         $tag        Name of the tag being processed.
 * @param RCP_Membership $membership Membership object.
 *
 * @since 1.3
 * @return string
 */
function rcpga_email_tag_group_owner_name( $user_id, $payment_id, $tag, $membership ) {

	if ( empty( $membership ) ) {
		return '';
	}

	$group = rcpga_get_group_by( 'membership_id', $membership->get_id() );

	if ( empty( $group ) ) {
		return '';
	}

	$owner = $group->get_owner();

	if ( empty( $owner ) ) {
		return '';
	}

	return $owner->display_name;

}

/**
 * Email template tag: group_owner_email
 * Name of the group owner.
 *
 * @param int            $user_id    ID of the member receiving the email.
 * @param int            $payment_id The ID of the latest payment made by the user.
 * @param string         $tag        Name of the tag being processed.
 * @param RCP_Membership $membership Membership object.
 *
 * @since 1.3
 * @return string
 */
function rcpga_email_tag_group_owner_email( $user_id, $payment_id, $tag, $membership ) {

	if ( empty( $membership ) ) {
		return '';
	}

	$group = rcpga_get_group_by( 'membership_id', $membership->get_id() );

	if ( empty( $group ) ) {
		return '';
	}

	$owner = $group->get_owner();

	if ( empty( $owner ) ) {
		return '';
	}

	return $owner->user_email;

}

/**
 * Template tag: invitelink
 * Invitation link to join the group.
 *
 * @param int            $user_id    ID of the member receiving the email.
 * @param int            $payment_id The ID of the latest payment made by the user.
 * @param string         $tag        Name of the tag being processed.
 * @param RCP_Membership $membership Membership object.
 *
 * @since 1.1
 * @return string
 */
function rcpga_email_tag_invite_link( $user_id, $payment_id, $tag, $membership ) {

	if ( empty( $membership ) ) {
		return '';
	}

	$group = rcpga_get_group_by( 'membership_id', $membership->get_id() );

	if ( empty( $group ) ) {
		return '';
	}

	return rcpga_generate_group_invite_link_for_user( $user_id, $group->get_group_id() );

}

/**
 * Template tag: invitername
 * Name of the user who sent the invitation - usually the group owner or admin.
 *
 * @param int            $user_id    ID of the member receiving the email.
 * @param int            $payment_id The ID of the latest payment made by the user.
 * @param string         $tag        Name of the tag being processed.
 * @param RCP_Membership $membership Membership object.
 *
 * @since 1.2.1
 * @return string
 */
function rcpga_email_tag_inviter_name( $user_id, $payment_id, $tag, $membership ) {

	$user = wp_get_current_user();

	return $user->display_name;

}