<?php
/**
 * Group Role & Capability Functions
 *
 * @package   rcp-group-accounts
 * @copyright Copyright (c) 2019, Restrict Content Pro team
 * @license   GPL2+
 * @since     2.0
 */

/**
 * Gets an array of available group roles
 *
 * @since 2.0
 * @return array
 */
function rcpga_get_group_roles() {
	return array( 'owner', 'admin', 'member', 'invited' );
}

/**
 * Gets an array of available group capabilities
 *
 * @since 2.0
 * @return array
 */
function rcpga_get_group_capabilities() {
	return array( 'manage_billing', 'manage_members', 'view_group' );
}

/**
 * Get the capabilities assigned to a given role.
 *
 * @param string $role Role slug.
 *
 * @since 2.0
 * @return array
 */
function rcpga_get_role_capabilities( $role ) {

	$capabilities = array();

	switch ( $role ) {
		case 'owner' :
			$capabilities[] = 'manage_billing';
			$capabilities[] = 'manage_members';
			$capabilities[] = 'view_group';
			break;

		case 'admin' :
			$capabilities[] = 'manage_members';
			$capabilities[] = 'view_group';
			break;

		case 'member' :
			$capabilities[] = 'view_group';
			break;
	}

	/**
	 * Filters the capabilities associated with a role.
	 *
	 * @param array  $capabilities Array of capabilities.
	 * @param string $role         Role slug.
	 *
	 * @since 2.0
	 */
	return apply_filters( 'rcpga_get_role_capabilities', $capabilities, $role );

}

/**
 * Determines whether or not a role has a certain capability.
 *
 * @param string $role       Role to check.
 * @param string $capability Capability to check.
 *
 * @since 2.0
 * @return bool
 */
function rcpga_role_can( $role, $capability ) {

	$role_capabilities = rcpga_get_role_capabilities( $role );

	return in_array( $capability, $role_capabilities );

}

/**
 * Retrieves the label for a group role
 *
 * @param string $role Role to retrieve the label for.
 *
 * @since 1.2
 * @return string
 */
function rcpga_get_member_role_label( $role ) {

	$label = '';

	switch ( $role ) {

		case 'owner' :
			$label = __( 'Owner', 'rcp-group-accounts' );
			break;

		case 'admin' :
			$label = __( 'Admin', 'rcp-group-accounts' );
			break;

		case 'member' :
			$label = __( 'Member', 'rcp-group-accounts' );
			break;

		case 'invited' :
			$label = __( 'Invited', 'rcp-group-accounts' );
			break;

	}

	/**
	 * Filters the role label.
	 *
	 * @param string $label Label formatted for display.
	 * @param string $role  Role slug.
	 */
	return apply_filters( 'rcpga_get_member_role_label', $label, $role );

}