<?php
/**
 * Member Functions
 *
 * @package   rcp-group-accounts
 * @copyright Copyright (c) 2019, Restrict Content Pro team
 * @license   GPL2+
 * @since     2.0
 */

use RCP\Database\Queries;

/**
 * Get a user's record within a certain group.
 *
 * @param int $user_id  ID of the user.
 * @param int $group_id ID of the group to get the user's record for.
 *
 * @since 2.0
 * @return RCPGA_Group_Member|false Group member object on success, false on failure.
 */
function rcpga_get_group_member( $user_id, $group_id ) {

	$members = rcpga_get_group_members( array(
		'user_id'  => absint( $user_id ),
		'group_id' => absint( $group_id )
	) );

	if ( empty( $members ) || ! isset( $members[0] ) ) {
		return false;
	}

	return reset( $members );

}

/**
 * Get group member by ID
 *
 * @param int $id Group member ID
 *
 * @since 2.0
 * @return RCPGA_Group_Member|false Group member object on success, false on failure.
 */
function rcpga_get_group_member_by_id( $id ) {

	$members_query = new Queries\Group_Members();

	return $members_query->get_item( absint( $id ) );

}

/**
 * Get group members
 *
 * @param array       $args               {
 *                                        Optional. Array of query arguments to override the defaults.
 *
 * @type int          $id                 A group member ID to only return that member. Default empty.
 * @type array        $id__in             Array of member IDs to include. Default empty.
 * @type array        $id__not_in         Array of group member IDs to exclude. Default empty.
 * @type int          $user_id            A user ID to filter by. Default empty.
 * @type array        $user_id__in        Array of user IDs to include. Default empty.
 * @type array        $user_id__not_in    Array of user IDs to exclude. Default empty.
 * @type int          $group_id           A group ID to filter by. Default empty.
 * @type array        $group_id_in        Array of group IDs to include. Default empty.
 * @type array        $group_id__not_in   Array of group IDs to exclude. Default empty.
 * @type string       $role               A specific member role to search for. Default empty.
 * @type array        $role__in           Filter by roles. Default empty.
 * @type array        $role__not_in       Array of roles to exclude. Default empty.
 * @type array        $date_query         Query all datetime columns together. See WP_Date_Query.
 * @type array        $date_created_query Date query clauses to limit groups by. See WP_Date_Query.
 *                                        Default null.
 * @type bool         $count              Whether to return a group member count (true) or array of group
 *                                        member objects. Default false.
 * @type string       $fields             Item fields to return. Accepts any column known names
 *                                        or empty (returns an array of complete group member objects).
 *                                        Default empty.
 * @type int          $number             Limit number of group members to retrieve. Default all.
 * @type int          $offset             Number of group members to offset the query. Used to build LIMIT
 *                                        clause. Default 0.
 * @type bool         $no_found_rows      Whether to disable the `SQL_CALC_FOUND_ROWS` query. Default true.
 * @type string|array $orderby            Accepts 'user_id', 'group_id', 'role', and 'date_created'.
 *                                        Also accepts false, an empty array, or 'none' to disable
 *                                        `ORDER BY` clause. Default 'user_id'.
 * @type string       $order              How to order results. Accepts 'ASC', 'DESC'. Default 'DESC'.
 * @type string       $search             Search term(s) to retrieve matching group members for. Default
 *                                        empty.
 * @type bool         $update_cache       Whether to prime the cache for found group members. Default false.
 *                    }
 *
 * @since 2.0
 * @return array
 */
function rcpga_get_group_members( $args = array() ) {

	$args = wp_parse_args( $args, array(
		'number' => 20
	) );

	$members_query = new Queries\Group_Members();

	$result = $members_query->query( $args );

	if ( ! empty( $args['group_id'] ) ) {
		/**
		 * Filters the list of group members.
		 *
		 * @param array $result   Array of RCPGA_Group_Member objects.
		 * @param int   $group_id ID of the group being queried.
		 */
		$result = apply_filters( 'rcpga_get_members', $result, $args['group_id'] );
	}

	return $result;

}

/**
 * Count the number of group members
 *
 * @param array $args Array of arguments to override the defaults.
 *
 * @see   rcpga_get_group_members() for list of accepted arguments.
 *
 * @since 2.0
 * @return int
 */
function rcpga_count_group_members( $args = array() ) {

	$args = wp_parse_args( $args, array(
		'count' => true
	) );

	$groups_query = new Queries\Group_Members( $args );

	$count = absint( $groups_query->found_items );

	if ( ! empty( $args['group_id'] ) ) {
		/**
		 * Filters the number of members in a group.
		 *
		 * @param int $count    Number of members in the group.
		 * @param int $group_id ID of the group.
		 */
		$count = apply_filters( 'rcpga_count_members', $count, $args['group_id'] );
	}

	return $count;

}

/**
 * Query for and return array of group members counts, keyed by role.
 *
 * @param array $args Query arguments to override the defaults.
 *
 * @since 2.0
 * @return array
 */
function rcpga_get_group_member_counts( $args = array() ) {

	$args = wp_parse_args( $args, array(
		'count'   => true,
		'groupby' => 'role'
	) );

	$groups_query = new Queries\Group_Members( $args );
	$counts       = rcp_format_counts( $groups_query, $args['groupby'] );

	// Reorder the array to sort by rank.
	$order = array( 'total', 'owner', 'admin', 'member', 'invited' );
	$final = array();

	foreach ( $order as $role ) {
		if ( isset( $counts[ $role ] ) ) {
			$final[ $role ] = $counts[ $role ];
		}
	}

	return $final;

}

/**
 * Delete all members from a certain group. This is used when a group is deleted.
 *
 * @param int $group_id ID of the group to delete members from.
 *
 * @since 2.0
 * @return int|false The number of members deleted, or false on error.
 */
function rcpga_delete_all_members_of_group( $group_id ) {

	global $wpdb;
	$table_name = rcpga_group_accounts()->table_group_members->get_table_name();

	$deleted = $wpdb->delete( $table_name, array(
		'group_id' => $group_id
	), array( '%d' ) );

	return $deleted;

}

/**
 * Determines if a user is a member of ANY group.
 *
 * @param int          $user_id ID of the user to check. Defaults to currently logged in user.
 * @param string|array $role    Set to filter results by role.
 *
 * @since 2.0
 * @return bool
 */
function rcpga_user_is_group_member( $user_id = 0, $role = '' ) {

	if ( empty( $user_id ) ) {
		$user_id = get_current_user_id();
	}

	$args = array(
		'user_id' => absint( $user_id )
	);

	if ( ! empty( $role ) ) {
		if ( is_array( $role ) ) {
			$args['role__in'] = array_map( 'sanitize_text_field', $role );
		} else {
			$args['role'] = sanitize_text_field( $role );
		}
	}

	$is_group_member = rcpga_count_group_members( $args ) > 0;

	if ( 'member' === $role ) {

		/**
		 * Filters whether or not the user is a group member. This is only triggered when checking the
		 * `member` role specifically.
		 *
		 * @param bool $is_group_member Whether or not the user is a group member with the role `member`.
		 * @param int  $user_id         ID of the user being checked.
		 */
		$is_group_member = apply_filters( 'rcpga_is_group_member', $is_group_member, $user_id );

	} elseif ( 'admin' === $role ) {

		/**
		 * Filters whether or not the user is a group admin. This is only triggered when checking the
		 * `admin` role specifically.
		 *
		 * @param bool $is_group_member Whether or not the user is a group member with the role `admin`.
		 * @param int  $user_id         ID of the user being checked.
		 */
		$is_group_member = apply_filters( 'rcpga_is_group_admin', $is_group_member, $user_id );

	} elseif ( 'owner' === $role ) {

		/**
		 * Filters whether or not the user is a group owner. This is only triggered when checking the
		 * `owner` role specifically.
		 *
		 * @param bool $is_group_member Whether or not the user is a group member with the role `owner`.
		 * @param int  $user_id         ID of the user being checked.
		 */
		$is_group_member = apply_filters( 'rcpga_is_group_owner', $is_group_member, $user_id );

	}

	return $is_group_member;

}

/**
 * Determines if a user is a member of a specific group.
 *
 * @param int $user_id  ID of the user to check.
 * @param int $group_id ID of the group to check.
 *
 * @since 2.0
 * @return bool
 */
function rcpga_user_is_member_of_group( $user_id, $group_id ) {

	$args = array(
		'user_id'  => absint( $user_id ),
		'group_id' => absint( $group_id )
	);

	$memberships = rcpga_get_group_members( $args );

	return ! empty( $memberships );

}

/**
 * Gets the user's role in a group.
 *
 * @param int $user_id
 * @param int $group_id
 *
 * @since 2.0
 * @return string|false Role slug on success, false on failure.
 */
function rcpga_get_user_group_role( $user_id, $group_id ) {

	$member = rcpga_get_group_member( $user_id, $group_id );

	if ( empty( $member ) ) {
		return false;
	}

	return $member->get_role();

}

/**
 * Add a new member to an existing group
 *
 * @param int   $group_id    ID of the group to add the user to.
 * @param array $args        {
 *                           Array of group member arguments.
 *
 * @type int    $user_id     Optional. ID of the user account. If omitted, `user_email` must be supplied.
 * @type string $user_email  Optional. Email address of the group member. If provided, a new account will be created
 *                          with this email. This is required if `user_id` is omitted.
 * @type string $user_login  Optional. New member's username. Used in newa ccount creation. Will fall back to the email
 *                          address if not provided.
 * @type string $first_name  Optional. First name of the member. Used in new account creation.
 * @type string $last_name   Optional. Last name of the member. Used in new account creation.
 * @type string $user_pass   Optional. Password for the new member account. Default is randomly generated.
 * @type string $role        Optional. Membership role: `owner`, `admin`, `member`, or `invited`. Default is `invited`.
 * @type bool   $send_invite Optional. Whether or not to send the invitation email. Defaults to true.
 * @type string $date_added  Optional. Date the member was added in MySQL format. Default is now.
 *                    }
 *
 * @since 2.0
 * @return int|WP_Error ID of the group member on success, or WP_Error on failure.
 */
function rcpga_add_group_member( $group_id, $args ) {

	$group = rcpga_get_group( $group_id );

	if ( empty( $group ) ) {
		return new WP_Error( 'invalid_group', __( 'Invalid group', 'rcp-group-accounts' ) );
	}

	return $group->add_member( $args );

}

/**
 * Determines whether or not a user has a certain capability within a group.
 *
 * @param int    $user_id
 * @param string $capability
 * @param int    $group_id
 *
 * @since 2.0
 * @return bool
 */
function rcpga_user_can( $user_id, $capability, $group_id ) {

	// Admins can do anything.
	if ( user_can( $user_id, 'manage_options' ) ) {
		return true;
	}

	$group_member = rcpga_get_group_member( $user_id, $group_id );

	if ( empty( $group_member ) ) {
		return false;
	}

	return $group_member->can( $capability );

}

/**
 * Send new user notifications
 *
 * @param int    $user_id ID of the user new user.
 * @param string $notify  Who to notify: 'both', 'admin', or 'user'.
 *
 * @since 1.1.1
 * @return void
 */
function rcpga_send_new_user_notifications( $user_id, $notify = 'both' ) {

	/**
	 * After the password reset key is generated and before the email body is created,
	 * add our filter to replace the URLs in the email body.
	 */
	add_action( 'retrieve_password_key', function () {

		add_filter( 'wp_mail', function ( $args ) {

			global $rcp_options;

			if ( ! empty( $rcp_options['hijack_login_url'] ) && ! empty( $rcp_options['login_redirect'] ) ) {

				// Rewrite the password reset link
				$args['message'] = str_replace( trailingslashit( network_site_url() ) . 'wp-login.php?action=rp', get_permalink( $rcp_options['login_redirect'] ) . '?rcp_action=lostpassword_reset', $args['message'] );

			}

			return $args;

		} );

	} );

	wp_send_new_user_notifications( $user_id, $notify );

}