<?php
/**
 * Group Member Actions
 *
 * @package   rcp-group-accounts
 * @copyright Copyright (c) 2019, Restrict Content Pro team
 * @license   GPL2+
 * @since     2.0
 */

namespace RCPGA;

/**
 * Send member invite if applicable
 *
 * @param int   $user_id  ID of the user who was just added.
 * @param array $args     Arguments used while adding the member.
 * @param int   $group_id ID of the group the user was added to.
 *
 * @since 2.0
 * @return void
 */
function maybe_send_member_invite( $user_id, $args, $group_id ) {

	if ( 'invited' == $args['role'] && ! empty( $args['send_invite'] ) ) {
		rcpga_send_group_invite( $user_id, $group_id );
	}

}

add_action( 'rcpga_add_member_to_group_after', __NAMESPACE__ . '\maybe_send_member_invite', 10, 3 );

/**
 * Ensure the gropu member has the same user role as the group owner.
 *
 * @param string $old_role        Old membership role, prior to this update.
 * @param string $new_role        New membership role that was just set.
 * @param int    $group_member_id ID of the group member.
 *
 * @since 2.0
 * @return void
 */
function maybe_set_group_member_user_role( $old_role, $new_role, $group_member_id ) {

	// Do not update role for invited role.
	if ( 'invited' == $new_role ) {
		return;
	}

	$group_member = rcpga_get_group_member_by_id( $group_member_id );

	if ( empty( $group_member ) ) {
		return;
	}

	$group_member->set_membership_role();

}

add_action( 'rcp_transition_group_member_role', __NAMESPACE__ . '\maybe_set_group_member_user_role', 10, 3 );

/**
 * Send new user notification when a group member transitions from "invited" to full member.
 *
 * @param string $old_role        Old membership role, prior to this update.
 * @param string $new_role        New membership role that was just set.
 * @param int    $group_member_id ID of the group member.
 *
 * @since 2.0
 * @return void
 */
function maybe_send_new_user_notification( $old_role, $new_role, $group_member_id ) {

	global $rcp_options;

	// Bail if previous role was not "invited".
	if ( 'invited' != $old_role ) {
		return;
	}

	$group_member = rcpga_get_group_member_by_id( $group_member_id );

	if ( empty( $group_member ) ) {
		return;
	}

	$notify = false;

	if ( ! empty( $rcp_options['disable_new_user_notices'] ) && empty( $rcp_options['disable_group_welcome_email'] ) ) {
		// Send notification to user only.
		$notify = 'user';
	} elseif ( empty( $rcp_options['disable_new_user_notices'] ) && ! empty( $args['disable_group_welcome_email'] ) ) {
		// Send notification to admin only.
		$notify = 'admin';
	} elseif ( empty( $rcp_options['disable_new_user_notices'] ) && empty( $rcp_options['disable_group_welcome_email'] ) ) {
		// Send notification to both user and admin.
		$notify = 'both';
	}

	if ( apply_filters( 'rcpga_send_invite_welcome_email', $notify, $group_member ) ) {
		// Backwards compatibility check in case $notify is just `true`.
		$notify = in_array( $notify, array( 'both', 'user', 'admin' ) ) ? $notify : 'both';

		rcpga_send_new_user_notifications( $group_member->get_user_id(), $notify );
	}

}

add_action( 'rcp_transition_group_member_role', __NAMESPACE__ . '\maybe_send_new_user_notification', 10, 3 );

/**
 * When a user account is deleted, remove this user from all groups they're in.
 *
 * @param int $user_id
 *
 * @since 2.0
 * @return void
 */
function remove_deleted_user_from_group( $user_id ) {

	$group_memberships = rcpga_get_group_members( array(
		'user_id' => $user_id
	) );

	if ( empty( $group_memberships ) ) {
		return;
	}

	foreach ( $group_memberships as $group_membership ) {
		/**
		 * @var \RCPGA_Group_Member $group_membership
		 */
		if ( 'owner' === $group_membership->get_role() ) {
			// If they're the owner of the group, delete the whole group.
			$group_membership->get_group()->delete();
		} else {
			// Just remove the member.
			$group_membership->remove();
		}
	}

}

add_action( 'deleted_user', __NAMESPACE__ . '\remove_deleted_user_from_group' );