<?php
/**
 * Group Meta
 *
 * @package   rcp-group-accounts
 * @copyright Copyright (c) 2020, Sandhills Development, LLC
 * @license   GPL2+
 * @since     2.1
 */

namespace RCPGA\Groups;

/**
 * Adds meta data field to a group
 *
 * @param int    $group_id   Group ID.
 * @param string $meta_key   Meta data name.
 * @param mixed  $meta_value Meta data value. Must be serializable if non-scalar.
 * @param bool   $unique     Optional. Whether the same key should not be added. Default false.
 *
 * @since 2.1
 * @return int|false ID of the new meta entry on success.
 */
function add_meta( $group_id, $meta_key, $meta_value, $unique = false ) {
	return add_metadata( 'rcp_group', $group_id, $meta_key, $meta_value, $unique );
}

/**
 * Removes meta data matching criteria from a group.
 *
 * You can match based on the key, or key and value. Removing based on key and value, will keep from removing duplicate
 * meta data with the same key. It also allows removing all meta data matching key, if needed.
 *
 * @param int    $group_id   Group ID.
 * @param string $meta_key   Meta data name.
 * @param mixed  $meta_value Meta data value. Must be serializable if non-scalar. Default empty.
 *
 * @since 2.1
 * @return bool True on successful delete, false on failure.
 */
function delete_meta( $group_id, $meta_key, $meta_value = '' ) {
	return delete_metadata( 'rcp_group', $group_id, $meta_key, $meta_value );
}

/**
 * Retrieves meta field for a group.
 *
 * @param int    $group_id      Group ID.
 * @param string $key           Optional. The meta key to retrieve. By default, returns data for all keys. Default
 *                              empty.
 * @param bool   $single        Optional, default is false. If true, return only the first value of the specified
 *                              meta_key. This parameter has no effect if meta_key is not specified.
 *
 * @since 2.1
 * @return mixed Will be an array if $single is false. Will be value of meta data field if $single is true.
 */
function get_meta( $group_id, $key = '', $single = false ) {
	return get_metadata( 'rcp_group', $group_id, $key, $single );
}

/**
 * Updates meta field based on group ID and key
 *
 * Use the $prev_value parameter to differentiate between meta fields with the
 * same key and group ID.
 *
 * If the meta field for the group does not exist, it will be added.
 *
 * @param int    $group_id   Group ID.
 * @param string $meta_key   Meta data key.
 * @param mixed  $meta_value Meta data value. Must be serializable if non-scalar.
 * @param mixed  $prev_value Optional. Previous value to check before removing. Default empty.
 *
 * @since 2.1
 * @return int|false Meta ID if the key didn't exist, true on successful update, false on failure.
 */
function update_meta( $group_id, $meta_key, $meta_value, $prev_value = '' ) {
	return update_metadata( 'rcp_group', $group_id, $meta_key, $meta_value, $prev_value );
}