<?php
/**
 * Group Functions
 *
 * @package   rcp-group-accounts
 * @copyright Copyright (c) 2019, Restrict Content Pro team
 * @license   GPL2+
 * @since     2.0
 */

use RCP\Database\Queries;

/**
 * Get a single group by its ID.
 *
 * @param int $group_id
 *
 * @since 3.0
 * @return RCPGA_Group|false
 */
function rcpga_get_group( $group_id ) {

	$group_query = new Queries\Groups();

	return $group_query->get_item( $group_id );

}

/**
 * Get a single group by a field/value pair.
 *
 * @param string $field Column to search in.
 * @param mixed  $value Value of the row.
 *
 * @since 2.0
 * @return RCPGA_Group|false
 */
function rcpga_get_group_by( $field = '', $value = '' ) {

	$group_query = new Queries\Groups();

	return $group_query->get_item_by( $field, $value );

}

/**
 * Returns an array of groups.
 *
 * @param array       $args               {
 *
 * @type int          $group_id           A group ID to only return that group. Default empty.
 * @type array        $group_id__in       Array of group IDs to include. Default empty.
 * @type array        $group_id__not_in   Array of group IDs to exclude. Default empty.
 * @type int          $owner_id           An owner user ID to filter by. Default empty.
 * @type array        $owner_id_in        Array of owner user IDs to include. Default empty.
 * @type array        $owner_id__not_in   Array of owner user IDs to exclude. Default empty.
 * @type string       $name               A specific group name to search for.
 * @type string       $code               A specific group code to filter by. Default empty.
 * @type int          $seats              Query for groups with this many seats. Default empty.
 * @type array        $date_query         Query all datetime columns together. See WP_Date_Query.
 * @type array        $date_created_query Date query clauses to limit groups by. See WP_Date_Query.
 *                                              Default null.
 * @type bool         $count              Whether to return a group count (true) or array of group
 *                                              objects. Default false.
 * @type string       $fields             Item fields to return. Accepts any column known names
 *                                              or empty (returns an array of complete group objects). Default
 *                                              empty.
 * @type int          $number             Limit number of groups to retrieve. Default 100.
 * @type int          $offset             Number of groups to offset the query. Used to build LIMIT
 *                                              clause. Default 0.
 * @type bool         $no_found_rows      Whether to disable the `SQL_CALC_FOUND_ROWS` query. Default true.
 * @type string|array $orderby            Accepts 'group_id', 'owner_id', 'name', 'seats', 'member_count'
 *                                              and 'date_created'. Also accepts false, an empty array, or
 *                                              'none' to disable `ORDER BY` clause. Default 'group_id'.
 * @type string       $order              How to order results. Accepts 'ASC', 'DESC'. Default 'DESC'.
 * @type string       $search             Search term(s) to retrieve matching groups for. Default empty.
 * @type bool         $update_cache       Whether to prime the cache for found groups. Default false.
 *                    }
 *
 * @since 2.0
 * @return array
 */
function rcpga_get_groups( $args = array() ) {

	$args = wp_parse_args( $args, array(
		'number' => 20
	) );

	$groups_query = new Queries\Groups();

	return $groups_query->query( $args );

}

/**
 * Count the number of groups.
 *
 * @param array $args Array of arguments.
 *
 * @see   rcpga_get_groups() for accepted arguments.
 *
 * @since 2.0
 * @return int
 */
function rcpga_count_groups( $args = array() ) {

	$args = wp_parse_args( $args, array(
		'count' => true
	) );

	$groups_query = new Queries\Groups( $args );

	return absint( $groups_query->found_items );

}

/**
 * @param array $args          {
 *
 * @type int    $owner_id      Required. ID of the group owner's user account.
 * @type int    $membership_id Optional. ID of the associated membership record. Warning: group will not be fully "active" if membership_id is omitted.
 * @type string $name          Required. Name of the group.
 * @type string $description   Optional. Group description.
 * @type int    $seats         Optional. Maximum number of seats. Default: auto determined based on membership level.
 * @type int    $member_count  Optional. Current member count. Default 1 (includes owner).
 * @type string $date_created  Optional. Date the group was created. Default now.
 *                    }
 *
 * @since 2.0
 * @return int|WP_Error ID of the new group on success, WP_Error on failure.
 */
function rcpga_add_group( $args = array() ) {

	$args = wp_parse_args( $args, array(
		'owner_id'      => 0,
		'membership_id' => 0,
		'name'          => '',
		'description'   => '',
		'seats'         => 0,
		'member_count'  => 1,
		'date_created'  => current_time( 'mysql' )
	) );

	$membership = ! empty( $args['membership_id'] ) ? rcp_get_membership( $args['membership_id'] ) : false;

	// Required fields.

	if ( empty( $args['owner_id'] ) ) {
		return new WP_Error( 'missing_owner_id', __( 'You must supply an owner ID.', 'rcp-group-accounts' ) );
	}

	/*if ( empty( $args['membership_id'] ) || ! $membership = rcp_get_membership( $args['membership_id'] ) ) {
		return new WP_Error( 'missing_membership_id', __( 'You must supply a membership ID.', 'rcp-group-accounts' ) );
	}*/

	if ( empty( $args['name'] ) ) {
		return new WP_Error( 'missing_name', __( 'You must supply a group name.', 'rcp-group-accounts' ) );
	}

	// Membership does not support groups.
	if ( ! empty( $membership ) && ! rcpga_is_level_group_accounts_enabled( $membership->get_object_id() ) ) {
		return new WP_Error( 'membership_level_not_supported', __( 'Membership level does not support group creation.', 'rcp-group-accounts' ) );
	}

	// Figure out how many seats this group gets.
	if ( empty( $args['seats'] ) && ! empty( $membership ) ) {
		$args['seats'] = rcpga_get_level_group_seats_allowed( $membership->get_object_id() );
	}

	$groups_query = new Queries\Groups();
	$added        = $groups_query->add_item( $args );

	if ( false === $added ) {
		return new WP_Error( 'groups_database_error', __( 'Failed to insert new group.', 'rcp-group-accounts' ) );
	}

	/**
	 * Triggers after the group is created.
	 *
	 * @param int   $added ID of the new group.
	 * @param array $args  Arguments used to create the group.
	 *
	 * @see rcpga_auto_add_owner_to_group() - 10 - Auto adds the owner to the group as a member.
	 */
	do_action( 'rcpga_db_groups_post_insert', $added, $args );

	return $added;

}

/**
 * Update a group
 *
 * @param int   $group_id ID of the group to update.
 * @param array $args     Array of data to update.
 *
 * @since 2.0
 * @return bool
 */
function rcpga_update_group( $group_id, $args = array() ) {

	$group = rcpga_get_group( $group_id );

	if ( empty( $group ) ) {
		return false;
	}

	return $group->update( $args );

}

/**
 * Delete a group
 *
 * @param int $group_id ID of the group to delete.
 *
 * @since 2.0
 * @return bool
 */
function rcpga_delete_group( $group_id ) {

	$group = rcpga_get_group( $group_id );

	if ( empty( $group ) ) {
		return false;
	}

	return $group->delete();

}