<?php
/**
 * Group Actions
 *
 * @package   rcp-group-accounts
 * @copyright Copyright (c) 2019, Restrict Content Pro team
 * @license   GPL2+
 * @since     2.0
 */

namespace RCPGA;

use RCP_Membership;
use RCPGA_Group;
use RCPGA_Group_Member;
use WP_Error;

/**
 * Handle routing group actions to the correct functions.
 * This handles admin and front-end actions in the same way.
 *
 * @since 2.0
 * @return void
 */
function action_router() {

	if ( empty( $_REQUEST['rcpga-action'] ) ) {
		return;
	}

	$action  = $_REQUEST['rcpga-action'];
	$message = '';
	$view    = isset( $_REQUEST['rcpga-view'] ) ? urldecode( $_REQUEST['rcpga-view'] ) : false;

	switch ( $action ) {

		case 'add-group' :
			$message = process_add_group();
			break;

		case 'edit-group' :
			$message = process_edit_group();
			break;

		case 'add-member' :
			$message = process_add_group_member();
			break;

		case 'import-members' :
			$message = process_import_members_to_group();
			break;

		case 'remove-member' :
			$message = process_remove_member_from_group();
			break;

		case 'make-admin' :
		case 'make-member' :
			$message = process_change_member_role();
			break;

		case 'resend-invite' :
			$message = process_resend_member_invite();
			break;

		case 'make-owner' :
			$message = process_change_owner();
			break;

		case 'leave-group' :
			$message = process_leave_group();
			break;

		case 'delete-group' :
			$message = process_delete_group();
			break;

	}

	/**
	 * Triggers after this action has triggered.
	 *
	 * @param string $action  Action being performed.
	 * @param string $message Message from the action.
	 */
	do_action( 'rcpga_action_router', $action, $message );

	if ( empty( $message ) ) {
		return;
	}

	$redirect_args = array(
		'rcpga-message' => urlencode( $message )
	);

	if ( ! empty( $_REQUEST['rcpga-group'] ) ) {
		$redirect_args['rcpga-group'] = urlencode( absint( $_REQUEST['rcpga-group'] ) );
	}

	if ( ! empty( $view ) ) {
		$redirect_args['rcpga-view'] = urlencode( $view );
	}

	$referer = ! empty( $_SERVER['HTTP_REFERER'] ) ? $_SERVER['HTTP_REFERER'] : wp_get_referer();

	if ( empty( $referer ) )  {
		/*
		 * If the referer is empty, fall back to using the Groups admin page if admin side, or
		 * the user account page if on the front-end.
		 */
		global $rcp_options;

		$referer = is_admin() && function_exists( '\RCPGA\Admin\get_groups_page' ) ? \RCPGA\Admin\get_groups_page() : get_permalink( $rcp_options['account_page'] );

		// If it's STILL empty, use the homepage.
		if ( empty( $referer ) ) {
			$referer = home_url( '/' );
		}
	}

	wp_safe_redirect( add_query_arg( $redirect_args, remove_query_arg( 'rcpga-action', $referer ) ) );
	exit;

}

add_action( 'init', __NAMESPACE__ . '\action_router' );

/**
 * Process adding a group.
 *
 * @since 2.0
 * @return string|void
 */
function process_add_group() {

	if ( empty( $_REQUEST['rcpga-action'] ) || 'add-group' != $_REQUEST['rcpga-action'] ) {
		return;
	}

	// Check permissions.
	if ( ! current_user_can( 'manage_options' ) ) {
		return 'no-permission';
	}

	// Nonce validation
	if ( empty( $_REQUEST['rcpga_add_group_nonce'] ) || ! wp_verify_nonce( $_REQUEST['rcpga_add_group_nonce'], 'rcpga_add_group' ) ) {
		return 'default-error';
	}

	// Group owner can only be specified in the admin area.
	if ( is_admin() ) {

		if ( empty( $_REQUEST['rcpga-user-email'] ) ) {
			return 'empty-email';
		}

		$user = get_user_by( 'email', $_REQUEST['rcpga-user-email'] );

		if ( ! empty( $user ) ) {
			$user_id = $user->ID;
		} else {
			// Create a new user.
			$user_id = wp_insert_user( array(
				'user_email' => sanitize_text_field( $_REQUEST['rcpga-user-email'] ),
				'user_login' => ! empty( $_REQUEST['rcpga-user-login'] ) ? sanitize_user( $_REQUEST['rcpga-user-login'] ) : sanitize_user( $_REQUEST['rcpga-user-email'] ),
				'first_name' => isset( $_REQUEST['rcpga-first-name'] ) ? sanitize_text_field( $_REQUEST['rcpga-first-name'] ) : '',
				'last_name'  => isset( $_REQUEST['rcpga-last-name'] ) ? sanitize_text_field( $_REQUEST['rcpga-last-name'] ) : '',
				'user_pass'  => ! empty( $_REQUEST['rcpga-user-password'] ) ? sanitize_text_field( $_REQUEST['rcpga-user-password'] ) : wp_generate_password( 24 ),
			) );

			if ( is_wp_error( $user_id ) ) {
				return 'default-error';
			}
		}

	} else {
		$user_id = get_current_user_id();
	}

	if ( empty( $user_id ) ) {
		return 'no-user';
	}

	// Single Memberships: Make sure the specified user does not already belong to a group.
	if ( ! rcp_multiple_memberships_enabled() && rcpga_user_is_group_member( $user_id ) ) {
		return 'has-group';
	}

	$args = array(
		'owner_id'    => absint( $user_id ),
		'name'        => wp_unslash( sanitize_text_field( $_REQUEST['rcpga-group-name'] ) ),
		'description' => ! empty( $_REQUEST['rcpga-group-description'] ) ? wp_unslash( wp_filter_post_kses( $_REQUEST['rcpga-group-description'] ) ) : '',
		'seats'       => ! empty( $_REQUEST['rcpga-group-seats'] ) ? absint( $_REQUEST['rcpga-group-seats'] ) : 0,
	);

	$group_id = rcpga_add_group( $args );

	if ( is_wp_error( $group_id ) ) {
		return 'default-error';
	}

	$redirect_args = array(
		'rcpga-group'   => urlencode( $group_id ),
		'rcpga-view'    => 'edit',
		'rcpga-message' => 'group-added'
	);

	wp_safe_redirect( Admin\get_groups_page( $redirect_args ) );
	exit;

}

/**
 * Process editing a group.
 *
 * @since 2.0
 * @return string|void
 */
function process_edit_group() {

	if ( empty( $_REQUEST['rcpga-action'] ) || 'edit-group' != $_REQUEST['rcpga-action'] ) {
		return;
	}

	// Make sure that the group is specified and exists.
	if ( empty( $_REQUEST['rcpga-group'] ) || ! $group = rcpga_get_group( absint( $_REQUEST['rcpga-group'] ) ) ) {
		return 'no-group';
	}

	// Check permissions.
	if ( ! rcpga_user_can( get_current_user_id(), 'manage_members', $group->get_group_id() ) ) {
		return 'no-permission';
	}

	// Nonce validation
	/*
	 * @todo This is disabled on the front-end for now because the nonce didn't exist in 1.x templates. We should enable it at some point.
	 */
	if ( is_admin() && ( empty( $_REQUEST['rcpga_edit_group_nonce'] ) || ! wp_verify_nonce( $_REQUEST['rcpga_edit_group_nonce'], 'rcpga_edit_group' ) ) ) {
		return 'default-error';
	}

	// Make sure group name is specified.
	if ( empty( $_REQUEST['rcpga-group-name'] ) ) {
		return 'empty-group-name';
	}

	// Make sure this code isn't already in use by another group.
	$code = ! empty( $_REQUEST['rcpga-group-code'] ) ? strtolower( $_REQUEST['rcpga-group-code'] ) : '';

	if ( strlen( $code ) > 255 ) {
		$code = substr( $code, 0, 255 );
	}

	if ( ! empty( $code ) && $group->get_code() != $code && rcpga_get_group_by( 'code', sanitize_text_field( $code ) ) ) {
		return 'code-unavailable';
	}

	$args = array(
		'name' => wp_unslash( sanitize_text_field( $_POST['rcpga-group-name'] ) )
	);

	if ( isset( $_REQUEST['rcpga-group-description'] ) ) {
		$args['description'] = wp_unslash( wp_filter_post_kses( $_REQUEST['rcpga-group-description'] ) );
	}

	if ( isset( $_REQUEST['rcpga-group-code'] ) ) {
		$args['code'] = wp_unslash( sanitize_text_field( $code ) );
	}

	// Seat number and membership can only be changed by admins.
	if ( current_user_can( 'rcp_view_members' ) && is_admin() && isset( $_REQUEST['rcpga-group-seats'] ) ) {
		$args['seats'] = absint( $_REQUEST['rcpga-group-seats'] );

		if ( ! empty( $_REQUEST['rcpga-group-membership-id'] ) ) {
			$args['membership_id'] = absint( $_REQUEST['rcpga-group-membership-id'] );
		}
	}

	$updated = $group->update( $args );

	if ( $updated ) {
		return 'group-updated';
	}

	return 'default-error';

}

/**
 * Process adding a group member.
 *
 * @since 2.0
 * @return string|void
 */
function process_add_group_member() {

	if ( empty( $_REQUEST['rcpga-action'] ) || 'add-member' != $_REQUEST['rcpga-action'] ) {
		return;
	}

	// Make sure that the group is specified and exists.
	if ( empty( $_REQUEST['rcpga-group'] ) || ! $group = rcpga_get_group( absint( $_REQUEST['rcpga-group'] ) ) ) {
		return 'no-group';
	}

	// Check permissions.
	if ( ! rcpga_user_can( get_current_user_id(), 'manage_members', $group->get_group_id() ) ) {
		return 'no-permission';
	}

	// Nonce validation
	/*
	 * @todo This is disabled on the front-end for now because the nonce didn't exist in 1.x templates. We should enable it at some point.
	 */
	if ( is_admin() && ( empty( $_REQUEST['rcpga_add_group_member_nonce'] ) || ! wp_verify_nonce( $_REQUEST['rcpga_add_group_member_nonce'], 'rcpga_add_group_member' ) ) ) {
		return 'default-error';
	}

	// Fail if group has no seats.
	if ( ! $group->can_add_members() ) {
		return 'seats-maxed';
	}

	global $rcp_options;

	$email = ! empty( $_REQUEST['rcpga-user-email'] ) ? $_REQUEST['rcpga-user-email'] : '';

	$args = array(
		'role'        => empty( $_REQUEST['rcpga-disable-invite-email'] ) && empty( $rcp_options['disable_group_invite_email'] ) ? 'invited' : 'member',
		'user_email'  => $email,
		'user_login'  => ! empty( $_REQUEST['rcpga-user-login'] ) ? sanitize_user( $_REQUEST['rcpga-user-login'] ) : sanitize_user( $email ),
		'first_name'  => isset( $_REQUEST['rcpga-first-name'] ) ? sanitize_text_field( $_REQUEST['rcpga-first-name'] ) : '',
		'last_name'   => isset( $_REQUEST['rcpga-last-name'] ) ? sanitize_text_field( $_REQUEST['rcpga-last-name'] ) : '',
		'user_pass'   => ! empty( $_REQUEST['rcpga-user-password'] ) ? sanitize_text_field( $_REQUEST['rcpga-user-password'] ) : wp_generate_password( 24 ),
		'send_invite' => empty( $_REQUEST['rcpga-disable-invite-email'] ) && empty( $rcp_options['disable_group_invite_email'] ),
	);

	$group_member_id = $group->add_member( $args );

	if ( is_wp_error( $group_member_id ) ) {
		return $group_member_id->get_error_code();
	}

	return 'group-member-added';

}

/**
 * Process importing group members
 *
 * @since 2.0
 * @return string|void
 */
function process_import_members_to_group() {

	if ( empty( $_REQUEST['rcpga-action'] ) || 'import-members' != $_REQUEST['rcpga-action'] ) {
		return;
	}

	if ( empty( $_REQUEST['rcpga-group'] ) || ! $group = rcpga_get_group( absint( $_REQUEST['rcpga-group'] ) ) ) {
		return 'no-group';
	}

	// Check permissions.
	if ( ! rcpga_user_can( get_current_user_id(), 'manage_members', $group->get_group_id() ) ) {
		return 'no-permission';
	}

	// Nonce validation
	/*
	 * @todo This is disabled on the front-end for now because the nonce didn't exist in 1.x templates. We should enable it at some point.
	 */
	if ( is_admin() && ( empty( $_REQUEST['rcpga_import_group_members_nonce'] ) || ! wp_verify_nonce( $_REQUEST['rcpga_import_group_members_nonce'], 'rcpga_import_group_members' ) ) ) {
		return 'default-error';
	}

	if ( empty( $_FILES['rcpga-group-csv']['tmp_name'] ) ) {
		return 'no-csv';
	}

	$import_file = ! empty( $_FILES['rcpga-group-csv'] ) ? $_FILES['rcpga-group-csv']['tmp_name'] : false;

	if ( ! $import_file ) {
		return 'default-error';
	}

	$members = array_map( 'str_getcsv', file( $import_file ) );
	array_walk( $members, function ( &$a ) use ( $members ) {
		$a = array_combine( $members[0], $a );
	} );
	array_shift( $members );

	$seats_left       = $group->get_seats() - $group->get_member_count();
	$members_imported = 0;

	if ( ! $members ) {
		return 'default-error';
	}

	$args = $errors = array();

	global $rcp_options;

	// do we send an invite email?
	$args['send_invite'] = empty( $_REQUEST['rcpga-disable-invite-email'] ) && empty( $rcp_options['disable_group_invite_email'] );
	$args['role']        = $args['send_invite'] ? 'invited' : 'member';

	// Disable new user invites for admins.
	if ( is_admin() ) {
		$rcp_options['disable_new_user_notices'] = true;
	}

	$i = 1;

	foreach ( $members as $member ) {

		// Skip if all fields are empty.
		if ( ! array_filter( $member ) ) {
			continue;
		}

		foreach ( $member as $member_key => $member_value ) {
			if ( substr( $member_key, 0, 3 ) == chr( hexdec( 'EF' ) ) . chr( hexdec( 'BB' ) ) . chr( hexdec( 'BF' ) ) ) {
				$sanitized_member_key = substr( $member_key, 3 );
				unset( $member[ $member_key ] );
				$member[ $sanitized_member_key ] = $member_value;
			}
		}

		// We check for new style column headers first, then the old style.

		if ( ! empty( $member['User Email'] ) ) {
			$args['user_email'] = sanitize_text_field( $member['User Email'] );
		} elseif ( ! empty( $member['email'] ) ) {
			$args['user_email'] = sanitize_text_field( $member['email'] );
		} else {
			$args['user_email'] = '';
		}

		// If we've reached the maximum number of seats - bail.
		if ( $members_imported >= $seats_left ) {
			$errors[] = sprintf( __( 'Row #%d (%s) not imported - seat limit reached.', 'rcp' ), $i, $args['user_email'] );
			continue;
		}

		if ( ! empty( $member['User Login'] ) ) {
			$args['user_login'] = sanitize_user( $member['User Login'] );
		} elseif ( ! empty( $member['user_login'] ) ) {
			$args['user_login'] = sanitize_user( $member['user_login'] );
		} else {
			$args['user_login'] = sanitize_user( $args['user_email'] );
		}

		if ( ! empty( $member['First Name'] ) ) {
			$args['first_name'] = sanitize_text_field( $member['First Name'] );
		} elseif ( ! empty( $member['first_name'] ) ) {
			$args['first_name'] = sanitize_text_field( $member['first_name'] );
		} else {
			$args['first_name'] = '';
		}

		if ( ! empty( $member['Last Name'] ) ) {
			$args['last_name'] = sanitize_text_field( $member['Last Name'] );
		} elseif ( ! empty( $member['last_name'] ) ) {
			$args['last_name'] = sanitize_text_field( $member['last_name'] );
		} else {
			$args['last_name'] = '';
		}

		if ( ! empty( $member['User Password'] ) ) {
			$args['user_pass'] = sanitize_text_field( $member['User Password'] );
		} elseif ( ! empty( $member['password'] ) ) {
			$args['user_pass'] = sanitize_text_field( $member['password'] );
		} else {
			$args['user_pass'] = wp_generate_password();
		}

		$group_member_id = $group->add_member( $args );
		$group_member    = ! is_wp_error( $group_member_id ) ? rcpga_get_group_member_by_id( $group_member_id ) : false;
		$user_id         = ! empty( $group_member ) ? $group_member->get_user_id() : $group_member_id;

		/**
		 * Triggers after this user has been imported.
		 *
		 * @param int|WP_Error                $user_id      User ID of the added user, or WP_Error.
		 * @param array                       $member       Row from the CSV file.
		 * @param RCPGA_Group_Member|WP_Error $group_member Group member object or WP_Error on failure.
		 */
		do_action( 'rcpga_import_member_to_group', $user_id, $member, $group_member );

		if ( is_wp_error( $group_member_id ) ) {
			$errors[] = sprintf( __( 'Row #%d (%s): %s', 'rcp-group-accounts' ), $i, $args['user_email'], $group_member_id->get_error_message() );
		} else {
			$members_imported ++;
		}

		$i ++;

	}

	/**
	 * Triggers after all members have been imported.
	 *
	 * @param RCPGA_Group $group Group object.
	 */
	do_action( 'rcpga_import_members_to_group', $group );

	update_option( 'rcpga_import_success_count_' . $group->get_group_id(), absint( $members_imported ) );

	if ( count( $errors ) ) {
		update_option( 'rcpga_import_errors_' . $group->get_group_id(), $errors );

		return 'group-members-imported-errors';
	} else {
		delete_option( 'rcpga_import_errors_' . $group->get_group_id() );
	}

	return 'group-members-imported';

}

/**
 * Process removing a group member
 *
 * @since 2.0
 * @return string|void
 */
function process_remove_member_from_group() {

	if ( empty( $_REQUEST['rcpga-action'] ) || 'remove-member' != $_REQUEST['rcpga-action'] ) {
		return;
	}

	if ( empty( $_REQUEST['_wpnonce'] ) || ! wp_verify_nonce( $_REQUEST['_wpnonce'], 'rcpga_remove_from_group' ) ) {
		return 'no-permission';
	}

	// First try to use group member ID.
	$group_member_id = ! empty( $_REQUEST['group-member-id'] ) ? absint( $_REQUEST['group-member-id'] ) : 0;

	// If we don't have group member ID, then we need a group and user ID.
	if ( empty( $group_member_id ) ) {
		if ( empty( $_REQUEST['rcpga-group'] ) ) {
			return 'no-group';
		}

		if ( empty( $_REQUEST['rcpga-member'] ) ) {
			return 'no-member';
		}

		$group_member = rcpga_get_group_member( absint( $_REQUEST['rcpga-member'] ), absint( $_REQUEST['rcpga-group'] ) );
	} else {
		$group_member = rcpga_get_group_member_by_id( $group_member_id );
	}

	if ( empty( $group_member ) ) {
		return 'default-error';
	}

	if ( ! rcpga_user_can( get_current_user_id(), 'manage_members', $group_member->get_group_id() ) ) {
		return 'no-permission';
	}

	$group_member->remove();

	return 'group-member-removed';

}

/**
 * Process updating a member's role to either "member" or "admin"
 *
 * @since 2.0
 * @return string|void
 */
function process_change_member_role() {

	if ( empty( $_REQUEST['rcpga-action'] ) || ! in_array( $_REQUEST['rcpga-action'], array(
			'make-admin',
			'make-member'
		) ) ) {
		return;
	}

	if ( empty( $_REQUEST['_wpnonce'] ) ) {
		return 'no-permission';
	}

	if ( 'make-admin' === $_REQUEST['rcpga-action'] && ! wp_verify_nonce( $_REQUEST['_wpnonce'], 'rcpga_make_admin' ) ) {
		return 'no-permission';
	}

	if ( 'make-member' === $_REQUEST['rcpga-action'] && ! wp_verify_nonce( $_REQUEST['_wpnonce'], 'rcpga_make_member' ) ) {
		return 'no-permission';
	}

	if ( ! rcpga_user_can( get_current_user_id(), 'manage_members', absint( $_REQUEST['rcpga-group'] ) ) ) {
		return 'no-permission';
	}

	// First try to use group member ID.
	$group_member_id = ! empty( $_REQUEST['group-member-id'] ) ? absint( $_REQUEST['group-member-id'] ) : 0;

	// If we don't have group member ID, then we need a group and user ID.
	if ( empty( $group_member_id ) ) {
		if ( empty( $_REQUEST['rcpga-group'] ) ) {
			return 'no-group';
		}

		if ( empty( $_REQUEST['rcpga-member'] ) ) {
			return 'no-member';
		}

		$group_member = rcpga_get_group_member( absint( $_REQUEST['rcpga-member'] ), absint( $_REQUEST['rcpga-group'] ) );
	} else {
		$group_member = rcpga_get_group_member_by_id( $group_member_id );
	}

	if ( empty( $group_member ) ) {
		return 'default-error';
	}

	$new_role = 'make-admin' == $_REQUEST['rcpga-action'] ? 'admin' : 'member';

	$group_member->set_role( $new_role );

	return 'group-member-updated';

}

/**
 * Process re-sending the group invite
 *
 * @since 2.0
 * @return string|void
 */
function process_resend_member_invite() {

	if ( empty( $_REQUEST['rcpga-action'] ) || 'resend-invite' != $_REQUEST['rcpga-action'] ) {
		return;
	}

	if ( empty( $_REQUEST['_wpnonce'] ) || ! wp_verify_nonce( $_REQUEST['_wpnonce'], 'rcpga_resend_invite' ) ) {
		return 'no-permission';
	}

	if ( ! rcpga_user_can( get_current_user_id(), 'manage_members', absint( $_REQUEST['rcpga-group'] ) ) ) {
		return 'no-permission';
	}

	// First try to use group member ID.
	$group_member_id = ! empty( $_REQUEST['group-member-id'] ) ? absint( $_REQUEST['group-member-id'] ) : 0;

	// If we don't have group member ID, then we need a group and user ID.
	if ( empty( $group_member_id ) ) {
		if ( empty( $_REQUEST['rcpga-group'] ) ) {
			return 'no-group';
		}

		if ( empty( $_REQUEST['rcpga-member'] ) ) {
			return 'no-member';
		}

		$group_member = rcpga_get_group_member( absint( $_REQUEST['rcpga-member'] ), absint( $_REQUEST['rcpga-group'] ) );
	} else {
		$group_member = rcpga_get_group_member_by_id( $group_member_id );
	}

	if ( empty( $group_member ) ) {
		return 'default-error';
	}

	$sent = $group_member->resend_invite();

	if ( ! $sent ) {
		return 'default-error';
	}

	return 'invite-sent';

}

/**
 * Process changing the group owner
 *
 * @since 2.0
 * @return string|void
 */
function process_change_owner() {

	if ( empty( $_REQUEST['rcpga-action'] ) || 'make-owner' != $_REQUEST['rcpga-action'] ) {
		return;
	}

	if ( empty( $_REQUEST['_wpnonce'] ) || ! wp_verify_nonce( $_REQUEST['_wpnonce'], 'rcpga_make_owner' ) ) {
		return 'no-permission';
	}

	if ( ! rcpga_user_can( get_current_user_id(), 'manage_members', absint( $_REQUEST['rcpga-group'] ) ) ) {
		return 'no-permission';
	}

	// First try to use group member ID.
	$group_member_id = ! empty( $_REQUEST['group-member-id'] ) ? absint( $_REQUEST['group-member-id'] ) : 0;

	// If we don't have group member ID, then we need a group and user ID.
	if ( empty( $group_member_id ) ) {
		if ( empty( $_REQUEST['rcpga-group'] ) ) {
			return 'no-group';
		}

		if ( empty( $_REQUEST['rcpga-member'] ) ) {
			return 'no-member';
		}

		$new_owner = rcpga_get_group_member( absint( $_REQUEST['rcpga-member'] ), absint( $_REQUEST['rcpga-group'] ) );
	} else {
		$new_owner = rcpga_get_group_member_by_id( $group_member_id );
	}

	if ( empty( $new_owner ) ) {
		return 'default-error';
	}

	$group              = $new_owner->get_group();
	$group_membership   = $group->get_membership();
	$current_owner_id   = $group->get_owner_id();
	$current_owner      = rcpga_get_group_member( $current_owner_id, $group->get_group_id() );
	$current_owner_user = get_userdata( $current_owner->get_user_id() );
	$new_owner_user     = get_userdata( $new_owner->get_user_id() );

	rcp_log( sprintf( 'Group Accounts: beginning changing group owner. Group ID: %d; Current Owner ID: %d; New Owner ID: %d.', $group->get_group_id(), $current_owner_id, $new_owner->get_user_id() ) );

	// Bail now if the old owner and new owner have the same ID.
	if ( absint( $current_owner_id ) == absint( $new_owner->get_user_id() ) ) {
		return 'default-error';
	}

	// Change owner ID in groups table.
	$group->update( array(
		'owner_id' => $new_owner->get_user_id()
	) );

	// Change roles in members table.
	$new_owner->set_role( 'owner' );
	$current_owner->set_role( 'admin' );

	$new_owner_customer = rcp_get_customer_by_user_id( $new_owner->get_user_id() );

	// Create new customer record.
	if ( empty( $new_owner_customer ) ) {
		$new_owner_customer_id = rcp_add_customer( array(
			'user_id' => $new_owner->get_user_id()
		) );

		if ( empty( $new_owner_customer_id ) ) {
			return 'group-ownership-change-error';
		}

		$new_owner_customer = rcp_get_customer( $new_owner_customer_id );
	}

	if ( empty( $new_owner_customer ) ) {
		return 'group-ownership-change-error';
	}

	// Copy membership data to new owner.
	$new_group_membership_id = $new_owner_customer->add_membership( array(
		'object_id'         => $group_membership->get_object_id(),
		'object_type'       => $group_membership->get_object_type(),
		'currency'          => $group_membership->get_currency(),
		'initial_amount'    => $group_membership->get_initial_amount(),
		'recurring_amount'  => $group_membership->get_recurring_amount(),
		'created_date'      => $group_membership->get_created_date( false ),
		'cancellation_date' => $group_membership->get_cancellation_date( false ),
		'expiration_date'   => $group_membership->get_expiration_date( false ),
		'auto_renew'        => $group_membership->is_recurring(),
		'times_billed'      => $group_membership->get_times_billed(),
		'maximum_renewals'  => $group_membership->get_maximum_renewals(),
		'status'            => $group_membership->get_status(),
	) );

	$new_group_membership = rcp_get_membership( $new_group_membership_id );

	// Cancel old group membership payment profile.
	if ( $group_membership->can_cancel() ) {
		$group_membership->cancel_payment_profile();
	} else {
		$group_membership->cancel();
	}
	$group_membership->add_note( sprintf( __( 'Membership cancelled and group ownership transferred to %s (membership ID #%d).', 'rcp-group-accounts' ), $new_owner_user->user_login, $new_group_membership_id ) );

	if ( ! empty( $new_group_membership ) ) {
		$new_group_membership->add_note( sprintf( __( 'Inherited group ownership and membership from previous group owner, %s.', 'rcp-group-accounts' ), $current_owner_user->user_login ) );
	}

	// Update membership_id in group record.
	$group->update( array(
		'membership_id' => absint( $new_group_membership_id )
	) );

	/**
	 * Triggers after the group owner is changed.
	 *
	 * @param int            $group_id             ID of the group.
	 * @param \WP_User       $current_owner_user   User object for the previous owner.
	 * @param \WP_User       $new_owner_user       User object for the new owner.
	 * @param RCP_Membership $group_membership     Membership for the previous owner.
	 * @param RCP_Membership $new_group_membership Membership for the new owner.
	 *
	 * @since 1.4
	 */
	do_action( 'rcpga_after_change_group_owner', $group->get_group_id(), $current_owner_user, $new_owner_user, $group_membership, $new_group_membership );

}

/**
 * Process leaving a group
 *
 * @since 2.0
 * @return string|void
 */
function process_leave_group() {

	if ( empty( $_REQUEST['rcpga-action'] ) || 'leave-group' != $_REQUEST['rcpga-action'] ) {
		return;
	}

	if ( empty( $_REQUEST['_wpnonce'] ) || ! wp_verify_nonce( $_REQUEST['_wpnonce'], 'rcpga_leave_group' ) ) {
		return 'no-permission';
	}

	global $rcp_options;

	$can_leave = ! empty( $rcp_options['groups_allow_member_removal'] );

	if ( ! $can_leave ) {
		return 'no-permission';
	}

	// First try to use group member ID.
	$group_member_id = ! empty( $_REQUEST['group-member-id'] ) ? absint( $_REQUEST['group-member-id'] ) : 0;

	// If we don't have group member ID, then we need a group and user ID.
	if ( empty( $group_member_id ) ) {
		if ( empty( $_REQUEST['rcpga-group'] ) ) {
			return 'no-group';
		}

		if ( empty( $_REQUEST['rcpga-member'] ) ) {
			return 'no-member';
		}

		$group_member = rcpga_get_group_member( absint( $_REQUEST['rcpga-member'] ), absint( $_REQUEST['rcpga-group'] ) );
	} else {
		$group_member = rcpga_get_group_member_by_id( $group_member_id );
	}

	if ( empty( $group_member ) ) {
		return 'default-error';
	}

	// If the current user doesn't match this group member, bail.
	if ( get_current_user_id() != $group_member->get_user_id() ) {
		return 'no-permission';
	}

	$group_member->remove();

	return 'removed-from-group';

}

/**
 * Process deleting a group
 *
 * @since 2.0
 * @return string|void
 */
function process_delete_group() {

	if ( empty( $_REQUEST['rcpga-action'] ) || 'delete-group' != $_REQUEST['rcpga-action'] ) {
		return;
	}

	if ( empty( $_REQUEST['_wpnonce'] ) || ! wp_verify_nonce( $_REQUEST['_wpnonce'], 'rcpga_delete_group_nonce' ) ) {
		return 'no-permission';
	}

	$group_id = ! empty( $_REQUEST['rcpga-group'] ) ? absint( $_REQUEST['rcpga-group'] ) : 0;

	if ( empty( $group_id ) ) {
		return 'no-group';
	}

	$group = rcpga_get_group( $group_id );

	if ( empty( $group ) ) {
		return 'no-group';
	}

	if ( ! current_user_can( 'manage_options' ) ) {
		return 'no-permission';
	}

	$group->delete();

	return 'group-deleted';

}

/**
 * After a group is created, automatically add the owner to the group as a member.
 *
 * @param int   $group_id ID of the group.
 * @param array $args     Arguments used to create the group.
 *
 * @since 2.0
 * @return void
 */
function auto_add_owner_to_group( $group_id, $args ) {

	// We're using a more direct DB query here to bypass checks for existing memberships, etc.
	$query = new \RCP\Database\Queries\Group_Members();

	$query->add_item( array(
		'group_id' => $group_id,
		'user_id' => absint( $args['owner_id'] ),
		'role'    => 'owner'
	) );

	$group = rcpga_get_group( $group_id );

	if ( $group ) {
		$group->update_member_count();
	}


}

add_action( 'rcpga_db_groups_post_insert', __NAMESPACE__ . '\auto_add_owner_to_group', 10, 2 );

/**
 * Re-count the number of group members after a new one is added
 *
 * @param int   $user_id  ID of the user who was just added.
 * @param array $args     Arguments used while adding the member.
 * @param int   $group_id ID of the group the user was added to.
 *
 * @since 2.0
 * @return void
 */
function auto_update_group_member_count( $user_id, $args, $group_id ) {

	$group = rcpga_get_group( $group_id );

	if ( empty( $group ) ) {
		return;
	}

	$group->update_member_count();

}

add_action( 'rcpga_add_member_to_group_after', __NAMESPACE__ . '\auto_update_group_member_count', 10, 3 );

/**
 * If multiple memberships is NOT enabled, we need to ensure each user can only be in one group at a time.
 * When a user is added to a new group, remove them from all other groups.
 *
 * @param int   $user_id  ID of the user who was just added.
 * @param array $args     Arguments used while adding the member.
 * @param int   $group_id ID of the group the user was added to.
 *
 * @since 2.0
 * @return void
 */
function maybe_remove_from_groups( $user_id, $args, $group_id ) {

	if ( rcp_multiple_memberships_enabled() ) {
		return;
	}

	$group_memberships = rcpga_get_group_members( array(
		'number'  => 9999,
		'user_id' => $user_id
	) );

	if ( empty( $group_memberships ) ) {
		return;
	}

	foreach ( $group_memberships as $group_membership ) {
		/**
		 * @var RCPGA_Group_Member $group_membership
		 */

		// Skip the new group.
		if ( $group_id == $group_membership->get_group_id() ) {
			continue;
		}

		// Remove from all other groups.
		$group_membership->remove();
	}

}

add_action( 'rcpga_add_member_to_group_after', __NAMESPACE__ . '\maybe_remove_from_groups', 10, 3 );

/**
 * When a group is deleted, remove all members in the group.
 *
 * @since 2.0
 */
add_action( 'rcpga_db_groups_post_delete', 'rcpga_delete_all_members_of_group', 10 );

/**
 * Process group creation form.
 *
 * @since 2.1
 */
function create_membership_group() {

	if ( empty( $_POST['rcpga_create_membership_group_nonce'] ) ) {
		return;
	}

	if ( ! wp_verify_nonce( $_POST['rcpga_create_membership_group_nonce'], 'rcpga_create_membership_group' ) ) {
		rcp_errors()->add( 'validation_failed', __( 'Validation failed. Please try again.', 'rcp-group-accounts' ), 'rcpga_create_group' );

		return;
	}

	$membership = ! empty( $_POST['rcpga_membership_id'] ) ? rcp_get_membership( absint( $_POST['rcpga_membership_id'] ) ) : false;
	if ( ! $membership instanceof RCP_Membership ) {
		rcp_errors()->add( 'invalid_membership', __( 'Invalid membership', 'rcp-group-accounts' ), 'rcpga_create_group' );

		return;
	}

	/*
	 * Check membership requirements:
	 *
	 * 		- Current user is the owner of the supplied membership.
	 * 		- Membership is active.
	 * 		- There's no associated group.
	 * 		- The membership level ID is Group Accounts-enabled.
	 */
	if (
		get_current_user_id() != $membership->get_customer()->get_user_id() ||
		! $membership->is_active() ||
		rcpga_get_group_by( 'membership_id', $membership->get_id() ) ||
		! rcpga_is_level_group_accounts_enabled( $membership->get_object_id() )
	) {
		rcp_errors()->add( 'requirements_not_met', __( 'You do not have permission to perform this action.', 'rcp-group-accounts' ), 'rcpga_create_group' );

		return;
	}

	// Group name is required.
	if ( empty( $_POST['rcpga-group-name'] ) ) {
		rcp_errors()->add( 'group_name_required', __( 'Please enter a group name', 'rcp-group-accounts' ), 'rcpga_create_group' );

		return;
	}

	$group_id = rcpga_add_group( array(
		'owner_id'      => $membership->get_customer()->get_user_id(),
		'membership_id' => $membership->get_id(),
		'name'          => wp_unslash( sanitize_text_field( $_POST['rcpga-group-name'] ) ),
		'description'   => ! empty( $_POST['rcpga-group-description'] ) ? wp_unslash( wp_filter_post_kses( $_POST['rcpga-group-description'] ) ) : '',
		'seats'         => rcpga_get_level_group_seats_allowed( $membership->get_object_id() )
	) );

	if ( is_wp_error( $group_id ) ) {
		rcp_errors()->add( $group_id->get_error_code(), $group_id->get_error_message(), 'rcpga_create_group' );

		return;
	}

	global $rcp_options;

	if ( ! empty( $rcp_options['rcpga_dashboard_page'] ) ) {
		$redirect_url = add_query_arg( array(
			'rcpga-group'   => urlencode( $group_id ),
			'rcpga-message' => 'group-added'
		), get_permalink( $rcp_options['rcpga_dashboard_page'] ) );

		wp_safe_redirect( esc_url_raw( $redirect_url ) );
		exit;
	}

}

add_action( 'init', __NAMESPACE__ . '\create_membership_group' );