<?php
/**
 * Group Object
 *
 * @package   rcp-group-accounts
 * @copyright Copyright (c) 2019, Restrict Content Pro
 * @license   GPL2+
 * @since     2.0
 */

// Exit if accessed directly
defined( 'ABSPATH' ) || exit;

class RCPGA_Group extends RCP\Base_Object {

	/**
	 * @var int
	 */
	protected $group_id = 0;

	/**
	 * @var int
	 */
	protected $owner_id = 0;

	/**
	 * @var int
	 */
	protected $membership_id = 0;

	/**
	 * @var string
	 */
	protected $name = '';

	/**
	 * @var string
	 */
	protected $description = '';

	/**
	 * @var string
	 */
	protected $code = '';

	/**
	 * @var int
	 */
	protected $seats = 0;

	/**
	 * @var int
	 */
	protected $member_count = 0;

	/**
	 * @var string
	 */
	protected $date_created = '';

	/**
	 * Magic getter to retrieve protected properties.
	 *
	 * @param string $key
	 *
	 * @since 2.0
	 * @return mixed|WP_Error
	 */
	public function __get( $key ) {

		$key = sanitize_key( $key );

		if ( method_exists( $this, 'get_' . $key ) ) {
			return call_user_func( array( $this, 'get_' . $key ) );
		} elseif ( property_exists( $this, $key ) ) {
			return $this->{$key};
		} else {
			return new WP_Error( 'rcpga-group-invalid-property', sprintf( __( 'Can\'t get property %s', 'rcp-group-accounts' ), $key ) );
		}

	}

	/**
	 * Get the group ID
	 *
	 * @since 2.0
	 * @return int
	 */
	public function get_group_id() {
		return absint( $this->group_id );
	}

	/**
	 * Get the owner ID
	 *
	 * @since 2.0
	 * @return int
	 */
	public function get_owner_id() {
		return absint( $this->owner_id );
	}

	/**
	 * Get the WP_User object for the group owner
	 *
	 * @since 2.0
	 * @return WP_User|false
	 */
	public function get_owner() {
		return get_userdata( $this->get_owner_id() );
	}

	/**
	 * Get the membership ID this group is associated with
	 *
	 * @since 2.0
	 * @return int
	 */
	public function get_membership_id() {

		$membership_id = absint( $this->membership_id );

		if ( ! empty( $membership_id ) ) {
			return $membership_id;
		}

		// Backfill membership ID for RCP 3.0+.

		if ( version_compare( RCP_PLUGIN_VERSION, '3.0', '<' ) ) {
			return $membership_id;
		}

		$owner_id = $this->get_owner_id();
		$customer = rcp_get_customer_by_user_id( $owner_id );

		// Bail if we don't have a customer for some reason.
		if ( empty( $customer ) ) {
			return $membership_id;
		}

		$membership = rcp_get_customer_single_membership( $customer->get_id() );

		if ( ! empty( $membership ) ) {
			$membership_id = $membership->get_id();

			$this->update( array(
				'membership_id' => absint( $membership_id )
			) );
		}

		return $membership_id;

	}

	/**
	 * Get the membership object this group is associated with
	 *
	 * @since 2.0
	 * @return RCP_Membership|false
	 */
	public function get_membership() {

		$membership_id = $this->get_membership_id();

		if ( empty( $membership_id ) ) {
			return false;
		}

		return rcp_get_membership( $membership_id );
	}

	/**
	 * Determines whether or not the group is active
	 *
	 * A group is active if the associated membership is active.
	 *
	 * @since 2.0
	 * @return bool
	 */
	public function is_active() {

		$membership = $this->get_membership();

		if ( empty( $membership ) ) {
			return false;
		}

		return $membership->is_active();

	}

	/**
	 * Gets the user role granted by this group's membership
	 *
	 * @since 2.0
	 * @return string|false Role slug on success, false if none can be determined.
	 */
	public function get_membership_role() {

		$membership = $this->get_membership();

		if ( empty( $membership ) ) {
			return false;
		}

		$membership_level_id = $membership->get_object_id();
		$membership_level    = rcp_get_subscription_details( $membership_level_id );

		if ( empty( $membership_level ) ) {
			return false;
		}

		return $membership_level->role;

	}

	/**
	 * Get the name of the group
	 *
	 * @since 2.0
	 * @return string
	 */
	public function get_name() {
		return $this->name;
	}

	/**
	 * Get the group description
	 *
	 * @since 2.0
	 * @return string
	 */
	public function get_description() {
		return $this->description;
	}

	/**
	 * Get the group code
	 *
	 * @since 2.1
	 * @return string
	 */
	public function get_code() {
		return $this->code;
	}

	/**
	 * Get the maximum number of seats available to the group
	 *
	 * @since 2.0
	 * @return int
	 */
	public function get_seats() {
		return absint( $this->seats );
	}

	/**
	 * Get the current member count
	 *
	 * @since 2.0
	 * @return int
	 */
	public function get_member_count() {
		return absint( $this->member_count );
	}

	/**
	 * Get the date the group was created
	 *
	 * @param bool $formatted Whether or not the returned date should be formatted for display.
	 *
	 * @since 2.0
	 * @return string
	 */
	public function get_date_created( $formatted = false ) {

		$date_created = $this->date_created;

		if ( $formatted ) {
			$date_created = date_i18n( get_option( 'date_format' ), strtotime( $date_created, current_time( 'timestamp' ) ) );
		}

		return $date_created;

	}

	/**
	 * Get the members of this group.
	 *
	 * @param array $args
	 *
	 * @since 2.0
	 * @return array An array of RCPGA_Group_Member objects.
	 */
	public function get_members( $args = array() ) {
		$args = wp_parse_args( $args, array(
			'group_id' => $this->get_group_id(),
			'number'   => 99999 // Get them all by default
		) );

		return rcpga_get_group_members( $args );
	}

	/**
	 * Get a specific member's record in this group.
	 *
	 * @param int $user_id
	 *
	 * @since 2.0
	 * @return RCPGA_Group_Member|false
	 */
	public function get_member( $user_id ) {
		return rcpga_get_group_member( $user_id, $this->get_group_id() );
	}

	/**
	 * Update the group
	 *
	 * @param array $data Array of data to update.
	 *
	 * @since 2.0
	 * @return bool
	 */
	public function update( $data = array() ) {

		$query = new \RCP\Database\Queries\Groups();

		return $query->update_item( $this->get_group_id(), $data );

	}

	/**
	 * Delete the group
	 *
	 * @since 2.0
	 * @return bool
	 */
	public function delete() {

		$query = new \RCP\Database\Queries\Groups();

		$result = $query->delete_item( $this->get_group_id() );

		/**
		 * Triggers after a group is deleted.
		 *
		 * @param int $group_id
		 */
		do_action( 'rcpga_db_groups_post_delete', $this->get_group_id() );

		return $result;

	}

	/**
	 * Update the member count
	 *
	 * @since 2.0
	 * @return void
	 */
	public function update_member_count() {

		$count = rcpga_count_group_members( array(
			'group_id' => $this->get_group_id()
		) );

		$this->update( array(
			'member_count' => absint( $count )
		) );

	}

	/**
	 * Whether or not members can be added to the group. Will return true if:
	 *      - Seats are available; AND
	 *      - The associated membership is active.
	 *
	 * @since 2.0
	 * @return bool
	 */
	public function can_add_members() {

		return $this->get_seats() > $this->get_member_count();

	}

	/**
	 * Add a new member to this group
	 *
	 * @param array $args        {
	 *                           Array of group member arguments.
	 *
	 * @type int    $user_id     Optional. ID of the user account. If omitted, `user_email` must be supplied.
	 * @type string $user_email  Optional. Email address of the group member. If provided, a new account will be
	 *                           created with this email. This is required if `user_id` is omitted.
	 * @type string $user_login  Optional. New member's username. Used in newa ccount creation. Will fall back to the
	 *                           email address if not provided.
	 * @type string $first_name  Optional. First name of the member. Used in new account creation.
	 * @type string $last_name   Optional. Last name of the member. Used in new account creation.
	 * @type string $user_pass   Optional. Password for the new member account. Default is randomly generated.
	 * @type string $role        Optional. Membership role: `owner`, `admin`, `member`, or `invited`. Default is
	 *                           `invited`.
	 * @type bool   $send_invite Optional. Whether or not to send the invitation email. Defaults to true.
	 * @type string $date_added  Optional. Date the member was added in MySQL format. Default is now.
	 *                    }
	 *
	 * @since 2.0
	 * @return int|WP_Error ID of the group member on success, WP_Error on failure.
	 */
	public function add_member( $args ) {

		$args = wp_parse_args( $args, array(
			'user_id'     => 0,
			'user_email'  => '',
			'user_login'  => '',
			'first_name'  => '',
			'last_name'   => '',
			'user_pass'   => wp_generate_password( 24 ),
			'role'        => 'invited',
			'send_invite' => true,
			'date_added'  => current_time( 'mysql' )
		) );

		$args['group_id'] = $this->get_group_id();

		/**
		 * Filters the arguments used for adding the group member.
		 *
		 * @param array $args
		 */
		$args = apply_filters( 'rcpga_invite_user_args', $args );

		if ( empty( $args['group_id'] ) ) {
			return new WP_Error( 'no_group', __( 'Please specify a group ID.', 'rcp-group-account' ) );
		}

		/**
		 * Create new user account if one doesn't already exist.
		 */
		$user_id = $args['user_id'];

		// Try to get existing user by email.
		if ( empty( $user_id ) && ! empty( $args['user_email'] ) ) {
			$user    = get_user_by( 'email', $args['user_email'] );
			$user_id = ! empty( $user ) ? $user->ID : 0;
		}

		if ( empty( $user_id ) ) {
			// Verify required fields.
			if ( empty( $args['user_email'] ) ) {
				return new WP_Error( 'empty-email', __( 'Please enter a valid email address.', 'rcp-group-account' ) );
			}

			if ( empty( $args['user_login'] ) ) {
				$args['user_login'] = $args['user_email'];
			}

			remove_action( 'user_register', 'rcp_user_register_add_subscription_level' );
			$user_id = wp_insert_user( $args );
			add_action( 'user_register', 'rcp_user_register_add_subscription_level' );
		}

		// We need a user ID.
		if ( is_wp_error( $user_id ) ) {
			return $user_id;
		} elseif ( empty( $user_id ) ) {
			return new WP_Error( 'invalid_user_id', __( 'Invalid user ID.', 'rcp-group-accounts' ) );
		}

		/**
		 * Perform permission checks to make sure this user can be added to a group.
		 */
		$can_add = true;

		if ( ! rcp_multiple_memberships_enabled() ) {
			// If multiple memberships is not enabled, then this user cannot already hold their own membership.
			if ( rcp_user_has_paid_membership( $user_id ) ) {
				$can_add = new WP_Error( 'active-user', __( 'Members with an active membership cannot be added to a group.', 'rcp-group-account' ) );
			} elseif ( rcpga_user_is_group_member( $user_id ) ) {
				// If multiple memberships is not enabled, then this user must not belong to another group.
				$can_add = new WP_Error( 'has-group', __( 'This user is already in a group. Users may only be part of one group at a time.', 'rcp-group-account' ) );
			}
		}

		/**
		 * Filters whether or not this user can be added to the group.
		 *
		 * @param bool|WP_Error $can_add Whether the user can be added to the group.
		 * @param int           $user_id ID of the user trying to be added.
		 * @param array         $args    Arguments used.
		 * @param RCPGA_Group   $this    Group object.
		 */
		$can_add = apply_filters( 'rcpga_can_add_user_to_group', $can_add, $user_id, $args, $this );

		if ( false === $can_add ) {
			return new WP_Error( 'member-not-added', __( 'This member could not be added.', 'rcp-group-accounts' ) );
		} elseif ( is_wp_error( $can_add ) ) {
			return new WP_Error( $can_add->get_error_code(), $can_add->get_error_message() );
		}

		/**
		 * Insert the member record into the database.
		 */
		$member_args = array(
			'user_id'    => $user_id,
			'group_id'   => $this->get_group_id(),
			'role'       => $args['role'],
			'date_added' => $args['date_added']
		);

		/**
		 * Filters the arguments used for inserting the group member into the database.
		 *
		 * @param array $member_args
		 */
		$member_args = apply_filters( 'rcpga_add_member_to_group_args', $member_args );

		$groups_query = new RCP\Database\Queries\Group_Members();

		$group_member_id = $groups_query->add_item( $member_args );

		if ( ! $group_member_id ) {
			return new WP_Error( 'group_member_insert_error', __( 'Failed to add new group member to the database.', 'rcp-group-accounts' ) );
		}

		/**
		 * Triggers after the new member has been added. This handles the following events:
		 *      - Sending the group invite (if set in $args). - maybe_send_member_invite()
		 *      - Updates the group member count. - auto_update_group_member_count()
		 *      - Maybe remove the user from all other groups. - maybe_remove_from_groups()
		 *
		 * @param int   $user_id  ID of the new user who was added.
		 * @param array $args     Arguments used when adding the user.
		 * @param int   $group_id ID of the group the user was added to.
		 */
		do_action( 'rcpga_add_member_to_group_after', $user_id, $args, $this->get_group_id() );

		return $group_member_id;

	}

}