<?php
/**
 * Group Member Object
 *
 * @package   rcp-group-accounts
 * @copyright Copyright (c) 2019, Restrict Content Pro team
 * @license   GPL2+
 * @since     2.0
 */

// Exit if accessed directly
defined( 'ABSPATH' ) || exit;

class RCPGA_Group_Member extends RCP\Base_Object {

	/**
	 * @var int
	 */
	protected $id = 0;

	/**
	 * @var int
	 */
	protected $user_id = 0;

	/**
	 * @var int
	 */
	protected $group_id = 0;

	/**
	 * @var string
	 */
	protected $role = '';

	/**
	 * @var string
	 */
	protected $activation_key = '';

	/**
	 * @var string
	 */
	protected $date_added = '';

	/**
	 * Magic getter to retrieve protected properties.
	 *
	 * @param string $key
	 *
	 * @since 2.0
	 * @return mixed|WP_Error
	 */
	public function __get( $key ) {

		$key = sanitize_key( $key );

		if ( method_exists( $this, 'get_' . $key ) ) {
			return call_user_func( array( $this, 'get_' . $key ) );
		} elseif ( property_exists( $this, $key ) ) {
			return $this->{$key};
		} else {
			return new WP_Error( 'rcpga-group-member-invalid-property', sprintf( __( 'Can\'t get property %s', 'rcp-group-accounts' ), $key ) );
		}

	}

	/**
	 * Get the member's unique ID
	 *
	 * @since 2.0
	 * @return int
	 */
	public function get_id() {
		return absint( $this->id );
	}

	/**
	 * Get the member's user ID number
	 *
	 * @since 2.0
	 * @return int
	 */
	public function get_user_id() {
		return absint( $this->user_id );
	}

	/**
	 * Get the group ID number
	 *
	 * @since 2.0
	 * @return int
	 */
	public function get_group_id() {
		return absint( $this->group_id );
	}

	/**
	 * Get the group object
	 *
	 * @since 2.0
	 * @return RCPGA_Group|false
	 */
	public function get_group() {
		return rcpga_get_group( $this->get_group_id() );
	}

	/**
	 * Get the member's role within the group
	 *
	 * @since 2.0
	 * @return string
	 */
	public function get_role() {

		$role = $this->role;

		/**
		 * Filters the member's role in the group.
		 *
		 * @param string             $role    Member's role in the group.
		 * @param int                $user_id ID of the user.
		 * @param RCPGA_Group_Member $this    Group member object.
		 */
		return apply_filters( 'rcpga_get_role', $role, $this->get_user_id(), $this );

	}

	/**
	 * Retrieves the member's activation key (hashed), prefixed with the request timestamp
	 *
	 * Format: {request timestamp}:{hashed key}
	 *
	 * @since 2.1.1
	 * @return string
	 */
	public function get_activation_key() {
		return $this->activation_key;
	}

	/**
	 * Get the date this member was added to the group
	 *
	 * @param bool $formatted Whether or not the returned date should be formatted for display.
	 *
	 * @since 2.0
	 * @return string
	 */
	public function get_date_added( $formatted = false ) {

		$date_added = $this->date_added;

		if ( $formatted ) {
			$date_added = date_i18n( get_option( 'date_format' ), strtotime( $date_added, current_time( 'timestamp' ) ) );
		}

		return $date_added;

	}

	/**
	 * Update a group member.
	 *
	 * @param array $data Data to update.
	 *
	 * @since 2.0
	 * @return bool
	 */
	public function update( $data = array() ) {

		$query = new \RCP\Database\Queries\Group_Members();

		return $query->update_item( $this->get_id(), $data );

	}

	/**
	 * Change the group member's role.
	 *
	 * @param string $new_role New role to set.
	 *
	 * @since 2.0
	 * @return void
	 */
	public function set_role( $new_role ) {

		// Update DB record.
		$this->update( array(
			'role' => $new_role
		) );

		/**
		 * Above function triggers this action hook:
		 * `rcp_transition_group_member_role`
		 *
		 * We use this hook to:
		 *
		 * 1. Set the member's user role to match the owner's.
		 * @see maybe_set_group_member_user_role()
		 *
		 * 2. Trigger the new user notification if applicable.
		 * @see maybe_send_new_user_notification()
		 */

	}

	/**
	 * Updates the member's user account "role" to match the group owner's role, which they were granted
	 * by the membership level.
	 *
	 * @since 2.0
	 */
	public function set_membership_role() {

		$user = get_userdata( $this->get_user_id() );
		$role = $this->get_group()->get_membership_role();

		/**
		 * Filters the role granted to this user for this group.
		 *
		 * @param string $role     Role to be granted.
		 * @param int    $user_id  ID of the user the role is about to be applied to.
		 * @param int    $group_id ID of the group.
		 */
		$role = apply_filters( 'rcpga_group_account_role', $role, $this->get_user_id(), $this->get_group_id() );

		// Bail if the user already has this role.
		if ( in_array( $role, $user->roles ) ) {
			return;
		}

		$user->add_role( $role );

	}

	/**
	 * Removes the member from this group
	 *
	 * @since 2.0
	 * @return bool
	 */
	public function remove() {

		/**
		 * Triggers before the member is removed from the group.
		 *
		 * @param int $user_id  ID of the user being removed.
		 * @param int $group_id ID of the group the user is being removed from.
		 */
		do_action( 'rcpga_db_group_members_pre_delete', $this->get_user_id(), $this->get_group_id() );

		/**
		 * Remove the group-granted role.
		 */
		$user       = get_userdata( $this->get_user_id() );
		$group_role = $this->get_group()->get_membership_role();

		/**
		 * Filters the role granted to this user for this group.
		 *
		 * @param string $group_role Role to be granted.
		 * @param int    $user_id    ID of the user the role is about to be applied to.
		 * @param int    $group_id   ID of the group.
		 */
		$group_role = apply_filters( 'rcpga_group_account_role', $group_role, $this->get_user_id(), $this->get_group_id() );

		$default_role = get_option( 'default_role', 'subscriber' );

		if ( $default_role != $group_role && ! empty( $user ) ) {
			$user->remove_role( $group_role );
			$user->add_role( $default_role );
		}

		/**
		 * Now delete the group member.
		 */
		$query   = new \RCP\Database\Queries\Group_Members();
		$deleted = $query->delete_item( $this->get_id() );

		/**
		 * Triggers after a member is removed from the group.
		 *
		 * @param int $user_id  ID of the user being removed.
		 * @param int $group_id ID of the group the user is being removed from.
		 */
		do_action( 'rcpga_remove_member', $this->get_user_id(), $this->get_group_id() );

		$this->get_group()->update_member_count();

		return ! empty( $deleted );

	}

	/**
	 * Determines whether or not this group member has a certain capability.
	 *
	 * @param string $capability Capability to check for.
	 *
	 * @since 2.0
	 * @return bool
	 */
	public function can( $capability ) {
		return rcpga_role_can( $this->get_role(), $capability );
	}

	/**
	 * Re-send the invitation email
	 *
	 * @since 2.0
	 * @return bool
	 */
	public function resend_invite() {

		// We shouldn't be resending invites to full members.
		if ( 'invited' != $this->get_role() ) {
			return false;
		}

		return rcpga_send_group_invite( $this->get_user_id(), $this->get_group_id() );

	}

}