<?php
/**
 * Group Members Table
 *
 * @package   rcp-group-accounts
 * @copyright Copyright (c) 2019, Restrict Content Pro team
 * @license   GPL2+
 * @since     2.0
 */

namespace RCP\Database\Tables;

// Exit if accessed directly
defined( 'ABSPATH' ) || exit;

use RCP\Database\Table;

/**
 * Set up the "rcp_group_members" database table.
 *
 * @package RCP\Database\Tables
 * @since   2.0
 */
final class Group_Members extends Table {

	/**
	 * @var string Table name
	 */
	protected $name = 'group_members';

	/**
	 * @var int Current database version
	 */
	protected $version = 202005011;

	/**
	 * @var array Array of upgrade versions and methods
	 */
	protected $upgrades = array(
		'201905231' => 201905231,
		'201905232' => 201905232,
		'201905233' => 201905233,
		'202005011' => 202005011,
	);

	/**
	 * Groups constructor.
	 *
	 * @access public
	 * @since  2.0
	 * @return void
	 */
	public function __construct() {
		parent::__construct();
	}

	/**
	 * Setup the database schema
	 *
	 * @access protected
	 * @since  2.0
	 * @return void
	 */
	protected function set_schema() {
		$this->schema = "id bigint(20) unsigned NOT NULL AUTO_INCREMENT,
			user_id bigint(20) unsigned NOT NULL,
			group_id bigint(20) unsigned NOT NULL DEFAULT '0',
			role tinytext NOT NULL,
			activation_key varchar(255) NOT NULL,
			date_added datetime NOT NULL,
			PRIMARY KEY (id),
			UNIQUE KEY user_id_group_id (user_id, group_id)";
	}

	/**
	 * If the old `wp_rcp_group_members_db_version` option exists, copy that value to our new version key.
	 * This will ensure new upgrades are processed on old installs.
	 *
	 * @since 2.0.3
	 */
	public function maybe_upgrade() {

		$group_members_db_version = get_option( $this->get_table_name() . '_db_version' );

		if ( ! empty( $group_members_db_version ) ) {
			update_option( $this->db_version_key, $group_members_db_version );

			delete_option( $this->get_table_name() . '_db_version' );
		}

		return parent::maybe_upgrade();

	}

	/**
	 * Upgrade to version 201905231
	 *      - Add `unsigned` to `user_id` and `group_id`
	 *
	 * @since 2.0
	 */
	protected function __201905231() {

		$result = $this->get_db()->query( "ALTER TABLE {$this->table_name} MODIFY user_id bigint(20) unsigned NOT NULL" );
		$result = $this->get_db()->query( "ALTER TABLE {$this->table_name} MODIFY group_id bigint(20) unsigned NOT NULL" );

		$success = $this->is_success( $result );

		if ( $success ) {
			rcp_log( sprintf( 'Group Accounts %s table upgrade to 201905231 successful.', $this->get_table_name() ) );
		} else {
			rcp_log( sprintf( 'Group Accounts %s table upgrade to 201905231 failure.', $this->get_table_name() ) );
		}

		return $success;

	}

	/**
	 * Upgrade to version 201905232
	 *      - Drop user_id primary key
	 *      - Add new column `id`.
	 *
	 * We have to do these in the same upgrade to avoid database errors.
	 *
	 * @return bool
	 */
	protected function __201905232() {

		// Look for column
		$result = $this->column_exists( 'id' );

		// Add column if it doesn't exist.
		if ( false === $result ) {
			$result = $this->get_db()->query( "ALTER TABLE {$this->table_name} DROP PRIMARY KEY, ADD COLUMN id bigint(20) unsigned NOT NULL AUTO_INCREMENT PRIMARY KEY FIRST;" );
		}

		$success = $this->is_success( $result );

		if ( $success ) {
			rcp_log( sprintf( 'Group Accounts %s table upgrade to 201905232 successful.', $this->get_table_name() ) );
		} else {
			rcp_log( sprintf( 'Group Accounts %s table upgrade to 201905232 failure.', $this->get_table_name() ) );
		}

		return $success;

	}

	/**
	 * Upgrade to version 201905233
	 *      - Add new unique key user_id,group_id
	 *
	 * @since 2.0
	 */
	protected function __201905233() {

		$has_index = $this->get_db()->query( "SHOW INDEX FROM {$this->table_name} WHERE Key_name = 'user_id_group_id'" );

		if ( ! $has_index ) {
			$added = $this->get_db()->query( "ALTER TABLE {$this->table_name} ADD UNIQUE KEY user_id_group_id(user_id, group_id)" );
		} else {
			$added = true;
		}

		$success = $this->is_success( $added );

		if ( $success ) {
			rcp_log( sprintf( 'Group Accounts %s table upgrade to 201905233 successful.', $this->get_table_name() ) );
		} else {
			rcp_log( sprintf( 'Group Accounts %s table upgrade to 201905233 failure.', $this->get_table_name() ) );
		}

		return $success;

	}

	/**
	 * Upgrade to version 202005011
	 *      - Add new column `activation_key`
	 *
	 * @return bool
	 */
	protected function __202005011() {

		// Look for column
		$result = $this->column_exists( 'activation_key' );

		// Add column if it doesn't exist.
		if ( false === $result ) {
			$result = $this->get_db()->query( "ALTER TABLE {$this->table_name} ADD COLUMN activation_key varchar(255) NOT NULL AFTER role;" );
		}

		$success = $this->is_success( $result );

		if ( $success ) {
			rcp_log( sprintf( 'Group Accounts %s table upgrade to 202005011 successful.', $this->get_table_name() ) );
		} else {
			rcp_log( sprintf( 'Group Accounts %s table upgrade to 202005011 failure.', $this->get_table_name() ) );
		}

		return $success;

	}

}