<?php
/**
 * Groups Table
 *
 * @package   rcp-group-accounts
 * @copyright Copyright (c) 2019, Restrict Content Pro team
 * @license   GPL2+
 * @since     2.0
 */

namespace RCP\Database\Tables;

// Exit if accessed directly
defined( 'ABSPATH' ) || exit;

use RCP\Database\Table;

/**
 * Set up the "rcp_groups" database table.
 *
 * @package RCP\Database\Tables
 * @since   2.0
 */
final class Groups extends Table {

	/**
	 * @var string Table name
	 */
	protected $name = 'groups';

	/**
	 * @var int Current database version
	 */
	protected $version = 202003302;

	/**
	 * @var array Array of upgrade versions and methods
	 */
	protected $upgrades = array(
		'201905231' => 201905231,
		'202003251' => 202003251,
		'202003252' => 202003252,
		'202003301' => 202003301,
		'202003302' => 202003302
	);

	/**
	 * Groups constructor.
	 *
	 * @access public
	 * @since  2.0
	 * @return void
	 */
	public function __construct() {
		parent::__construct();
	}

	/**
	 * Setup the database schema
	 *
	 * @access protected
	 * @since  2.0
	 * @return void
	 */
	protected function set_schema() {
		$this->schema = "group_id bigint(20) unsigned NOT NULL AUTO_INCREMENT,
			owner_id bigint(20) unsigned NOT NULL DEFAULT '0',
			membership_id bigint(20) UNSIGNED NOT NULL DEFAULT '0',
			name mediumtext NOT NULL DEFAULT '',
			description longtext NOT NULL DEFAULT '',
			code varchar(255) NOT NULL DEFAULT '',
			seats bigint(20) unsigned NOT NULL DEFAULT '0',
			member_count bigint(20) unsigned NOT NULL DEFAULT '0',
			date_created datetime NOT NULL,
			PRIMARY KEY (group_id),
			KEY owner_id (owner_id),
			KEY code (code),
			KEY membership_id (membership_id)";
	}

	/**
	 * If the old `wp_rcp_groups_db_version` option exists, copy that value to our new version key.
	 * This will ensure new upgrades are processed on old installs.
	 *
	 * @since 2.0.3
	 */
	public function maybe_upgrade() {

		$groups_db_version = get_option( $this->get_table_name() . '_db_version' );

		if ( ! empty( $groups_db_version ) ) {
			update_option( $this->db_version_key, $groups_db_version );

			delete_option( $this->get_table_name() . '_db_version' );
		}

		return parent::maybe_upgrade();

	}

	/**
	 * Upgrade to version 201905231
	 *      - Add `unsigned` to `group_id`, `owner_id`, `seats`, and `member_count`
	 *
	 * @since 2.0
	 */
	protected function __201905231() {

		$result = $this->get_db()->query( "ALTER TABLE {$this->table_name} MODIFY group_id bigint(20) unsigned NOT NULL AUTO_INCREMENT" );
		$result = $this->get_db()->query( "ALTER TABLE {$this->table_name} MODIFY owner_id bigint(20) unsigned NOT NULL" );
		$result = $this->get_db()->query( "ALTER TABLE {$this->table_name} MODIFY seats bigint(20) unsigned NOT NULL" );
		$result = $this->get_db()->query( "ALTER TABLE {$this->table_name} MODIFY member_count bigint(20) unsigned NOT NULL" );

		$success = $this->is_success( $result );

		if ( $success ) {
			rcp_log( sprintf( 'Group Accounts %s table upgrade to 201905231 successful.', $this->get_table_name() ), true );
		} else {
			rcp_log( sprintf( 'Group Accounts %s table upgrade to 201905231 failure.', $this->get_table_name() ), true );
		}

		return $success;

	}

	/**
	 * Upgrade to version 202003251
	 *      - Add `membership_id` column if it doesn't exist. This upgrade was actually done in 1.4.1 but it's
	 *        been moved here.
	 *
	 * @link  https://github.com/restrictcontentpro/rcp-group-accounts/issues/335
	 *
	 * @since 2.0.3
	 * @return bool
	 */
	protected function __202003251() {

		$result = $this->column_exists( 'membership_id' );

		if ( ! $result ) {
			$this->get_db()->query( "ALTER TABLE {$this->table_name} ADD membership_id bigint(20) UNSIGNED NOT NULL AFTER owner_id" );
		}

		$success = $this->is_success( $result );

		if ( $success ) {
			rcp_log( sprintf( 'Group Accounts %s table upgrade to 202003251 successful.', $this->get_table_name() ), true );
		} else {
			rcp_log( sprintf( 'Group Accounts %s table upgrade to 202003251 failure.', $this->get_table_name() ), true );
		}

		return $success;

	}

	/**
	 * Upgrade to version 202003252
	 *      - Drop `owner_id` key and re-add as non-unique. This upgrade was actually done in 1.4.1 but it's been
	 *        moved here.
	 *
	 * @link  https://github.com/restrictcontentpro/rcp-group-accounts/issues/335
	 *
	 * @since 2.0.3
	 * @return bool
	 */
	protected function __202003252() {

		$result = $this->get_db()->query( "ALTER TABLE {$this->table_name} DROP INDEX owner_id" );

		if ( $result ) {
			$result = $this->get_db()->query( "ALTER TABLE {$this->table_name} ADD KEY owner_id (owner_id)" );
		}

		$success = $this->is_success( $result );

		if ( $success ) {
			rcp_log( sprintf( 'Group Accounts %s table upgrade to 202003252 successful.', $this->get_table_name() ), true );
		} else {
			rcp_log( sprintf( 'Group Accounts %s table upgrade to 202003252 failure.', $this->get_table_name() ), true );
		}

		return $success;

	}

	/**
	 * Upgrade to version 202003301
	 *        - Add `code` column.
	 *
	 * @since 2.1
	 */
	protected function __202003301() {

		$result = $this->column_exists( 'code' );

		if ( ! $result ) {
			$result = $this->get_db()->query( "ALTER TABLE {$this->table_name} ADD COLUMN code varchar(255) NOT NULL DEFAULT '' AFTER description" );
		}

		if ( $result ) {
			$this->get_db()->query( "ALTER TABLE {$this->table_name} ADD INDEX code (code)" );

			rcp_log( sprintf( 'Group Accounts %s table upgrade to 202003301 successful.', $this->get_table_name() ), true );
		} else {
			rcp_log( sprintf( 'Group Accounts %s table upgrade to 202003301 successful.', $this->get_table_name() ), true );
		}

		return $this->is_success( $result );

	}

	/**
	 * Upgrade to version 202003302
	 *        - Add `membership_id` key
	 *
	 * @since 2.1
	 */
	protected function __202003302() {

		if ( ! $this->get_db()->query( "SHOW INDEX FROM {$this->table_name} WHERE Key_name = 'membership_id'" ) ) {
			rcp_log( sprintf( 'Group Accounts %s table upgrade - adding membership_id index.', $this->get_table_name() ), true );
			$result = $this->get_db()->query( "ALTER TABLE {$this->table_name} ADD INDEX membership_id (membership_id)" );
		} else {
			$result = true;
		}

		$success = $this->is_success( $result );

		if ( $success ) {
			rcp_log( sprintf( 'Group Accounts %s table upgrade to 202003302 successful.', $this->get_table_name() ), true );
		} else {
			rcp_log( sprintf( 'Group Accounts %s table upgrade to 202003302 failure.', $this->get_table_name() ), true );
		}

		return $success;

	}

}